// Copyright 2023 Autodesk, Inc.  All rights reserved.
//
// Use of this software is subject to the terms of the Autodesk license
// agreement provided at the time of installation or download, or which
// otherwise accompanies this software in either electronic or hard copy form.

/**
 * \file
 * Autodesk Analytics Program
 */

#pragma once
#include "ai_api.h"
#include "ai_map.h"
#include "ai_string.h"

/**
 * We can add product metadata to provide further information to e.g. track which plugin
 * and host application is authoring the render session (see \ref AiADPAddProductMetadata)
 * 
 * The following metadata names are automatically handled as part of the data analytics.
 */
#define AI_ADP_HOSTNAME       AtString("ADDIN_PARENT_NAME")
#define AI_ADP_HOSTVERSION    AtString("ADDIN_PARENT_VERSION")
#define AI_ADP_PLUGINNAME     AtString("ADDIN_NAME")
#define AI_ADP_PLUGINVERSION  AtString("ADDIN_VERSION")

enum class AtADPDialogMode { ONLY_FIRST_TIME = 1, ALWAYS = 2 };
/**
 * Display a modal window that lets the user opt-in/out of various
 * analytics.
 *
 * @param mode Set to ALWAYS to force the window to display so the user can
 * change the opt-in settings. ONLY_FIRST_TIME will only show the window if
 * there is no prior record on the user's computer nor online that they have
 * previously seen this dialog window. The default of ALWAYS is normally what
 * should be chosen as Arnold is already internally calling the ONLY_FIRST_TIME
 * variant.
 */
AI_API void AiADPDisplayDialogWindow(AtADPDialogMode mode = AtADPDialogMode::ALWAYS);

/**
 * Prevent the dialog window from opening. This is useful when running Arnold
 * on a headless computer, such as on a render farm, where a window popping up
 * cannot be easily serviced. If the opt-in dialog window is not raised, this
 * means there is no way to change the opt-in setting.
 */ 
AI_API void AiADPDisableDialogWindow();

/**
 * Call this after displaying the intro to user so we don't keep displaying intro message.
 * This will also opt-in the user, so call this before AiADPIsOptedIn().
 */
AI_API AI_DEPRECATED void AiADPSetIntroShown();

/**
 * Set to true if user agreed to optin and false if opted out.
 */
AI_API void AiADPSetOptedIn(bool wants_optin);

/**
 * \return true if user agreed to opt in and false if opted out.
 */
AI_API bool AiADPIsOptedIn();

/**
 * Strings and URLs to display to user when displaying ADP related dialog boxes.
 *
 * There are two dialog boxes:
 *
 * INTRO is used when the user has never before been informed about the
 * Autodesk Analytics program. This will include an introduction_intro string
 * with instructions for how to change the opt-in/out settings using kick,
 * since all plugins come with kick. If your plugin has its own way for
 * changing the optin settings, and it probably should, then instructions for
 * how to do so should go in the custom_optin_command argument and the
 * introduction_intro string will then include that as well. For instance:
 * custom_optin_command="in the C4DtoA > Help menu".  Otherwise,
 * custom_optin_command can be NULL.
 *
 * CHANGE is used after the user has been shown the intro dialog and wants to
 * change opt-in status. Note that this mode has an additional "checkbox"
 * string and instead of "introduction_intro" should use the
 * "introduction_change" string.
 *
 * The other strings are the "title", "learnMoreURL" and associated
 * "learnMoreText" link, and the "privacyURL" and associated "privacyText" link.
 *
 * /note this is deprecated and will eventually be removed as it no longer
 * returns all the proper strings. Use AiDisplayDialogWindow() instead
 */
AI_API AI_DEPRECATED void AiADPDialogStrings(AtParamValueMap& strings, const char* custom_optin_command);

/**
* Provide additional data to the product analytics, such as information about the
* client that created the Arnold render session. This has to be called before
* data is collected, for instance, before \ref AiRenderBegin.
*
* This example code sets a plugin name:
* \code
* AiADPAddProductMetadata(AI_ADP_PLUGINNAME, AtString("HtoA"));
* \endcode
*
* @param name   Name for the additional product analytics data entry
* @param value  Value for the additional product analytics data entry
*/
AI_API void AiADPAddProductMetadata(AtString name, AtString value);
