# Copyright 2023 Autodesk, Inc.  All rights reserved.
#
# Use of this software is subject to the terms of the Autodesk license
# agreement provided at the time of installation or download, or which
# otherwise accompanies this software in either electronic or hard copy form.

from ctypes import *
from .arnold_common import ai
from .ai_types import *
from .ai_array import *
from .ai_render import *

AI_DEVICE_TYPE_CPU = 0   # Render using the CPU
AI_DEVICE_TYPE_GPU = 1   # Render using the GPU

AI_DEVICE_MEMORY_TOTAL = 0
AI_DEVICE_MEMORY_FREE = 1
AI_DEVICE_MEMORY_USED = 2

AiDeviceTypeIsSupported = ai.AiDeviceTypeIsSupported
AiDeviceTypeIsSupported.argtypes = [c_int, POINTER(AtString)]
AiDeviceTypeIsSupported.restype = c_bool

AiDeviceSelect = ai.AiDeviceSelect
AiDeviceSelect.argtypes = [POINTER(AtRenderSession), c_int, POINTER(AtArray)]
AiDeviceSelect.restype = c_int

AiDeviceAutoSelect = ai.AiDeviceAutoSelect
AiDeviceAutoSelect.argtypes = [POINTER(AtRenderSession)]
AiDeviceAutoSelect.restype = c_int

AiDeviceGetSelectedType = ai.AiDeviceGetSelectedType
AiDeviceGetSelectedType.argtypes = [POINTER(AtRenderSession)]
AiDeviceGetSelectedType.restype = c_int

AiDeviceGetSelectedIds = ai.AiDeviceGetSelectedIds
AiDeviceGetSelectedIds.argtypes = [POINTER(AtRenderSession), c_int]
AiDeviceGetSelectedIds.restype = POINTER(AtArray)

AiDeviceGetCount = ai.AiDeviceGetCount
AiDeviceGetCount.argtypes = [c_int]
AiDeviceGetCount.restype = c_int

AiDeviceGetIds = ai.AiDeviceGetIds
AiDeviceGetIds.argtypes = [c_int]
AiDeviceGetIds.restype = POINTER(AtArray)

AiDeviceGetNameFunc = ai.AiDeviceGetName
AiDeviceGetNameFunc.argtypes = [c_int, c_uint]
AiDeviceGetNameFunc.restype = AtStringReturn

def AiDeviceGetName(device_type, device_id):
    return AtStringToStr(AiDeviceGetNameFunc(device_type, device_id))

AiDeviceGetMemoryMB = ai.AiDeviceGetMemoryMB
AiDeviceGetMemoryMB.argtypes = [c_int, c_uint, c_int]
AiDeviceGetMemoryMB.restype = c_uint

AtGPUCachePopulateMode = c_int
AI_GPU_CACHE_POPULATE_BLOCKING     = AtGPUCachePopulateMode(0)
AI_GPU_CACHE_POPULATE_NON_BLOCKING = AtGPUCachePopulateMode(1)

AtGPUCachePopulateCallback = CFUNCTYPE(None, c_void_p, AtRenderErrorCode, c_float, AtPythonString)
AiGPUCachePopulateFunc = ai.AiGPUCachePopulate
AiGPUCachePopulateFunc.argtypes = [AtGPUCachePopulateMode, c_int, AtGPUCachePopulateCallback, c_void_p]
AiGPUCachePopulateFunc.restype = AtRenderErrorCode

def default_ai_cache_populate_callback(cUserdata, status, fraction_done, msg):
   if fraction_done==0.0:
      print('[AiGPUCachePopulate] Running ...')
      return
   if status != AtRenderErrorCode:
      print('[AiGPUCachePopulate] %s (%.1f%% done)' % (msg, 100.0*fraction_done))
   else:
      print('[AiGPUCachePopulate] Error')
   if fraction_done==1.0:
      print('[AiGPUCachePopulate] Finished')
   if cUserdata:
      pyUserdata = ctypes.cast(cUserdata, POINTER(ctypes.py_object)).contents.value
      if pyUserdata:
         print('[AiGPUCachePopulate]: %s' % str(pyUserdata))

def AiGPUCachePopulate(mode=AI_GPU_CACHE_POPULATE_NON_BLOCKING, num_proc=0, callback_func=None, user_data=None):
   global cAiGPUCachePopulateCallback
   if callback_func:
      cAiGPUCachePopulateCallback = AtGPUCachePopulateCallback(callback_func)
   else:
      cAiGPUCachePopulateCallback = AtGPUCachePopulateCallback(default_ai_cache_populate_callback)
   if not user_data:
      user_data = {}
   cUserdata = ctypes.cast(pointer(ctypes.py_object(user_data)), c_void_p)
   return AiGPUCachePopulateFunc(mode, num_proc, cAiGPUCachePopulateCallback, cUserdata)

cAiGPUCachePopulateCallback = None

AiGPUCachePopulateStatusFunc = ai.AiGPUCachePopulateStatus
AiGPUCachePopulateStatusFunc.argtypes = [POINTER(c_float)]
AiGPUCachePopulateStatusFunc.restype = AtRenderStatus

def AiGPUCachePopulateStatus():
   f = ctypes.c_float(0.0)
   e = AiGPUCachePopulateStatusFunc(ctypes.byref(f))
   return (e, f.value)

AiGPUCachePopulateRemainingSeconds = ai.AiGPUCachePopulateRemainingSeconds
AiGPUCachePopulateRemainingSeconds.restype = c_float

AiGPUCachePopulateTerminate = ai.AiGPUCachePopulateTerminate

AiGPUCacheSetDirectory = ai.AiGPUCacheSetDirectory
AiGPUCacheSetDirectory.argtypes = [AtPythonString]

AiGPUCacheGetDirectory = ai.AiGPUCacheGetDirectory
AiGPUCacheGetDirectory.restype = AtStringReturn


