#!/bin/bash
#set -x
# apply_ssp.sh
# 
# Avaya Inc. Proprietary (Restricted)
# Solely for authorized persons having a need to know
# pursuant to Company instructions.
#
# Copyright (c) 2002-2011 Avaya Inc. All Rights Reserved.
# THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Avaya Inc.
# The copyright notice above does not evidence any actual
# or intended publication of such source code.
#

# ===========================================================================
# Purpose:   Script for Linux security updates installation
#
# Required:  A self extracting script containing AVP_Utilities_ID.txt and security updates 
#            in rpm format. The earlier version rpms cannot be rolled back once 
#            the security update rpms are installed.
#
# BASEDIR:   /opt/util/resources/patch-update/
#
# Files: 
#  $BASEDIR/index.dat - index of all patches and corresponding rpms
#  $BASEDIR/$PATCHNUM - a dir is made for each patch/update number and current
#                        rpms are stored.
# Eg: 
#     update -u patch_111.bin
#         patch_111.bin contains 
#             AVP_Utilities_ID.txt that has the patch number "111" with no space. 
#             1.rpm, 2.rpm, 3.rpm
#
# Note: 
#    1) To find out the patch numbers, use 'swversion' command
#    2) Installing this script will:
#       - stop and start aesvcs/wdinit services 
#       - backup/restore the database
#       - server reboot at the end
#    3) No roll back and uninstall support for this Linux patching process
# ===========================================================================

prompt()
{
    echo ""
    echo "********************************** WARNING **********************************"
    echo "Linux security updates will require a server reboot at the end of installation" 
    echo "Do you want to continue? y or n" 
    echo "*****************************************************************************"
    echo -n "> "
    read answer
    echo
    if [ "$answer" = "n" ] || [ "$answer" = "N" ]; then
        echo "Exiting Linux security update script..." | tee -a $UPDATELOG
        exit 1
    fi
    if [ "$answer" != "y" -a "$answer" != "Y" ]; then
        echo "Response $answer is not recognized, " | tee -a $UPDATELOG
        echo "Exiting Linux security update script..." | tee -a $UPDATELOG
        exit 1
    fi
}

#db_backup()
#{}

pre-install()
{
    echo "Entering pre-install function..." | tee -a $UPDATELOG
    
	# copy some files which might get overwritten by the new rpms
	export TMPDIR=`mktemp -d /tmp/lsu-backup-XXXX`

	cp /etc/openldap/schema/core.schema $TMPDIR
    return
}

post-install()
{
  echo "Entering post-install function..." | tee -a $UPDATELOG

  # all rpm related configuration files and back up restore code changes will ne included in this section

 #Kernel_opts changes introduced with kernel version (kernel 3.10.0-1062.9.1.el7)
  if [ -f /opt/avaya/common_services/KERNEL_options_flags_file ]
  then
     /usr/bin/diff ./KERNEL_options_flags_file.patch /opt/avaya/common_services/KERNEL_options_flags_file >/dev/null 2>&1
     if [ $? -ne 0 ];then
        /usr/bin/cp -pf /opt/avaya/common_services/KERNEL_options_flags_file /opt/avaya/common_services/KERNEL_options_flags_file.orig
        /usr/bin/cp -pf ./KERNEL_options_flags_file.patch /opt/avaya/common_services/KERNEL_options_flags_file
      fi
  else
     /usr/bin/cp -pf ./KERNEL_options_flags_file.patch /opt/avaya/common_services/KERNEL_options_flags_file
  fi
  
 if [ -f /opt/util/bin/kernel_opts.sh ]
  then
     /usr/bin/diff ./kernel_opts.sh.patch /opt/util/bin/kernel_opts.sh >/dev/null 2>&1
     if [ $? -ne 0 ];then
       /usr/bin/cp -pf /opt/util/bin/kernel_opts.sh /opt/util/bin/kernel_opts.sh.orig
       /usr/bin/cp -pf ./kernel_opts.sh.patch /opt/util/bin/kernel_opts.sh
     fi
  else
       /usr/bin/cp -pf ./kernel_opts.sh.patch /opt/util/bin/kernel_opts.sh
  fi
  #AVPUTIL-1240 Removal of nmpa-ncat rpm. Need to remove nc command usage from encryptionRemoteKey file. Replace it with curl
  #AVPUTIL-1322 Apply the encryptionRemoteKey patch only if target is 8.1E OVA
  # Get the release version of target AVPU
  TARGET_AVPU_RELEASE=`grep -oP '(?<=FullVersion\>).*(?=\<\/FullVersion)' /opt/avaya/common_services/ovf_file`
  # *** The OVA version should be updated for each release of the AVP Utilities suite of software ***
  ENCRYPTION_OVA_VERSION=AVPU-8.1.0.0.0.09-e65-E
  if [ "$TARGET_AVPU_RELEASE" = "$ENCRYPTION_OVA_VERSION" ] && grep -q "\$BIN\/nc" /opt/avaya/common_services/encryptionRemoteKey; then
     /usr/bin/cp -pf /opt/avaya/common_services/encryptionRemoteKey /opt/avaya/common_services/encryptionRemoteKey.orig
     /usr/bin/cp -pf ./encryptionRemoteKey.patch /opt/avaya/common_services/encryptionRemoteKey
  fi
  
  #AVPUTIL-1214 - Create symlink for python3  
  ln -s /opt/rh/rh-python38/root/usr/bin/python3.8 /usr/bin/python3

  chown csadmin:susers /opt/avaya/common_services/KERNEL_options_flags_file /opt/util/bin/kernel_opts.sh 
  chmod 644 /opt/avaya/common_services/KERNEL_options_flags_file
  chmod 755 /opt/util/bin/kernel_opts.sh

if grep -q 'minlen=6' /etc/pam.d/cs-auth;then
  sed -i 's/minlen=6//g' /etc/pam.d/cs-auth
fi

if grep -q 'delay=1000000' /etc/pam.d/sshd ;then 
  sed -i 's/delay=1000000//g' /etc/pam.d/sshd
fi

#AVPUTIL-1189
if grep -q 'AllowTcpForwarding' /etc/ssh/sshd_config ;then
	sed -i '/AllowTcpForwarding/d' /etc/ssh/sshd_config
fi
echo "AllowTcpForwarding no" >> /etc/ssh/sshd_config

#STIG FIX - V-81015 and V-81015
MODE_FLAG_FILE=/opt/avaya/common_services/mode_flag_file
MODE=$(cat $MODE_FLAG_FILE)
if [[ "$MODE" == "Hardened Mode for DoD" ]];then
	PROFILE="/opt/avaya/common_os/config/dod.profile"
	if grep 'active.*no' /etc/audisp/plugins.d/au-remote.conf >/dev/null ; then
		sed -i '/active/c\active = yes' /etc/audisp/plugins.d/au-remote.conf
	fi
	service auditd restart
fi


#remove iwl rpms
rpm -qa | grep iwl >/dev/null
if [ $? -eq 0 ];then
 rpm -qa | grep 'iwl.*firmware' | xargs rpm -e
fi

#remove nmap-ncat rpm
rpm -qa | grep nmap-ncat >/dev/null
if [ $? -eq 0 ];then
 rpm -qa | grep 'nmap-ncat' | xargs rpm -e
fi 

# AVPUTIL-1322 : Remove the clevis rpms that are already installed via SSP-005-02 on non-encryption OVA systems
if [ "$TARGET_AVPU_RELEASE" != "$ENCRYPTION_OVA_VERSION" ]; then
	rpm -qa | grep clevis >/dev/null
	if [ $? -eq 0 ];then
		rpm -qa | grep 'clevis' | xargs rpm -e
	fi
fi

# AVPUTIL-1322 : Also set rd.neednet=0 in /etc/default/grub
grep -q rd.neednet /etc/default/grub
if [ $? -ne 0 ]; then
	echo "Updating grub configuration"
	sed -i 's/biosdevname=0/biosdevname=0 rhgb quiet rd.neednet=0/g' /etc/default/grub
	command grub2-mkconfig -o /boot/grub2/grub.cfg
fi 

#remove aic94xx-firmware & Red_Hat_Enterprise_Linux-Release_Notes rpm - as per Legal team
rpm -qa | grep 'aic94xx-firmware\|Red_Hat_Enterprise_Linux-Release_Notes' >/dev/null
if [ $? -eq 0 ];then
 rpm -qa | grep 'aic94xx-firmware\|Red_Hat_Enterprise_Linux-Release_Notes' | xargs rpm -e
fi

}


#
#------------------- main -----------------------
#
UPDATELOG=${1}
PATCHDIR=/opt/util/resources/patch-update
SECURITY_INDEX_FILE=$PATCHDIR/security.dat
AVPU_VERSION_FILE="version.txt"
PATCH_NUMBER_FILE="AVP_Utilities_ID.txt"
RHSA_FILE="AVP_Utilities_Details.txt"
RPMLIST_FILE="rpmlist.txt"
DESC_FILE=`ls *.desc`

# AVP_Utilities_ID.txt contains the patch version
if [ ! -f $PATCH_NUMBER_FILE ]; then
    echo "ERROR: $PATCH_NUMBER_FILE not found. Please ensure the archive has AVP_Utilities_ID.txt file" | tee -a $UPDATELOG
    exit 1
fi

#RHSA_FILE contains list of fixed RHSAs.
if [ ! -f $RHSA_FILE ]; then
   echo "ERROR: $RHSA_FILE not found. Please ensure the archive has AVP_Utilities_Details.txt file" | tee -a $UPDATELOG
   exit 1
fi

#DESC_FILE contains description about SSP being applied.
if [ ! -f $DESC_FILE ]; then
   echo "ERROR: $DESC_FILE not found. Please ensure the archive has Description file" | tee -a $UPDATELOG
   exit 1
fi

cp -pf $RHSA_FILE /var/disk/logs/fixed_RHSA.txt
chmod 444 /var/disk/logs/fixed_RHSA.txt

# return patch number or release string based on option
if [[ "$1" = "-d" ]]; then
    ls *.rpm
    exit 101
elif [[ "$1" = "-n" ]]; then
    cat $PATCH_NUMBER_FILE | awk '{print $1}'
    exit 102
elif [[ "$1" = "-r" ]]; then
    cat $AVPU_VERSION_FILE | awk '{print $1}'
    exit 103
elif [[ "$1" = "-v" ]]; then
    [[ -f $PATCH_FILE ]] && exit 104 || exit 1
fi

patchnum=`cat $PATCH_NUMBER_FILE|awk '{print $1}'`
avpuversion=`cat $AVPU_VERSION_FILE|awk '{print $1}'`

# verify patch/update and release numbers
if [[ -z $patchnum ]] || [[ -z $avpuversion ]]; then
    echo "ERROR: Missing patch version or AVPU version." | tee -a $UPDATELOG
    exit 1
fi

echo "Patch Name: $patchnum" | tee -a $UPDATELOG
# verify the release string matched
/opt/util/bin/swversion | grep "^Application Version:" | grep `cat $AVPU_VERSION_FILE` >/dev/null 2>&1
if [[ $? -ne 0 ]]; then
    echo "ERROR: Version mis-matched! This patch cannot be applied to the current release."  | tee -a $UPDATELOG
    exit 1
fi

if [[ ! -d $PATCHDIR ]]; then
    mkdir $PATCHDIR
fi

# verify whether the patch already installed
if [[ -f $SECURITY_INDEX_FILE ]]; then
    grep -qx $patchnum $SECURITY_INDEX_FILE
    x=$?
    if [[ $x -eq 0 ]]; then   # found
        echo "Update $P already installed; Please use "swversion" command to list patches"  | tee -a $UPDATELOG
        exit 1
    fi
fi

# prompt installer to continue
prompt

# run pre-install 
#pre-install

# database backup
#db_backup

# AVPUTIL-1322 : Don't install clevis-dracut and clevis-systemd rpms for non-encryption OVA
# Get the release version of target AVPU
TARGET_AVPU_RELEASE=`grep -oP '(?<=FullVersion\>).*(?=\<\/FullVersion)' /opt/avaya/common_services/ovf_file`
# *** The encryption OVA version should be updated for each release of the AVP Utilities suite of software ***
ENCRYPTION_OVA_VERSION=AVPU-8.1.0.0.0.09-e65-E
if [ "$TARGET_AVPU_RELEASE" != "$ENCRYPTION_OVA_VERSION" ]; then
	ls -l | grep 'clevis' >/dev/null
	if [ $? -eq 0 ]; then
		rm -f clevis*
	else
		echo "No clevis rpms found in this patch" | tee -a $UPDATELOG
	fi
fi

# the optional rpmlist can specify the rpm installing order
# create rpmlist if not exists
if [[ ! -f $RPMLIST_FILE ]]; then
    ls *.rpm > $RPMLIST_FILE
fi

rpms=`cat $RPMLIST_FILE`
echo "" | tee -a $UPDATELOG
echo "***** Installing Linux rpms *****" | tee -a $UPDATELOG
echo "$rpms" >> $UPDATELOG



#echo " * `rpm -qp *.rpm 2>/dev/null`"
rpm -Uvh --force --nodeps *.rpm | tee -a $UPDATELOG

# run post-install
post-install

# Updating patch index file - override file to have latest ssp patch information only
echo "====" > $SECURITY_INDEX_FILE
echo $patchnum >> $SECURITY_INDEX_FILE
cat $DESC_FILE >> $SECURITY_INDEX_FILE
cat $RPMLIST_FILE >> $SECURITY_INDEX_FILE
echo "====" >> $SECURITY_INDEX_FILE

exit 0
