#!/usr/bin/env python
#
# This process run as daemon and monitor log
#  
#   1) It has class AuthMonitor which currently monitors authentication
#    and continously read /var/log/secure using FileTail class.
#   2) It generates alarm in syslog if number of failed attempts exceed
#      failed_login_count in interval failed_login_duration. Both parameters
#      defined in /etc/opt/avpualarm.conf
#   3) It also uses UtilDaemon class to run as daemon.
#   

import sys
import time
import socket
import syslog
import subprocess
import os
import threading
import signal
import select
import re
import json
import Queue

from six.moves      import configparser
from datetime       import datetime

# Below modules are custom modules defined in AVPU.
from utilDaemon     import UtilDaemon
from filetail       import FileTail

# This file is a flat file database to store data related to authentication
data_file = "/etc/opt/.login_alarm_data_db"

# This file is a flat file database to store data related to command access
acc_file = "/etc/opt/.acc_alarm_data_db"

# This file is being monitored by FileTail
monitor_file = "/var/log/secure"

# Configuration file for failed_login
conf_file = "/etc/opt/avpualarm.conf"

# message types in queue 
login_msg = 1
access_msg = 2

# Exit flag for terminating from thread
exitFlag = False

# syslog configuration
syslog.openlog('loginmon', syslog.LOG_CONS, syslog.LOG_LOCAL5)


# Class for message queue login msg
class authMessage:
    def __init__(self, msg_type, log_timestamp, auth_user):
        self.msg_type = msg_type
        self.log_timestamp = log_timestamp
        self.auth_user = auth_user


# Class for message queue access failure msg 
class accMessage(authMessage):
    def __init__(self, msg_type, log_timestamp, auth_user, command):
        authMessage.__init__(self, msg_type, log_timestamp, auth_user)
        self.command = command

##############################################################################
# Thread to process messages in message queue and generate alarm
# AuthMonitor run method will queue messages and this thread will process 
##############################################################################
class logProcessThread(threading.Thread):
        ##############################################################################
        # Initialize Class
        ##############################################################################
        def __init__(self, threadID, name, q, config_count=6, config_interval=600, config_duration=900):
            threading.Thread.__init__(self)
            self.threadID = threadID
            self.name = name
            self.q = q
            self.login_count = config_count
            self.login_interval = config_interval
            self.access_duration = config_duration


        ##############################################################################
        # run method.
        ##############################################################################
        def run(self):
            self.process_data()

        ##############################################################################
        # Process data from queue.
        ##############################################################################
        def process_data(self):
            global exitFlag
            q = self.q
            while not exitFlag:
               if not q.empty():
                  data = q.get()
                  if data.msg_type == login_msg:
                     self.monitor_failure_log(data.log_timestamp, data.auth_user)
                  elif data.msg_type == access_msg:
                     self.monitor_failure_cmd(data.log_timestamp, data.auth_user, data.command)
               else:
                  self.cleanup_db()
                  self.cleanup_access_db()
                  time.sleep(2)   



        ##############################################################################
        # Delete all records from access failure db which are older than duration from now().
        ##############################################################################
        def cleanup_access_db(self):
            config_duration = self.access_duration
            if(os.path.exists(acc_file)):
               with open(acc_file, "r") as inFile:
                  if os.stat(acc_file).st_size > 0:
                     inFile.seek(0)
                     alarm_data = json.load(inFile)
                     if not alarm_data:
                         # nothing to delete
                         return

                     for key, value in alarm_data.items():
                         for inner_key, inner_value in value.items():
                             max_time = alarm_data[key][inner_key]
                             maxTimeFormat = datetime.strptime(max_time, '%b %d %Y %H:%M:%S')
                             currentTime = datetime.now()
                             diffSeconds = int((currentTime - maxTimeFormat).total_seconds())
                             if diffSeconds > config_duration:
                                del alarm_data[key][inner_key]

                     for key, value in alarm_data.items():
                         if not alarm_data[key]:
                            del alarm_data[key]

                     with open(acc_file, 'w+') as outfile:
                        json.dump(alarm_data, outfile)

            return


        #############################################################################################
        # Delete all records from login failure db which are older than config_interval from now().
        #############################################################################################
        def cleanup_db(self):
            config_interval = self.login_interval
            if(os.path.exists(data_file)):
               with open(data_file, "r") as inFile:
                  if os.stat(data_file).st_size > 0:
                     inFile.seek(0)
                     monitor_data = json.load(inFile)
                     if not monitor_data:
                         # nothing to delete
                         return

                     for key, value in monitor_data.items():
                         if "max_time" in value:
                             max_time = monitor_data[key]["max_time"]
                             maxTimeFormat = datetime.strptime(max_time, '%b %d %Y %H:%M:%S')
                             currentTime = datetime.now()
                             diffSeconds = int((currentTime - maxTimeFormat).total_seconds())
                             if diffSeconds > config_interval:
                                del monitor_data[key]

                     with open(data_file, 'w+') as outfile:
                        json.dump(monitor_data, outfile)

            return 


        ##############################################################################
        # process failure command log and check if alarm is generated.
        ##############################################################################
        def monitor_failure_cmd(self, timestamp, user, cmd):
            config_duration = self.access_duration
            #  safety check
            if user == "" or config_duration < 30 or config_duration > 900:
               return

            try:
                currentYear = datetime.now().year
                logTime = timestamp
                logTimeYear = logTime[:7] + str(currentYear) + " " + logTime[7:]
                logTimeFormat = datetime.strptime(logTimeYear, '%b %d %Y %H:%M:%S')
                if(os.path.exists(acc_file)):
                   with open(acc_file, "r") as inFile:
                      if os.stat(acc_file).st_size > 0:
                         inFile.seek(0)
                         alarm_data = json.load(inFile)
                         if user in alarm_data:
                            if cmd in alarm_data[user]:
                               alarmTime = datetime.strptime(alarm_data[user][cmd], '%b %d %Y %H:%M:%S')
                               alarmSeconds = int((logTimeFormat - alarmTime).total_seconds())
                               if alarmSeconds > config_duration:
                                  alarm_data[user][cmd]=logTimeYear
                                  syslog.syslog(syslog.LOG_ALERT, "ACCESS_FAULT " + '" User {} not authorized to run command - {} "'.format(user,cmd))
                               else:
                                  # Alarm is already generated wait for config interval to lapse
                                  return

                            else:
                               alarm_data[user][cmd]=logTimeYear
                               syslog.syslog(syslog.LOG_ALERT, "ACCESS_FAULT " + '" User {} not authorized to run command - {} "'.format(user,cmd))
                         else:
                            alarm_data[user]={}
                            alarm_data[user][cmd]=logTimeYear
                            syslog.syslog(syslog.LOG_ALERT, "ACCESS_FAULT " + '" User {} not authorized to run command - {} "'.format(user,cmd))
                      else:
                         alarm_data = {}
                         alarm_data[user]={}
                         alarm_data[user][cmd]=logTimeYear
                         syslog.syslog(syslog.LOG_ALERT, "ACCESS_FAULT " + '" User {} not authorized to run command - {} "'.format(user,cmd))
                else:
                   alarm_data = {}
                   alarm_data[user]={}
                   alarm_data[user][cmd]=logTimeYear
                   syslog.syslog(syslog.LOG_ALERT, "ACCESS_FAULT " + '" User {} not authorized to run command - {} "'.format(user,cmd))

                with open(acc_file, 'w+') as outfile:
                   json.dump(alarm_data, outfile)

            except OSError, e:
                sys.stderr.write("Error in failed log: %d (%s)\n" % (e.errno, e.strerror))
            return


        ##############################################################################
        # Process Failed authentication log line and generate alarm if needed.
        ##############################################################################
        def monitor_failure_log(self, timestamp, user):
            parser = configparser.ConfigParser()
            config_count = self.login_count
            config_interval = self.login_interval
            #  safety check
            if user == "" or config_count <= 1 or config_count > 300 or config_interval < 30 or config_interval > 900:
               return

            try:
                currentYear = datetime.now().year
                logTime = timestamp
                logTimeYear = logTime[:7] + str(currentYear) + " " + logTime[7:]
                logTimeFormat = datetime.strptime(logTimeYear, '%b %d %Y %H:%M:%S')
		if(os.path.exists(data_file)):
		   with open(data_file, "r") as inFile:
		      if os.stat(data_file).st_size > 0:
		         inFile.seek(0)
			 monitor_data = json.load(inFile)
			 
			 if user in monitor_data:
			    loop_count = 0
			    login_count = monitor_data[user]["count"]
			    count = 1
                            maxDiffSec = 0
                            noAlarmStr = ""
                            monitor_data[user]["max_time"] = logTimeYear

                            if "alarm" in monitor_data[user]:
                               alarmTime = datetime.strptime(monitor_data[user]["alarm"], '%b %d %Y %H:%M:%S') 
                               alarmSeconds = int((logTimeFormat - alarmTime).total_seconds())
                               if alarmSeconds > config_interval:
                                  del monitor_data[user]["alarm"]
                               else:
                                  # Alarm is already generated wait for config_interval to lapse
                                  return

			    while (loop_count < login_count):
			       loop_count = loop_count + 1
			       if str(loop_count) in monitor_data[user]["timestamp"]:
			          oldLogTime = datetime.strptime(monitor_data[user]["timestamp"][str(loop_count)], '%b %d %Y %H:%M:%S')
                                  diffSeconds = int((logTimeFormat - oldLogTime).total_seconds())
				  if diffSeconds <= config_interval:
				     count = count + 1
                                     if maxDiffSec < diffSeconds:
                                        maxDiffSec = diffSeconds
                                        noAlarmStr = monitor_data[user]["timestamp"][str(loop_count)]
				  else:
				     del monitor_data[user]["timestamp"][str(loop_count)]

			    login_count = login_count + 1
			    monitor_data[user]["timestamp"][str(login_count)] = logTimeYear
			    monitor_data[user]["count"] = login_count
			    if count >= config_count:
			       syslog.syslog(syslog.LOG_ALERT, "AUTH_FAULT " + '" Maximum login attempts by user - {} "'.format(user))
                               if noAlarmStr != "":
                                  monitor_data[user]["alarm"] = noAlarmStr
			 else:
			    count = 1
			    monitor_data[user] = {}
			    monitor_data[user]["timestamp"] = {}
			    monitor_data[user]["timestamp"][count] = logTimeYear
                            monitor_data[user]["max_time"] = logTimeYear
			    monitor_data[user]["count"] = 1
			
		      else:
		         count = 1
			 monitor_data = {}
			 monitor_data[user] = {}
			 monitor_data[user]["timestamp"] = {}
			 monitor_data[user]["timestamp"][count] = logTimeYear
                         monitor_data[user]["max_time"] = logTimeYear
			 monitor_data[user]["count"] = 1
			 
	        else:
	           count = 1
		   monitor_data = {}
		   monitor_data[user] = {}
		   monitor_data[user]["timestamp"] = {}
		   monitor_data[user]["timestamp"][count] = logTimeYear
                   monitor_data[user]["max_time"] = logTimeYear
		   monitor_data[user]["count"] = 1
		

	        with open(data_file, 'w+') as outfile:
	           json.dump(monitor_data, outfile)
          
            except OSError, e:
                sys.stderr.write("Error in failed log: %d (%s)\n" % (e.errno, e.strerror))
            return

   
class AuthMonitor(UtilDaemon):
        ##############################################################################
        # Extract User from Invalid User.
        ##############################################################################
        def invalid_user(self, user):
            
            if len(user) > 0:
                  userName = user.strip()
                  userName = user.rstrip("\n")
            return userName


        ##############################################################################
        # Extract User from Failed Authentication.
        ##############################################################################
        def failed_auth_user(self, line):
            res = line.split("=")
            user = ""
            if len(res) > 7:
               user_con = res[7].split(" ")
               user = user_con[0]
               user = user.strip()
               user = user.rstrip("\n")
            return user


        ##############################################################################
        # Extract User from Failed access log.
        ##############################################################################
        def extract_command(self, line):
            res = line.split("=")
            cmd = ""
            if len(res) > 4:
               if " " in res[4]:
                  cmd_con = res[4].split(" ")
                  cmd = cmd_con[0]
               else:
                  cmd = res[4]

               cmd = cmd.strip()
               cmd = cmd.rstrip("\n")
            return cmd

        ##############################################################################
        # Override run method
        ##############################################################################
        def run(self):
            if(os.path.exists(monitor_file)):
                parser = configparser.ConfigParser()
                msgQueue = Queue.Queue()
                config_count = 6
                config_interval = 600
                config_duration = 900
                global exitFlag
                ssh_timeout = False
                try:
                   if(os.path.exists(conf_file)):
                      parser.read(conf_file)
                      config_count = parser.getint('failed_login','failed_attempt_count')
                      config_interval = parser.getint('failed_login','failed_attempt_duration')
                      config_duration = parser.getint('failed_access','failed_access_duration')
                except OSError, e:
                   sys.stderr.write("Error in parsing conf file: %d (%s)\n" % (e.errno, e.strerror))

                logThread = logProcessThread( 1, "logProcessThread", msgQueue, config_count, config_interval, config_duration)
                logThread.start()

                # Use FileTail to continously monitor /var/log/secure 
                logfile = FileTail(monitor_file)
                
                for line in logfile:
                   splitLog = line.split()  
                   if splitLog[4][0:4] == "sshd":
                      if re.search("Failed keyboard-interactive",line) and re.search("invalid user",line):
                         user = self.invalid_user(splitLog[10])
                         if user != "":
                            timestamp = line[0:15]
                            m = authMessage( login_msg, timestamp, user)
                            msgQueue.put(m)
                      elif re.search("authentication failure",line):
                         user = self.failed_auth_user(line)
                         if user != "":
                            timestamp = line[0:15]
                            m = authMessage( login_msg, timestamp, user)
                            msgQueue.put(m)
                      elif re.search("Timeout, client not responding",line):
                           ssh_timeout = True
                      elif re.search("session closed for user",line):
                           splitLine = line.split(':') 
                           if splitLine[5] != "":
                              logSsh = splitLine[5].strip()
                              logSsh = splitLine[5].rstrip("\n")
                              if (ssh_timeout == True):
                                 syslog.syslog(syslog.LOG_ALERT, "SSH_TMOUT " + '"SSH{}"'.format(logSsh))
                                 ssh_timeout = False
                              else:
                                 syslog.syslog(syslog.LOG_ALERT, "SSH_CLSD " + '"SSH{}"'.format(logSsh))
                   elif splitLog[4][0:4] == "sudo":
                      if re.search("command not allowed", line) and re.search("COMMAND=",line):
                         user = self.invalid_user(splitLog[5])
                         command = self.extract_command(line) 
                         if user != "" and command != "":
                            timestamp = line[0:15]
                            m = accMessage(access_msg, timestamp, user, command)
                            msgQueue.put(m)
                # process has stopped
                exitFlag = true
                logThread.join()
