#!/usr/bin/python
# vim:ts=4:sw=4 expandtab
#############################################################################
#
# Copyright Avaya Inc., All Rights Reserved.
#
# THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Avaya Inc.
#
# The copyright notice above does not evidence any actual or intended
# publication of such source code.
#
# Some third-party source code components may have  been modified from their
# original versions by Avaya Inc.
#
# The modifications are Copyright Avaya Inc., All Rights Reserved.
#
# Avaya - Confidential & Restricted. May not be distributed further without
# written permission of the Avaya owner.
#
#############################################################################
"""

Internal script used to configure the CRD files used for trust 
initialization.
 
Commands:
    - password: set or clear the keystore password
    - id: set ID certificate values 

ID Options:
    -s : service ID
    --cn : common name to assign
    --san: subject alt name to assign
    --days: number of days the certificate should be valid for (subject to
        CA limits)

Return Codes:
    - 0: Success
    - 1: Failure
    - 2: Usage
"""

import argparse
import codecs
from xml.dom.minidom import parse
import getpass
import os
from subprocess import *
import sys

TM_HOME = '/opt/avaya/tm'
CRD_INPUT_FILE = TM_HOME + '/CRDJEE_Input.xml' 

# path to CRD files that needs to be setup with the password
CRD_PASSWD_FILES = [ 
    TM_HOME + '/CRDJEETM.xml', 
    TM_HOME + '/CRDUpgradeTM.xml' ]

# service IDs 
service_ids = [ 'mgmt', 'spiritalias', 'syslog' ]

###############################################################
# ID certificate operations
###############################################################
"""
Updates the ID cert values for the given service

"""
def update_id_cert(serviceID, common_name=None, 
        subject_alt_name=None, days=None):
    
    doc = parse(CRD_INPUT_FILE)
    root = doc.documentElement
    
    idCertNodes = doc.getElementsByTagName('INPUT_IDCert')
    for idCertNode in idCertNodes:
        serviceIDNode = idCertNode.getElementsByTagName('ServiceID')[0]
        if serviceIDNode.firstChild.data == serviceID:

            if common_name:
                common_name_node = idCertNode.getElementsByTagName('SS_Cert_CN')
                common_name_node[0].replaceChild(
                    doc.createTextNode(common_name), 
                    common_name_node[0].firstChild)

            if subject_alt_name:
                subject_alt_name_node = \
                    idCertNode.getElementsByTagName('SS_Cert_Subject_Alt_Name') 
                if subject_alt_name_node:
                    value_node = \
                        subject_alt_name_node[0].getElementsByTagName('Value')
                    text_node = doc.createTextNode(subject_alt_name) 
                    if value_node[0].firstChild == None:
                        value_node[0].appendChild(text_node)
                    else:
                        value_node[0].replaceChild(text_node, 
                            value_node[0].firstChild)

            if days:
                days_node = idCertNode.getElementsByTagName('SS_Valid_Days')
                days_node[0].replaceChild(doc.createTextNode(days),
                    days_node[0].firstChild)

    # save the changes
    with codecs.open(CRD_INPUT_FILE, "w","utf-8") as f:
        doc.writexml(f)

###############################################################
# Update or Clear keystore password
###############################################################
def update_keystore_password(passwd):

    for crd_file in CRD_PASSWD_FILES:
        if os.path.exists(crd_file) == False:
            continue

        doc = parse(crd_file)

        # set the password in all of the keystore/trustore elements
        keystoreNodes = doc.getElementsByTagName('KEY_STORE')
        truststoreNodes = doc.getElementsByTagName('TRUST_STORE')
        allStoreNodes = keystoreNodes + truststoreNodes
        
        for store in allStoreNodes:
            # set it to the passed in password
            passwdNode = store.getElementsByTagName('CertStorePassword')[0] 
            passwd_txt_node = doc.createTextNode(passwd) 
            if passwdNode.firstChild == None: 
                passwdNode.appendChild(passwd_txt_node) 
            else: 
                passwdNode.replaceChild(passwd_txt_node, passwdNode.firstChild)

        # save the CRD file with the passwords filled in
        with codecs.open(crd_file, "w","utf-8") as f:
            doc.writexml(f)

###############################################################
# Password operations
###############################################################
def password_cmd(args):
    if not (args.clear or args.passwd):
        print "password command requires --clear or --set option"
        sys.exit(2)
    
    if (args.clear and args.passwd):
        print "password command cannot have both --clear and --set"
        sys.exit(2)

    if args.clear:
        update_keystore_password('')
    else:
        if not os.path.exists('/etc/opt/tm_initialize.sh'):
           filename = "/etc/opt/passwd"
           with open(filename, 'w') as pemFile:
              pemFile.write(args.passwd)
        
        update_keystore_password(args.passwd)

###############################################################
# ID cert operations
###############################################################
def id_cert_cmd(args):
    
    if args.id == 'securitymodule':
        # shortcut for both SIP and HTTP
        for id in [ 'securitymodule_sip', 'securitymodule_http' ]:
            update_id_cert(id, args.common_name, args.subject_alt_name, 
                args.days)
    else:
        update_id_cert(args.id, args.common_name, args.subject_alt_name, 
            args.days)

###############################################################
# Main
###############################################################

parser = argparse.ArgumentParser()

subparsers = parser.add_subparsers(title='commands')
password_parser = subparsers.add_parser('password', help='Password operations')
password_parser.set_defaults(func=password_cmd)
password_parser.add_argument('--set', dest='passwd',  metavar='PASSWORD')
password_parser.add_argument('--clear', action="store_true", default=False,
    help="Remove any passwords from CRD files")

id_parser = subparsers.add_parser('id', help='ID certificate operations')
id_parser.set_defaults(func=id_cert_cmd)
id_parser.add_argument('-s', '--service', dest='id', action="store", 
    type=str, 
    help='service ID of certificate', 
    choices=service_ids,
    required=True)
id_parser.add_argument('--cn', dest='common_name', action='store', 
    type=str, help='Common name to assign')
id_parser.add_argument('--san', dest='subject_alt_name', action='store', 
    type=str, help='Subject Alternate Name to assign')
id_parser.add_argument('--days', dest='days', action='store',
    type=str, help='Number of days the certificate should be valid')
    
args = parser.parse_args()
args.func(args)
