#!/bin/bash
# vim:ts=4:sw=4

###############################################################################
# Initializes Trust Management and Database Replication
#
# initTM Return Codes:
#
# 	0: Success
# 	1: Could not connect to System Manager
#	2: Required process did not start
#	3: Unable to configure WebSphere
#	4: Unknown error occurred
#	5: Incorrect enrollment password
#	6: User choose not to install demo certificates
#	7: User choose not to run the force option
#	8: Invalid option
#	9: Demo certs cannot be installed when FIPS is enabled
#   10: Potential time difference that caused failure
# 
###############################################################################

# check if user is root - do a sudo if not
[ $UID -eq 0 ] || exec sudo $0 $@

# This script is not allowed to be run in debug mode
set +xv

COMMON_HOME=/opt/util
COMMON_SERVICES=/opt/avaya/common_services
source $COMMON_HOME/bin/tm/TMEnv.sh

UPGRADE_PATH=/opt/Avaya/upgrade
PREV_UPGRADE_PATH=/opt/Avaya/prev-upgrade
SPIRIT_HOME=/opt/spirit
export SPIRIT_HOME

TMCLIENT=$COMMON_HOME/bin/tm/TMClient.sh
TMPATH=/opt/avaya/tm
SPIRIT_TMPATH=$SPIRIT_HOME/security
SMGR_DET="$COMMON_SERVICES/test"
RESTORE_PATH=/etc/opt/tm_initialize.sh

JAVA_CMD=/usr/java/latest/bin/java

# initialize flags
UPGRADE=no
INITIALIZE=yes
THIRDPARTY_INIT=no
FORCE=no
CLEAN_ONLY=no
RESTART=yes
SILENT=no

logger='logger -t initTM'

# number of times to retry connection failures
ATTEMPTS=0
MAX_NUMBER_OF_ATTEMPTS=1

# Text Format Variables
BOLD=$(tty -s && tput bold)
NORMAL=$(tty -s && tput sgr0)

# FIPS mode flag
fips_mode=0

FORCE_WARNING="\n${BOLD}WARNING:${NORMAL} All certificates will be erased and replaced with default certificates.\nDo you wish to continue?\n"

CLEAN_WARNING="\n${BOLD}WARNING:${NORMAL} All certificates and configuration will be erased.\n"

FAILURE_MSG="\nUnable to initialize trust management. This task will have to be manually\nexecuted in order for trust management to work. To initialize trust management\nexecute /opt/avaya/common_services/initTM.\n"

function populate_install_prop(){
    smgrip=`grep -o '<nodeIP>.*</nodeIP>' $SMGR_DET | sed 's/\(<nodeIP>\|<\/nodeIP>\)//g'`
    smgrfqdn=`grep -o '<nodeFQDN>.*</nodeFQDN>' $SMGR_DET | sed 's/\(<nodeFQDN>\|<\/nodeFQDN>\)//g'`
    echo "SmgrIP=$smgrip" > $COMMON_SERVICES/install.properties
    echo "SmgrFQDN=$smgrfqdn" >> $COMMON_SERVICES/install.properties
    sed -i '/ejbcaIP=/c\ejbcaIP='$smgrip /opt/avaya/tm/trust.properties
}

function get_smgr_details(){
        # SMGR is not configured yet. So ask user for the same.
        echo ""
        read -p "Enter SMGR's IP address: " IP_ADDRESS
        ipcalc -cs $IP_ADDRESS
            if [[ $? -ne 0 ]]; then
                echo "Please enter valid IP address. Aborting.."
                exit 1
            fi
            $COMMON_SERVICES/Set_SMGR $IP_ADDRESS &> /dev/null
            if [[ $? -ne 0 ]]; then
                echo "Could not retrieve the SMGR's FQDN. Please check the IP adress. Aborting.."
                exit 1
            fi
            populate_install_prop
}
    
#####################################################################
# Check if SMGR details are configured or not
#####################################################################
function check_smgr_details(){
    if [ -f $SMGR_DET ]; then
        smgrcheck=`grep -o '<nodeIP>.*</nodeIP>' $SMGR_DET | sed 's/\(<nodeIP>\|<\/nodeIP>\)//g'`
        [ -z $smgrcheck ] && get_smgr_details || populate_install_prop
    else
        get_smgr_details
    fi
}

#####################################################################
# Collects the Enrollment Password
#####################################################################

function collectEnrollmentPassword() {

    # make sure this is an interactive session
        tty -s
        if [ $? != 0 ]
        then
                echo "Not an interactive session, exiting."
                exit 1
        fi

        # make sure to turn echo back on if the user aborts
        trap "stty echo; echo; exit 1" INT TERM EXIT

        local TRY_AGAIN="try again or hit Ctrl-C to abort."

        SCEPPassword=""
        while [ "$SCEPPassword" == "" ]
        do
                echo -e "\nEnter the Enrollment Password that matches the value in System "
                echo "Manager administration (Security -> Certificates --> Enrollment Password)."

                # collect the password
                stty -echo
                read -p "Enrollment Password: " passwd1; echo
                stty echo
                if [ "$passwd1" == "" ]
                then
                        echo -e "\nERROR! A value must be entered, $TRY_AGAIN"
                        continue;
                fi
                stty -echo
                read -p "Enrollment Password again: " passwd2; echo
                stty echo
                if [ "$passwd1" != "$passwd2" ]
                then
                        echo -e "\nERROR! Passwords do not match, $TRY_AGAIN"
                        continue;
                fi
                SCEPPassword=$passwd1
        done

        # reset the traps
        trap - INT TERM EXIT
}


function setup_logging() {
	
	# only set it up if necessary so that rsyslog is not restart every time
	# initTM is run.
	if [ ! -f /etc/rsyslog.d/initTM.conf ]
	then
		cat > /etc/rsyslog.d/initTM.conf <<- EOF
		\$template installFormat,"%timestamp:1:3:date-rfc3164% %timestamp:9:10:date-rfc3339% %timestamp:8:15:date-rfc3164% %msg%\n"
		\$umask 0000
		\$FileGroup susers
		\$FileCreateMode 0640
		if \$syslogtag contains 'initTM' then /var/log/Avaya/initTM.log;installFormat
		& stop
		EOF
	
		# restart syslog so the changes are picked up
		systemctl restart rsyslog.service
	fi
}	

# log to syslog and print to the screen
function printErrorMessage() {
	echo "Trust Management Initialization failed." | $logger -s 2>&1 | \
		sed 's/initTM: //'
	echo -e $FAILURE_MSG
}

function printUsage() {

/bin/cat << EOF

Usage: initTM [OPTIONS] [PASSWORD]

 -f, --force	force the initialization of trust even if trust has
  		already been established
 -c, --clean    Clear out previous configuration and certificates.
 -r, --retry	Retry for up to 10 minutes if unable to connect to the SMGR
 -s, --silent	Silent mode (no user prompts).
EOF

}

# logs to initTM.log & the console
# in: $1 - message to log
function log() 
{
	$logger -s "$@" 2>&1 | sed 's/initTM: //' 
}

# in: $1 - ip address or FQDN to ping
# out: 0 = ping was successful , >0 ping failed
function pingcheck()
{
	ping -c1 -q $1 >/dev/null 2>&1
	RC=$?
	[ $RC -ne 0 ] && echo "WARNING: Ping of $2 at $1 failed."
	return $RC
}

# sets up trust with SMGR using SCEP
function initialize() {
	
	log "Trust Management Initialization with System Manager started."

	fips_mode=`sysctl crypto.fips_enabled | awk -F' = ' '{print $2}'`
	if [ $fips_mode -eq 1 ]; then
          # Initialization does not work with BCFIPS enabled so override
          # it so trust can be established.
          export JAVA_OPTS="-Djava.security.properties=$COMMON_HOME/conf/tm.java.security"
        fi

	if [ -f $TMPATH/CRDThirdParty_Input.xml ]
	then
		if [ "$THIRDPARTY_INIT" = "yes" ]
		then
			# Eventhough an enrollment password is not needed for 
			# initialization, it needs to be set for runUpdate methods
			SCEPPassword="not_used"
		fi
		
		# The user is initializing with third party certificates, not SMGR
		runUpdate CRDThirdPartyTM.xml CRDThirdParty_Input.xml CRDJEE.xml
		
		# The third party file is no longer needed
		rm -f CRDThirdParty*.xml
		
		# A new password has been generated make sure trust is configured
		# correctly
		$COMMON_HOME/bin/tm/setup_trust_config.sh
		
	elif [ "$UPGRADE" = "yes" ]
	then
		log "Initializing with upgrade configuration"
		runUpdate CRDUpgradeTM.xml CRDUpgrade_Input.xml CRDJEE.xml
	else
		runUpdate CRDJEETM.xml CRDJEE_Input.xml CRDJEE.xml
	fi
	
	# Additional trusted certificates can be added prior to running initTM
	# so load up those certificates.
	ls $TMPATH/trustedcerts/* &> /dev/null
	if [ $? = 0 ]
	then
		echo "Adding trusted certificates"
		for CERT in $(ls $TMPATH/trustedcerts/* 2>/dev/null)
		do
			$COMMON_HOME/bin/certTool --import $CERT -t -s all -y &>/dev/null
			if [ $? != 0 ]
			then
				echo "WARNING: Failed to add $CERT"
			else
				# done loading the certificate
				rm -f $CERT
			fi
		done	
	fi

	if [ -f $SPIRIT_HOME/security/spirit-trust.bcfks ]
	then
		mv $SPIRIT_HOME/security/spirit-trust.bcfks \
			$SPIRIT_HOME/security/spirit-trust.bcfks.bak
	fi
	if [ -f $SPIRIT_HOME/security/spirit-identity.bcfks ]
	then
		mv $SPIRIT_HOME/security/spirit-identity.bcfks \
			$SPIRIT_HOME/security/spirit-identity.bcfks.bak
	fi

	# spirit has been known to have expired certs so remove them
	$TMCLIENT removeExpiringTrustedCerts

	log "Trust Management Initialization completed."
}

# While in 

# Calls update method to create certificates.  It handles any errors and
# retries if a connection cannot be made.
function runUpdate() {

	CRDTM=$1
	CRDINPUT=$2
	CRD=$3

	TEMPFILE=$(mktemp)

	while [ $ATTEMPTS -lt $MAX_NUMBER_OF_ATTEMPTS ]
	do
		ATTEMPTS=$[$ATTEMPTS+1]

		$TMCLIENT update $SCEPURL $SCEPPassword $CRDTM $CRDINPUT \
			$CRD &> $TEMPFILE
	
		egrep -q "errorCode|Exception" $TEMPFILE
		if [ "$?" = "0" ]
		then
			egrep -q "Forbidden|Bad Request" $TEMPFILE
			if [ "$?" = "0" ]
			then
				echo
				log "Enrollment password is incorrect or expired."
				rm -f $TEMPFILE 
			
				failureCleanUp
				printErrorMessage
				exit 5
			fi

			grep -q "unable to find valid certification" $TEMPFILE
			if [ "$?" = "0" ]
			then
				echo 
				log "Unable to validate certificate chain."
				log "Verify that the time is set correctly on System Manager and Session Manager"
				rm -f $TEMPFILE
				failureCleanUp
				printErrorMessage
				exit 10
			fi
	
			grep -q "Connection" $TEMPFILE
			if [ "$?" = "0" ]
			then
				if [ $ATTEMPTS -eq $MAX_NUMBER_OF_ATTEMPTS ]
				then
					log "Unable to connect to System Manager."
					printErrorMessage
					exit 1
				else
					log "Unable to connect to System Manager.  Trying again in 60 seconds."
 					sleep 60
					continue
				fi
			fi

			# must be an unknown error
			log "Unknown error encountered, see /var/log/Avaya/initTM.log for details."
			cat $TEMPFILE | $logger
			rm -f $TEMPFILE
			failureCleanUp
			exit 4
		fi
		
		# Must have been successful
		rm -f $TEMPFILE
		break
	done
}

#######################################################################
# Function that handles the force option
#######################################################################
function force() {

	# only show warnings if trust has been previously established
	if [ -f $TMPATH/TMClientInv.xml ]
	then
		if [ "$CLEAN_ONLY" = "yes" ]
		then 
			echo -e $CLEAN_WARNING
		else
			echo -e $FORCE_WARNING
		fi

		if [ "$SILENT" = "no" ]
		then
			read -n1 -p "Do you want to continue? (y/n) " RESP
		else
			RESP="y"
		fi

		if [ "$RESP" != "y" ] && [ "$RESP" != "Y" ]
		then
			echo
			$logger "User decided not to use the force option."
			exit 7
		fi
		echo

		# Before wiping out everything make sure that check to see if the
		# demo certificates were used before.  If the demo flag is already
		# set then there is no need check the current configuration.
	fi
	
	# remove TM files needed to force re-initialization
	rm -f $TMPATH/TMClientInv.xml
	rm -f $TMPATH/EncryptionKeyFile
	rm -f $SPIRIT_TMPATH/spirit-*.jks 2>/dev/null
	rm -f $SPIRIT_TMPATH/spirit-*.bcfks 2>/dev/null
	rm -rf $TMPATH/keystore $TMPATH/truststore
	rm -f $TMPATH/smgrca
	rm -f $TMPATH/.key
	rm -f $TMPATH/CRDUpgrade*xml


	# restore the default spirit truststore 
	if [ -f $SPIRIT_TMPATH/spirit-trust.bcfks.bak ]
	then
		cp $SPIRIT_TMPATH/spirit-trust.bcfks.bak \
			$SPIRIT_TMPATH/spirit-trust.bcfks
	fi

	# we don't want files from the upgrade either
	if [ -e $UPGRADE_PATH ]
	then
		cleanupUpgrade
		UPGRADE=no
	fi

}

# Simple function that moves the files from /opt/Avaya/upgrade
# to /opt/Avaya/prev-upgrade
function cleanupUpgrade() {
	echo "Saving upgrade files to $PREV_UPGRADE_PATH"
	rm -rf $PREV_UPGRADE_PATH
	mv $UPGRADE_PATH $PREV_UPGRADE_PATH
}

# Cleans up files after a failed attempt
function failureCleanUp() {

	# Some files are generated even in the failure case. 
	# Remove them so next time initTM can start fresh.
	rm -f $TMPATH/TMClientInv.xml 2>/dev/null
	rm -rf $TMPATH/truststore 2>/dev/null
}

#######################################################################
# Function that upgrades the certificates
#######################################################################
function upgrade() {

	# should never be the case
	[ "$UPGRADE" = "no" ] && return

	echo "Upgrading saved certificates"
	$COMMON_HOME/bin/tm/upgrade_certs.sh
	cleanupUpgrade
}

#######################################################################
# main
#######################################################################
for arg in "$@"
do
	if [ "{$arg}" = "{-f}" -o "{$arg}" = "{--force}" ]
	then
		# Force initialization even if trust has already been 
		# established.
		FORCE=yes
        elif [ "{$arg}" = "{-c}" -o "{$arg}" = "{--clean}" ]
	then
		# clear configuration data but do not re-init
		CLEAN_ONLY=yes

	elif [ "{$arg}" = "{-r}" -o "{$arg}" = "{--retry}" ]
	then
		MAX_NUMBER_OF_ATTEMPTS=10
	elif [ "{$arg}" = "{-s}" -o "{$arg}" = "{--silent}" ]
	then
		SILENT=yes
	elif [ "{$arg}" = "{-?}" -o "{$arg}" = "{-h}" -o "{$arg}" = "{--help}" ]
	then
		printUsage
		exit 0
	else
		case "$arg" in
			-*)
				printUsage
				exit 1;;

			*)  SCEPPassword=$arg;;
		esac
	fi
done

check_smgr_details

source $COMMON_SERVICES/install.properties

# set SCEP related variable(s) (needs to be done after install.properties)
SCEPURL="https://$SmgrIP"

setup_logging

if [ "$FORCE" = "yes" ] || [ "$CLEAN_ONLY" = "yes" ]
then
	force

	if [ "$CLEAN_ONLY" = "yes" ] 
	then
		# Stop processes requiring certificates
		echo "Stopping applications..."
		    service spiritAgent stop
		# Do not initialize at this time
		exit 0
	fi
fi


if [ -f $TMPATH/CRDThirdParty_Input.xml ]
then
	# Set a flag if all of the initialization is going to be done without
	# SMGR.  This is the case for hardened systems with EJBCA removed.
	grep -q SCEP $TMPATH/CRDThirdParty_Input.xml 
	[ $? != 0 ] && THIRDPARTY_INIT=yes
fi

# Download and saved the SMGR root CA so it can be referenced
# in other scripts like reconfigure.  Not required if only third
# party certificates are being loaded.
SCEP_PATH="$SCEPURL:10443/ejbca/publicweb/apply/scep/pkiclient.exe"
SMGRCA=$TMPATH/truststore/smgrca

if [ ! -f $SMGRCA ] && [ "$THIRDPARTY_INIT" = "no" ]
then
	GETCA_URL="${SCEP_PATH}?operation=GetCACert&message=CAID"
	
	log "Downloading System Manager root CA"
	mkdir $TMPATH/truststore &> /dev/null
	curl -k -f --connect-timeout 15 --silent $GETCA_URL > $SMGRCA
	RC=$?
	if [ "$RC" != "0" ]
	then
		log "Unable to download System Manager root CA, rc = $RC"
		rm -f $SMGRCA
	fi
fi

# See if this is an upgrade by checking the upgrade path.  The upgrade.sh
# will create the upgrade path.
if [ -f $UPGRADE_PATH/$TMPATH/TMClientInv.xml ]
then
	UPGRADE=yes
fi

	
# Determine if initialization is necessary. Count the number
# of expected certs and compare to the number of certs in the
# backup.  If the expected number of certs are greater than
# the number in the back, then initialization is required to 
# pick up the new cert(s).
EXPECTED_CERTS=$(grep "<ServiceID>" $TMPATH/CRDJEE_Input.xml | wc -l)
if [ "$UPGRADE" = "yes" ]
then
	BACKUP_CERTS=$(grep "<ServiceID>" \
		$UPGRADE_PATH/$TMPATH/CRDJEE_Input.xml | wc -l)

	if [ $EXPECTED_CERTS -gt $BACKUP_CERTS ]
	then
		# Upgrading from an older release so set up the CRD files
		# to match previous certificates.
		log "Setting up CRD file for upgrade"
		$COMMON_HOME/bin/tm/setup_for_upgrade.py
	else
		# The upgrade path has all of the necessary certs so
		# restore them.
		INITIALIZE=no
	
		$COMMON_HOME/bin/tm/restore_trust.sh

		cleanupUpgrade
	fi
else
	# The user is running initTM again after trust
	# has already been established.  More than
	# likely they are doing this to run initDRS.
	CERT_COUNT=$(find $TMPATH/keystore -name "*.pem" -o -name "*.bcfks" \
		-o -name "*.jks" 2>/dev/null | wc -l 2>/dev/null)
	[ $EXPECTED_CERTS -le $CERT_COUNT ] && INITIALIZE=no
fi

if [ "$UPGRADE" = "yes" ] && [ ! -f $UPGRADE_PATH/$TMPATH/EncryptionKeyFile ]
then
	# The encryption changed so it is necessary to upgrade the 
	# TMClientInv file.
	$JAVA_CMD -cp "$TM_CLASSPATH" TMInvUpgrade $UPGRADE_PATH/$TMPATH \
		&> /dev/null
fi
	
$COMMON_HOME/bin/tm/setup_trust_config.sh

# only collect the password if it is not already set and it is needed
# to initialize
if [ -z "$SCEPPassword" -a "$INITIALIZE" = "yes" -a "$THIRDPARTY_INIT" = "no" ]
then
	# collect the password
	while [ "$SCEPPassword" == "" ]
	do
		collectEnrollmentPassword
	done
fi

log "Trust Management Initialization started."

if [ "$INITIALIZE" = "yes" ]
then
	# make sure that we can ping the SMGR
	i="0"
	while [ $i -lt 10 ]
	do
		pingcheck $SmgrIP "System Manager"
		if [ $? -eq 0 ]
		then
			break
		fi
		i=$[$i+1]
		log "Trying again in 60 seconds"
		sleep 60
	done

	if [ $i = 10 ]
	then
		log "Failed to contact $SmgrIP"
		exit 1
	fi

	# get the certs from SMGR
	initialize

	# Now that all of the CRD files and certificate configuration
	# is done, restore the previous certificates.
	[ "$UPGRADE" = "yes" ] && upgrade

else
		log "Trust already established, skipping"
fi

# Clear the keystore passwords from the input files
for CRD_FILE in $TMPATH/CRDJEETM.xml $TMPATH/CRDUpgradeTM.xml
do
	if [ -f $CRD_FILE ]
	then
		sed -i "s/\(<CertStorePassword>\).*\(<.*>\)/\1\2/" $CRD_FILE
	fi
done

# Adjust the permissions because jboss is not run as root
chown -R java:root $TMPATH
chmod 750 $TMPATH/keystore $TMPATH/truststore $TMPATH/revocation
chmod 700 $TMPATH/trustedcerts
chmod -R 600 $TMPATH/keystore/*
chmod -R 600 $TMPATH/truststore/*
chmod 640  $TMPATH/keystore/tmservice* $TMPATH/truststore/tmservice*

log "Setting up SAL-Agent certificates"
$COMMON_HOME/bin/tm/update_spirit_certs.sh -r

# Restore third party cert if initTM is initialized for the first time
if [ ! -f $RESTORE_PATH ]
then
     $COMMON_HOME/bin/tm/certTool.py -r -s spirit -f $fips_mode
fi

# Setup Syslog Certificates and Restart Syslog
log "Setting up Syslog certificates and Restarting Syslog."
$COMMON_HOME/bin/tm/update_syslog_certs.sh -r

log "Setting up LDAP certificate"
$COMMON_HOME/bin/tm/update_ldap_certs.sh

log "Trust Management initialization completed successfully."

log "Checking to see if tm-client service is running or not."

/usr/bin/systemctl enable trust_client

$COMMON_HOME/bin/tm/trust_service.sh status | grep -q "not"

if [ $? -eq 0 ]; then
 echo "Starting the service"
 service trust_client start
fi

