#!/bin/bash

#############################################################################
#
# Copyright Avaya Inc., All Rights Reserved.
#
# THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Avaya Inc.
#
# The copyright notice above does not evidence any actual or intended
# publication of such source code.
#
# Some third-party source code components may have  been modified from their
# original versions by Avaya Inc.
#
# The modifications are Copyright Avaya Inc., All Rights Reserved.
#
# Avaya - Confidential & Restricted. May not be distributed further without
# written permission of the Avaya owner.
#
#############################################################################
#
# This script changes JBoss revocation settings
#

MGMT_CONFIG_FILE=/opt/jboss/bin/standalone.conf


DEFAULT_REVOCATION_FOLDER=/opt/avaya/tm/revocation
DEFAULT_CRL_FOLDER=${DEFAULT_REVOCATION_FOLDER}/crl
DEFAULT_REVOCATION_PROPERTIES_FILE=${DEFAULT_REVOCATION_FOLDER}/revocation.properties

CRL_FOLDER=$DEFAULT_CRL_FOLDER
REVOCATION_PROPERTIES_FILE=$DEFAULT_REVOCATION_PROPERTIES_FILE
REVOCATION_PROPERTIES_FOLDER=`dirname $REVOCATION_PROPERTIES_FILE`

# Revocation option properties
PROPERTY_REVOCATION_VALIDATION=REVOC_VALIDATION
PROPERTY_REVOCATION_TYPE=REVOC_TYPE
PROPERTY_REVOCATION_TYPE_PREFERENCE=TYPE_PREFERENCE
PROPERTY_REVOCATION_CHECK_METHOD=CHECK_METHOD

function updateProperties()
{
    # Updates the revocation properties file to current release 
    # or creates a default revocation properties file if one doesn't exist.
    if [ -e $REVOCATION_PROPERTIES_FILE ]
    then
            # update existing revocation.properties file to current release
            sed -i "s/RevocationValidation=/${PROPERTY_REVOCATION_VALIDATION}=/" $REVOCATION_PROPERTIES_FILE
            sed -i "s/RevocationTypePreference=/${PROPERTY_REVOCATION_TYPE_PREFERENCE}=/" $REVOCATION_PROPERTIES_FILE
            sed -i "s/RevocationType=/${PROPERTY_REVOCATION_TYPE}=/" $REVOCATION_PROPERTIES_FILE
            sed -i "s/CheckMethod=/${PROPERTY_REVOCATION_CHECK_METHOD}=/" $REVOCATION_PROPERTIES_FILE
    else
            # create a revocation.properties file using default values
            # unless other non-default options are also include on the command line
            updateConfig=true
    fi
}

function logMessage()
{
    logger -t revocationOptions.sh -p daemon.info "$1"
}

function getDebug()
{
	optionString="JAVA_OPTS=\"\$JAVA_OPTS -Djava.security.debug=certpath\""
	grep "$optionString" $MGMT_CONFIG_FILE
}

function setDebug()
{
	# Enable/disable Debug
	enableDebug="$1"

	# Set JBoss Debug options
      if [ -e $MGMT_CONFIG_FILE ]
      then
		optionString="JAVA_OPTS=\\\"\\\$JAVA_OPTS -Djava.security.debug=certpath\""
	  	if [ "$enableDebug" = "true" ]
  		then
  			# enable
	  		sed -i "/${optionString}/s/.*/${optionString}/" $MGMT_CONFIG_FILE
	  	else
  			# disable
	  		sed -i "/${optionString}/s/.*/\# ${optionString}/" $MGMT_CONFIG_FILE
	  	fi
	else
		echo -e "\nCannot set debug option for JBoss! Config file not found.\n"
	fi

	if [ "$WAS_DEBUG_REVOCATION" == "true" ] 
	then
		# Set WebSphere Debug options
	      if [ -e $WAS_CONFIG_FILE ]
	      then
			# disable
			sed -i.bak s/'"-Djava.security.debug=certpath" '//g $WAS_CONFIG_FILE
			if [ "$enableDebug" = "true" ]
			then
				# enable
				sed -i s/'$DEBUG'/'"-Djava.security.debug=certpath" $DEBUG'/ $WAS_CONFIG_FILE
			fi
		else
			echo -e "\nCannot set debug option for WebSphere! Config file not found.\n"
		fi
	fi

	logMessage "enableDebug=$enableDebug"
}

function init() {
        # init - clears revocationProperties file and crl folder
        logMessage "init: %REVOCATION_PROPERTIES_FILE $CRL_FOLDER"
        rm -f $REVOCATION_PROPERTIES_FILE
        rm -f $CRL_FOLDER/*
}

function printUsage() {

/bin/cat << EOF

Usage: revocationOptions.sh [OPTIONS] 

 -n, --none       disable revocation checking
 -b, --besteffort enable best-effort revocation checking
 -m, --mandatory  enable mandatory revocation checking
 -c, --crl        enable CRLs
 -o, --ocsp       enable OCSP
 -e, --endentity  enable end-entity only checking
 -v, --verbose    enable verbose output
 -l, --list       list CRLs
 --debug          enable debug
 --nodebug        disable debug
 --init           delete revocation.properties file and CRLs
 --purge          purge (delete) CRLs
 --touch          touch CRLs
 --update         update revocation.properties file to current release 
EOF

}

function purgeCrls() {
	# purge crls
	logMessage "purge crls: $CRL_FOLDER"
	rm -f $CRL_FOLDER/*
}

function setRevocationOptions()
{
    Header="# Revocation Properties updated by `logname` on `date` (`basename $0`)"
    (
        echo "$Header"
        echo "$PROPERTY_REVOCATION_VALIDATION=$revocationValidation"
        echo "$PROPERTY_REVOCATION_TYPE=$revocationType"
        echo "$PROPERTY_REVOCATION_TYPE_PREFERENCE=$revocationTypePreference"
        echo "$PROPERTY_REVOCATION_CHECK_METHOD=$revocationCheckMethod"
        
    ) > $REVOCATION_PROPERTIES_FILE
    chmod 644 $REVOCATION_PROPERTIES_FILE > /dev/null 2>&1
    chown java $REVOCATION_PROPERTIES_FILE > /dev/null 2>&1
    chgrp susers $REVOCATION_PROPERTIES_FILE > /dev/null 2>&1
    logMessage "`cat $REVOCATION_PROPERTIES_FILE`"
}

function showCrls() {
	if [ -e "${CRL_FOLDER}" ]
	then
		echo "$CRL_FOLDER:"
		cd $CRL_FOLDER

		for crlFile in *.pem
		do 
			if [ -e "$crlFile" ]
			then
				ls -l $crlFile
			fi
		done

		if [ "$enableVerbose" = "true" ]
		then
			for crlFile in `find $CRL_FOLDER -type f -name "*.pem" -print`
			do 
				echo
				echo $crlFile
				echo "=============================================================================="
				openssl crl -in  $crlFile -text
			done
		fi
	fi
}

function touchCrls() {
	# touch crls
	logMessage "touch crls: $CRL_FOLDER"
	for i in $CRL_FOLDER/*
	do
		if [ -e "$i" ]
		then
	      	touch $i
		fi
	done
}

#######################################################################
# main
#######################################################################

# Set default values
revocationValidation="NONE"
revocationType="CRL"
revocationTypePreference="CRL"
revocationCheckMethod="ONLY_END_ENTITY"

enableCrl=false
enableOcsp=false
preferOcspIfBoth=false

enableDebug=false
disableDebug=false
enableInit=false
enablePurgeCrls=false
enableTouchCrls=false
enableVerbose=false
listCrls=false
updateConfig=false

for arg in "$@"
do
	if [ "{$arg}" = "{-n}" -o "{$arg}" = "{--none}" ]
	then
		# Disable revocation
		revocationValidation="NONE"
		updateConfig=true
	elif [ "{$arg}" = "{-b}" -o "{$arg}" = "{--besteffort}" ]
	then
		# Best effort revocation
		revocationValidation="BEST_EFFORT"
		updateConfig=true
	elif [ "{$arg}" = "{-m}" -o "{$arg}" = "{--mandatory}" ]
	then
		# Mandatory revocation
		revocationValidation="MANDATORY"
		updateConfig=true
	elif [ "{$arg}" = "{-c}" -o "{$arg}" = "{--crl}" ]
	then
		# Enable crl
		enableCrl=true
		preferOcspIfBoth=true
		updateConfig=true
	elif [ "{$arg}" = "{-o}" -o "{$arg}" = "{--ocsp}" ]
	then
		# Enable ocsp
		enableOcsp=true
		preferOcspIfBoth=false
		updateConfig=true
	elif [ "{$arg}" = "{-e}" -o "{$arg}" = "{--endentity}" ]
	then
		# Enable end entity only
		revocationCheckMethod="ONLY_END_ENTITY"
		updateConfig=true
	elif [ "{$arg}" = "{-l}" -o "{$arg}" = "{--list}" ]
	then
		# List CRLs
		listCrls=true
	elif [ "{$arg}" = "{--init}" ]
	then
		# Init CRLs
		enableInit=true
	elif [ "{$arg}" = "{--purge}" ]
	then
		# Purge CRLs
		enablePurgeCrls=true
	elif [ "{$arg}" = "{--touch}" ]
	then
		# Touch CRLs
		enableTouchCrls=true
	elif [ "{$arg}" = "{--debug}" ]
	then
		# Enable debug
		enableDebug=true
	elif [ "{$arg}" = "{--nodebug}" ]
	then
		# Enable debug
		disableDebug=true
    elif [ "{$arg}" = "{--update}" ]
    then
            # update or create default properties for current release
            updateProperties		
	elif [ "{$arg}" = "{-v}" -o "{$arg}" = "{--verbose}" ]
	then
		# Enable verbose
		enableVerbose=true
	elif [ "{$arg}" = "{-?}" -o "{$arg}" = "{-h}" -o "{$arg}" = "{--help}" ]
	then
		printUsage
		exit 0
	else
		case "$arg" in
			-*)
				printUsage
				exit 1;;
			*)  	
				printUsage
				exit 1;;
		esac
	fi
done

if [ "$enableCrl" = "true" ] 
then
	if [ "$enableOcsp" = "true" ]
	then
		revocationType="BOTH"
		if [ "$preferOcspIfBoth" = "true" ]
		then
			revocationTypePreference="OCSP"
		else
			revocationTypePreference="CRL"
		fi
	else
		revocationType="CRL"
		revocationTypePreference="CRL"
	fi
else
	if [ "$enableOcsp" = "true" ]
	then
		revocationType="OCSP"
		revocationTypePreference="OCSP"
	else
		revocationType="CRL"
		revocationTypePreference="CRL"
	fi
fi

if [ ! -e "$CRL_FOLDER" ]
then
	echo "No CRL Folder ($CRL_FOLDER)"
fi


if [ ! -e "$REVOCATION_PROPERTIES_FOLDER" ]
then
	echo "No Revocation Properties Folder ($REVOCATION_PROPERTIES_FOLDER)"
else 
	if [ "$updateConfig" = "true" ]
	then
		setRevocationOptions
	fi
fi

if [ "$enableDebug" = "true" ]
then
	echo "Enabling Debug Logging..."
	setDebug true
fi

if [ "$disableDebug" = "true" ]
then
	echo "Disabling Debug Logging..."
	setDebug false
fi

if [ "$enableTouchCrls" = "true" ]
then
	echo "Touching CRLs..."
	touchCrls
fi

if [ "$enablePurgeCrls" = "true" ]
then
	echo "Purging CRLs..."
	purgeCrls
fi

if [ "$enableInit" = "true" ]
then
	echo "Deleting Revocation Properties File and CRLs..."
	init
fi

if [ -e "$REVOCATION_PROPERTIES_FILE" ]
then
	cat $REVOCATION_PROPERTIES_FILE
else
	echo "No Revocation Properties File ($REVOCATION_PROPERTIES_FILE)"
fi

if [ "$listCrls" = "true" ]
then
	echo
	showCrls
fi

if [ "$enableVerbose" = "true" ]
then
	echo
	if [ "$REVOCATION_PROPERTIES_FILE" != "$DEFAULT_REVOCATION_PROPERTIES_FILE" ]
	then
		echo "Revocation Properties File (${REVOCATION_PROPERTIES_FILE}) not set to default (${DEFAULT_REVOCATION_PROPERTIES_FILE})"
	else
		echo "Revocation Properties File: ${REVOCATION_PROPERTIES_FILE}"
	fi

	if [ "$CRL_FOLDER" != "$DEFAULT_CRL_FOLDER" ]
	then
		echo "CRL Folder (${CRL_FOLDER}) not set to default (${DEFAULT_CRL_FOLDER})"
	else
		echo "CRL Folder: ${CRL_FOLDER}"
	fi
	echo
	getDebug 

fi
