#!/bin/bash

#########################################################################
# Update Security Module with the latest CRLs.  
#
# Security Module CRLs are stored in:
#	/opt/Avaya/asset/packages/active-version/etc/cert/crl
#
# This script takes the trust management administered CRLs and
# distibutes them to the Security Module CRL directory.
#########################################################################

source /opt/Avaya/install.properties
source /opt/Avaya/Session_Manager/bin/functions.sh

ASSET_ROOT=/opt/Avaya/asset/packages/active-version
ASSET_BIN=$ASSET_ROOT/bin
ASSET_CONFIG=$ASSET_ROOT/etc
ASSET_CRL_DIR=$ASSET_CONFIG/cert/crl
ASSET_CRL_BACKUP_DIR=$ASSET_CRL_DIR/backup
ASSET_SMGR_CRL_DIR=$ASSET_CRL_DIR/smgr_crl
NGINX_CRLS_PEM_FILE=http_crls.pem
JBOSS_HOME=/opt/Avaya/wildfly-10.1.0.Final
TM_HOME=$JBOSS_HOME/standalone/configuration/tm
TM_CRL_DIR=$TM_HOME/revocation/crl
CRL_PEM_FILE_FORMAT=_crl.pem
NGINX_CRL_MERGE_SCRIPT=$ASSET_BIN/updateNginxCRL.sh


mkdir -p ${ASSET_CRL_BACKUP_DIR}
mkdir -p $ASSET_SMGR_CRL_DIR

###############
# CRL PEM files
###############

# Make a backup of the current files
# SMGR replicated CRLs have common format of <crl_text_id>${CRL_PEM_FILE_FORMAT}
crlPemFiles=$(ls ${ASSET_CRL_DIR}/*${CRL_PEM_FILE_FORMAT} 2> /dev/null | wc -l)
if [ $crlPemFiles -gt 0 ]
then
	cp -p --force ${ASSET_CRL_DIR}/*${CRL_PEM_FILE_FORMAT} ${ASSET_CRL_BACKUP_DIR}/.
	if [ $? -ne 0 ]
	then
		echo "Failed to backup Security Module CRL PEM files"
                # For now just complain backup failed without exit
        	#exit 1
	fi 
	chmod 400 ${ASSET_CRL_BACKUP_DIR}/*${CRL_PEM_FILE_FORMAT}
fi

# Copy PEM format CRLs from TM truststore CRL to Asset CRL repository
# First remove current CRL PEM files since latest CRL refresh may have removed some
mkdir -p $ASSET_SMGR_CRL_DIR

for crl in `ls $ASSET_SMGR_CRL_DIR/*${CRL_PEM_FILE_FORMAT} 2> /dev/null`
do
	echo "Removing CRL ${ASSET_CRL_DIR}/$(basename $crl)"
	rm -f ${ASSET_CRL_DIR}/$(basename $crl)
done 

# Empty SMGR CRL PEM file cache prior to refresh
rm -f $ASSET_SMGR_CRL_DIR/*${CRL_PEM_FILE_FORMAT}

for crl in `ls ${TM_CRL_DIR}/*${CRL_PEM_FILE_FORMAT} 2> /dev/null`
do
	echo "Copying CRL $crl to ${ASSET_CRL_DIR}"
	cp --force $crl ${ASSET_CRL_DIR}/.
	if [ $? -ne 0 ]
	then
		echo "Failed to copy current Security Module CRLs to backup"
        	exit 1
	fi

	chmod 440 ${ASSET_CRL_DIR}/$(basename $crl)
	chown asset:asset ${ASSET_CRL_DIR}/$(basename $crl)

	# keep record of this update
	touch $ASSET_SMGR_CRL_DIR/$(basename $crl)
done

# Remove expired CRL PEM files
for crl in `ls ${ASSET_CRL_DIR}/*${CRL_PEM_FILE_FORMAT} 2> /dev/null`
do
	nextupdate=`openssl crl -in $crl -noout -nextupdate`
	if [ `echo $nextupdate | awk -F= '{print $2}' | xargs -I{} date -d {} +%s` -le `date +%s` ]
	then
		echo "Removing expired CRL $crl $nextupdate"
		rm -f $crl
		rm -f $ASSET_SMGR_CRL_DIR/$(basename $crl)
	#else
		#echo "Not expired CRL $crl $nextupdate"
	fi
done


###############
# CRL DER files
###############

# Make backup of existing DER files
# Assume all CRL DER files using r0 suffix
crlDerFiles=$(ls ${ASSET_CRL_DIR}/*.r0 2> /dev/null | wc -l)
if [ $crlDerFiles -gt 0 ]
then
        cp -p --force ${ASSET_CRL_DIR}/*r0 ${ASSET_CRL_BACKUP_DIR}/.
        if [ $? -ne 0 ]
        then
                echo "Failed to backup Security Module CRL DER files"
                # For now just complain backup failed without exit
                #exit 1
        fi
        chmod 400 ${ASSET_CRL_BACKUP_DIR}/*.r0
fi

# First remove current CRL DER files since latest CRL refresh may have removed some
for crl in `ls $ASSET_SMGR_CRL_DIR/*.r0 2> /dev/null`
do
	echo "Removing CRL ${ASSET_CRL_DIR}/$(basename $crl)"
	rm -f ${ASSET_CRL_DIR}/$(basename $crl)
done 

# Empty SMGR CRL DER file cache prior to refresh
rm -f $ASSET_SMGR_CRL_DIR/*.r0

for crl in `ls ${TM_CRL_DIR}/*${CRL_PEM_FILE_FORMAT} 2> /dev/null`
do
	crlhash=`openssl crl -in $crl -hash -noout`
        if [ $? -ne 0 ]
        then
		echo "Failed to get hash for CRL $crl"
	else
		crlhashfile=${ASSET_CRL_DIR}/$crlhash.r0	
		openssl crl -in $crl -outform DER -out $crlhashfile
        	if [ $? -ne 0 ]
        	then
			echo "Failed to transform CRL $crl to $crlhashfile"
		else
			echo "Transformed CRL $crl to $crlhashfile"

			# Remove the CRL if already expired
			nextupdate=`openssl crl -inform DER -in $crlhashfile -noout -nextupdate`
			if [ `echo $nextupdate | awk -F= '{print $2}' | xargs -I{} date -d {} +%s` -le `date +%s` ]
			then
				echo "Removing expired CRL $crlhashfile $nextupdate"
				rm -f $crlhashfile
			else
				#echo "Not expired CRL $crlhashfile $nextupdate"
				
				# keep record of this CRL update
				touch $ASSET_SMGR_CRL_DIR/$(basename $crlhashfile)

				chmod 440 $crlhashfile
				chown asset:asset $crlhashfile
			fi
		fi
	fi
done


#############################################
# Setup HTTP CRLs using PEM files
#############################################

# nginx requires a single file with all PEM format CRLs
cp -p --force ${ASSET_CRL_DIR}/${NGINX_CRLS_PEM_FILE} ${ASSET_CRL_BACKUP_DIR}/.

if [ -x ${NGINX_CRL_MERGE_SCRIPT} ]
then
	# Use nginx CRL merge script with refresh option to update with all
	# current available CRLs
	${NGINX_CRL_MERGE_SCRIPT} refresh
else
	cat ${ASSET_CRL_DIR}/*${CRL_PEM_FILE_FORMAT} > ${ASSET_CRL_DIR}/${NGINX_CRLS_PEM_FILE}
	chmod 440 ${ASSET_CRL_DIR}/${NGINX_CRLS_PEM_FILE}
fi
chown asset:asset ${ASSET_CRL_DIR}/${NGINX_CRLS_PEM_FILE}

# Set sym link used by nginx for reading CRL PEM file
ln -s --force ${ASSET_CRL_DIR}/${NGINX_CRLS_PEM_FILE} ${ASSET_CONFIG}/cert/http/activeCRLFile
