#!/bin/bash
# vim:ts=4:sw=4

#############################################################################
#
# Copyright Avaya Inc., All Rights Reserved.
#
# THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF Avaya Inc.
#
# The copyright notice above does not evidence any actual or intended
# publication of such source code.
#
# Some third-party source code components may have  been modified from their
# original versions by Avaya Inc.
#
# The modifications are Copyright Avaya Inc., All Rights Reserved.
#
# Avaya - Confidential & Restricted. May not be distributed further without
# written permission of the Avaya owner.
#
#############################################################################

#############################################################################
# Upgrades the certificates from previous releases.
#
# Return Codes:
# 0 - Upgrade complete successfully
# 1 - Upgrade complete, need to reinstall demo certs
# 2 - Upgrade failed
#############################################################################

TMPATH=/opt/avaya/tm
UPG_PATH=/opt/Avaya/upgrade/$TMPATH
COMMON_HOME=/opt/util
INSTALL_PROPS=/opt/avaya/common_services/install.properties
JAVA=/usr/java/latest/bin/java
KEYTOOL=/usr/java/latest/bin/keytool
BCKEYTOOL=$COMMON_HOME/bin/bckeytool
TMCLIENT=$COMMON_HOME/bin/tm/TMClient.sh

# Identity certs keystores 
MGMT_KEYSTORE_JKS="keystore/container_keystore.jks"
MGMT_KEYSTORE="keystore/container_keystore.bcfks"
SPIRIT_KEYSTORE_JKS="keystore/spirit-identity.jks"
SPIRIT_KEYSTORE="keystore/spirit-identity.bcfks"
SYSLOG_KEYSTORE="keystore/syslog_keystore.pem"


# Trusted lists
SYSLOG_TRUSTSTORE="truststore/syslog_truststore.pem"

source $INSTALL_PROPS
source $COMMON_HOME/bin/tm/TMEnv.sh

logger='logger -t tmupgrade'

# return code
RC=0

###################################################################
# Issued by SMGR CA
###################################################################
function isIssuedBySMGR() {
	
	CERT_TXT_FILE=$1
	CERT_FILE=$(echo $CERT_TXT_FILE | sed 's/\.txt//' | xargs basename)

	if [ ! -f $CERT_TXT_FILE ]
	then
		$logger "$CERT_FILE is a new certificate"
		return 1
	fi
	
	grep -q "Issuer:.*CN=default" $CERT_TXT_FILE
	if [ $? = 0 ]
	then
		$logger "$CERT_FILE is issued by older SMGR"
		return 1
	fi
	
	grep -q "Issuer:.*System Manager" $CERT_TXT_FILE
	if [ $? = 0 ]
	then
		$logger "$CERT_FILE is issued by SMGR"
		return 1
	fi
	
	# Handle the case where the SMGR CA is set to a different CN
	if [ -f $UPG_PATH/truststore/smgrca ]
	then
		SMGRCA=$(openssl x509 -inform der -noout -issuer -in \
			$UPG_PATH/truststore/smgrca)
	elif [ -f $TMPATH/truststore/smgrca ]
	then
		SMGRCA=$(openssl x509 -inform der -noout -issuer -in \
			$TMPATH/truststore/smgrca)
	fi
	SMGR_CN=$(echo $SMGRCA | sed 's/.*CN=\(.*\)/\1/' | cut -d'/' -f1)
	
	$logger "SMGR CA = $SMGRCA"
	if [ "$SMGRCA" != "" ]
	then
		grep -q "Issuer:.*$SMGR_CN" $CERT_TXT_FILE
		if [ $? = 0 ]
		then
			$logger "$CERT_FILE is issued by SMGR"
			return 1
		fi
	fi

	$logger "$CERT_FILE is NOT issued by SMGR"
	return 0
	
}

###################################################################
# Upgrades Management Certificates
###################################################################
function upgradeMgmtIdCert() {

	if [ -f $UPG_PATH/$MGMT_KEYSTORE_JKS.txt ]
	then
		isIssuedBySMGR $UPG_PATH/$MGMT_KEYSTORE_JKS.txt
	else
		isIssuedBySMGR $UPG_PATH/$MGMT_KEYSTORE.txt
	fi
	USING_SMGR_CA=$?

	if [ $USING_SMGR_CA == 1 ]
	then
		$logger "Keeping newer version of SMGR issued Mgmt certificate"
	else
		$logger "Replacing $MGMT_KEYSTORE with upgrade version"
		
		if [ -f $UPG_PATH/$MGMT_KEYSTORE_JKS ]
		then
			cp -f $UPG_PATH/$MGMT_KEYSTORE_JKS $TMPATH/$MGMT_KEYSTORE_JKS

			# keystore will be converted to bcfks below
			rm -f $TMPATH/$MGMT_KEYSTORE
		else
			cp -f $UPG_PATH/$MGMT_KEYSTORE $TMPATH/$MGMT_KEYSTORE
		fi
	fi
}


###################################################################
# Upgrading SPIRIT's ID Certificate
###################################################################
function upgradeSpiritIdCert() {

	# Due to a password change from a hardcoded password, do not bring
	# forward the old SPIRIT keystore (jks format). 
	if [ -f $UPG_PATH/$SPIRIT_KEYSTORE_JKS ]
	then
		$logger "Discarding JKS version of SPIRIT keystore, using new version"
	else
		isIssuedBySMGR $UPG_PATH/$SPIRIT_KEYSTORE.txt
		USING_SMGR_CA=$?
	
		if [ $USING_SMGR_CA = 1 ]
		then
			$logger "Keeping newer version of SMGR issued WebSphere certificate"
		else
			$logger "Replacing $SPIRIT_KEYSTORE with upgrade version"
			cp -f $UPG_PATH/$SPIRIT_KEYSTORE $TMPATH/$SPIRIT_KEYSTORE
		fi
	fi
		
}


# Upgrade Syslog certificate (introduced in 8.1)
function upgradeSyslogIdCert() {

    isIssuedBySMGR $UPG_PATH/$SYSLOG_KEYSTORE.txt
    USING_SMGR_CA=$?
    
    if [ $USING_SMGR_CA = 1 ]
    then
        $logger "Keeping newer version of SMGR issued Syslog certificate"
    else
        $logger "Replacing $SYSLOG_KEYSTORE with upgrade version"
        cp -f $UPG_PATH/$SYSLOG_KEYSTORE $TMPATH/$SYSLOG_KEYSTORE
    fi
}

###################################################################
# MAIN
###################################################################
if [ ! -d $UPG_PATH ]
then
	echo "Upgrade directory does not exist, exiting."
	exit 2
fi

$logger "Starting trust upgrade"


if [ -f $UPG_PATH/TMClientInv.xml ]
then
		upgradeMgmtIdCert
		upgradeSpiritIdCert
		upgradeSyslogIdCert

fi

$logger "Merging trust stores"
$JAVA -Dlog4j.configuration=$TM_LOG4J -classpath "$TM_CLASSPATH" \
	com.avaya.asm.mgmt.tmcli.TMUpgrade

# make sure the copied files have the correct permissions
chown -R java:root $TMPATH

# Make sure the files are in bcfks format
$TMCLIENT convertKeystoresToBCFKS

$logger "Finished trust upgrade"

exit $RC
