# Copyright 2012 Avaya Inc. All Rights Reserved.

"""
Provides clients for IP Office Web Services API and Whois service.
"""

import unittest
import urllib.request as urllib2
import sys
import time
import threading
from http.server import BaseHTTPRequestHandler,HTTPServer
from xml.etree.ElementTree import XML, ElementTree, Element, SubElement

from core.system import shell
from core.common import log, version, configuration

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2011, Avaya Inc."

# Logging tag
TAG  = 'se-ipoffice-ws'


class WSError(Exception):
    """
    IP Office web service generic errors like
    network and authentication errors.
    """
    def __init__(self, message='', err_code=None, err_description=None):
        self.message = message
        self.err_code = err_code
        self.err_description = err_description

    def __repr__(self):
        return str(self.__dict__)

    def __str__(self):
        return str(self.__dict__)


class WSClient(object):
    """
    IP Office Web Services API client.

    Used to extract Server Edition system list from local IP Office.
    """
    DEFAULT_PROTOCOL    = 'https'
    DEFAULT_PORT        =  8443
    USER_AGENT          = 'WebControlPanel/' + version.APP_VERSION.get("VERSION") + "-" + version.APP_VERSION.get("RELEASE")
    DEFAULT_USER        = configuration.USERNAME
    DEFAULT_TIMEOUT     = 20 # socket operation timeout

    RETRIES = 3 # number of connection attempts
    DELAY   = 2 # interval in seconds between two connection attempts

    def __init__(self,
                 host='127.0.0.1',
                 protocol=DEFAULT_PROTOCOL,
                 port=DEFAULT_PORT,
                 username=DEFAULT_USER,
                 password=DEFAULT_USER,
                 timeout=DEFAULT_TIMEOUT):
        self.host = host
        self.protocol = protocol
        self.port = port
        self.ws_base_url = self.protocol + "://" + self.host + ":" + str(self.port) + "/ws"
        self.timeout = timeout
        username = self.get_ws_username(username)
        self.base_request = 'curl -k -A "%s" -u %s:%s %s' % (self.USER_AGENT, username, password.replace(" ", "\ "),
                                                             self.ws_base_url)
        auth_manager = urllib2.HTTPPasswordMgrWithDefaultRealm()
        auth_manager.add_password(None, self.ws_base_url, username, password)
        auth_handler = urllib2.HTTPBasicAuthHandler(auth_manager)
        self.opener = urllib2.build_opener(auth_handler)
        self.opener.addheaders = [('User-Agent', self.USER_AGENT)]

    # see IPOFFICE-47703
    def get_ws_username(self, user_name):
        if user_name.endswith('\\'):
            user_name = user_name + '\\'
        user_name = user_name.replace(':', '\\:')
        return user_name

    def get_authentication(self):
        xml_str = self._get("/security/authenticate")
        return xml_str

    def get_backup_status(self):
        xml_str = self._get('/backup/get_progress')
        return xml_str

    def get_upgrade_status(self):
        xml_str = self._get('/upgrade/get_progress')
        return xml_str

    def _get(self, url):
        ws_url = self.ws_base_url + url
        request = self.base_request + url
        retries = 0
        while True:
            retries += 1
            try:
                xml_str = shell.sudo_execute(request, parse=shell.RAW)
                log.debug(TAG, "GET: response for <%s>:\n%s" % (ws_url, xml_str))
                err = self._parse_error(xml_str)
                if err:
                    raise WSError("web service error <%s>" % str(err), err[0], err[1])
                else:
                    return xml_str
            except Exception as e:
                if retries < self.RETRIES:
                    time.sleep(self.DELAY)
                else:
                    err_code = None
                    err_description = None
                    if hasattr(e, 'err_code'):
                        err_code = getattr(e, 'err_code')
                    if hasattr(e, 'err_description'):
                        err_description = getattr(e, 'err_description')
                    raise WSError("web service http error <%s>" % str(e), err_code, err_description)

    def _parse_error(self, xml_str):
        doc = XML(xml_str)
        err = doc.find(".//error")
        if err:
            err_code = err.findtext(".//error_code")
            err_description = err.findtext(".//error_desc")
            return err_code, err_description
        return None


class TestGeneral(unittest.TestCase):

    def test_parse_error(self):
        xml_str = """<?xml version="1.0" encoding="UTF-8"?>
        <response status="0">
        <data>
        <ws_object>
        <Authentication>
        <error>
        <error_code>110</error_code>
        <error_desc>Service User not found</error_desc>
        <cause>1</cause>
        <cause_data>0</cause_data>
        </error>
        </Authentication>
        </ws_object>
        </data>
        </response>
        """

        wsclient = WSClient()
        err = wsclient._parse_error(xml_str)

        self.assertEqual('110', err[0])
        self.assertEqual('Service User not found', err[1])

class TestWSClientWithMockServer(unittest.TestCase):

    port = 54331

    def setUp(self):
        self.httpd = HTTPServer(('127.0.0.1', self.port), WSMockHandler)
        t = threading.Thread(target=self.httpd.serve_forever)
        t.start()

    def tearDown(self):
        self.httpd.shutdown()
        TestWSClientWithMockServer.port += 1

    def test_get_server_edition_systems(self):
        pass
        """
        wsclient = WSClient(protocol='http', port=self.port)
        servers = wsclient.get_server_edition_systems()

        self.assertEqual(4, len(servers))
        first_system = servers[0]
        self.assertEqual('127.0.0.1', first_system['NodeAddress'])
        self.assertEqual('PrimaryServer', first_system['NodeType'])
        self.assertEqual('Running', first_system['NodeState'])
        """


class WSMockHandler(BaseHTTPRequestHandler):

    def do_HEAD(self):
        self.send_response(200)
        self.send_header('Content-Type', 'text/xml')
        self.end_headers()

    def do_GET(self):
        self.send_response(200)
        self.send_header('Content-Type', 'text/xml')
        self.end_headers()

        if '/upgrade/get_progress' in self.path:
            # /upgrade/get_progress
            self.wfile.write("""<?xml version="1.0" encoding="UTF-8"?>
            <response status="1">
            <data>
            <ws_object>
            <ipo_object>
            <upgrade>
            <progress>
            <current>
            <ref>4</ref>
            <task>idle</task>
            <startDtTm>2011-07-12T00:46:18</startDtTm>
            <ms>284915285</ms>
            <msSoFar>42306215</msSoFar>
            </current>
            <previous>
            <ref>1</ref>
            <task>currentInventory</task>
            <step>collectingHighLevel</step>
            <stepStatus>completed</stepStatus>
            <startDtTm>2011-07-08T17:41:52</startDtTm>
            <startMS>133624</startMS>
            <secondsTotal>1</secondsTotal>
            <msTotal>298</msTotal>
            </previous>
            </progress>
            </upgrade>
            </ipo_object>
            </ws_object>
            </data>
            </response>
            """)
        else:
            # /system/general
            self.wfile.write("""<?xml version="1.0" encoding="UTF-8"?>
                <response status="0">
                  <data>
                    <record>
                      <systemName>ABE_VM</systemName>
                      <ipAddressLAN1>127.0.0.1</ipAddressLAN1>
                      <subnetMaskLAN1>255.255.255.0</subnetMaskLAN1>
                      <dhcpModeLAN1>2</dhcpModeLAN1>
                      <ipAddressLAN2>0.0.0.0</ipAddressLAN2>
                      <subnetMaskLAN2>0.0.0.0</subnetMaskLAN2>
                      <dhcpModeLAN2>1</dhcpModeLAN2>
                      <systemContact></systemContact>
                      <systemLocation></systemLocation>
                      <systemContactInfo></systemContactInfo>
                      <ServerEditionSystems>
                        <NodeInformation>
                          <NodeAddress>127.0.0.1</NodeAddress>
                          <NodeType>PrimaryServer</NodeType>
                          <NodeState>Running</NodeState>
                        </NodeInformation>
                        <NodeInformation>
                          <NodeAddress>192.168.42.237</NodeAddress>
                          <NodeType>SecondaryServer</NodeType>
                          <NodeState>Stopped</NodeState>
                        </NodeInformation>
                        <NodeInformation>
                          <NodeAddress>192.168.42.236</NodeAddress>
                          <NodeType>Expansion</NodeType>
                          <NodeState>Running</NodeState>
                        </NodeInformation>
                        <NodeInformation>
                          <NodeAddress>192.168.45.5</NodeAddress>
                          <NodeType>Expansion</NodeType>
                          <NodeState>Stopped</NodeState>
                        </NodeInformation>
                      </ServerEditionSystems>
                      <systemReboot>false</systemReboot>
                      <Last-Modified>Tue, 20 Dec 2011 13:32:23 GMT</Last-Modified>
                    </record>
                  </data>
                </response>""")

    def do_POST(self):
        if '/upgrade/cancel' in self.path:
            self.wfile.write("""<?xml version="1.0" encoding="utf-8"?>
                <response status="1"
                xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
                <data>
                <ws_object>
                <ipo_object gui="">
                <upgrade>
                <sId>sId</sId>
                <cancel>
                <successResult>
                <successText>Set recovery point cancelled</successText>
                </successResult>
                </cancel>
                </upgrade>
                </ipo_object>
                </ws_object>
                </data>
                </response>
                """)
        else:
            # /upgrade/pull_upgrade
            self.wfile.write("""<?xml version="1.0" encoding="utf-8"?>
                <response status="1" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
                <data>
                <ws_object>
                <ipo_object gui="">
                <upgrade>
                <sId>sId</sId>
                <pull_upgrade>
                <successResult>
                <successText>pull started</successText>
                </successResult>
                </pull_upgrade>
                </upgrade>
                </ipo_object>
                </ws_object>
                </data>
                </response>""")


class TestWSClientWithRealServer(unittest.TestCase):

    def test_authentication_failed(self):
        code, out, err = shell.sudo_execute2('service ipoffice status')
        if 'is running' in out:
            wsclient = WSClient(username='Not-Existing')
            self.assertRaises(WSError, wsclient.get_server_edition_systems)

    def test_get_business_edition_systems(self):
        pass
        """
        code, out, err = shell.sudo_execute2('service ipoffice status')
        if 'is running' in out:
            wsclient = WSClient()
            servers = wsclient.get_server_edition_systems()

            self.assertTrue(len(servers) >= 1)
        """

class TestWSClientConnectionFailed(unittest.TestCase):

    def test_get_midsize_edition_systems(self):
        wsclient = WSClient(host="something-that-does-not-exist", port=12345)
        self.assertRaises(WSError, wsclient.get_server_edition_systems)

if __name__ == '__main__':
    unittest.main()
