# Copyright 2011 Avaya Inc. All Rights Reserved.

"""
Sub-application for IP Office Midmarket Edition (8.1) features.
"""

import json
import os
import configobj
import web
import threading
from base64 import b64decode

from xml.etree.ElementTree import XML, ElementTree, Element, SubElement

import services_rest_api
import updates_rest_api
import audit

from core.common import configuration
from core.common import log
from core.common import vendor_i18n
from core.common import version
from core.common import utils
from core.services import manager
from core.common.decorators import synchronized
from core.system import network

from core.ext import ipoffice
from core.ext import apps_manager
from core.ext import security
from core.ext import ipoffice_ws


# Logging tag
TAG = 'serveredition-webapp'

# Used to synchronize IP Office configuration updates
ipoffice_lock = threading.Lock()

urls = ('/ipoffice/mode',           'IPOfficeModeResource',
        '/ipoffice/licence',        'IPOfficeLicenceResource',
        '/hardware/sid',            'HardwareIdResource',
        '/host_id',                 'HostIdResource',
        '/onex/config',             'OnexConfigResource',
        '/installer',               'InstallerResource',
        '/hardware/profiles',       'HardwareProfilesResource',
        '/ipoffice/companding',     'IPOfficeCompandingLawResource',
        '/ipoffice/virtualization', 'IPOfficeVirtualizationResource',
        '/autologin',               'Autologin',
        '/operation/status',        'OperationStatusResource')

APP = web.application(urls, locals())
URL = '/serveredition'

UNSECURED = {'GET': ('/hardware/sid',  '/host_id'),
             'PUT': (),
             'POST': ('/autologin'),
             'DELETE': ()}

def auth_processor(handle):
    """
    If the user is not authenticated the
    client will be redirected to login page.
    """
    if web.ctx.path in UNSECURED[web.ctx.method.upper()]:
        return handle()
    else:
        if not web.ctx.session.get('user'):
            web.ctx.status = '401'
        else:
            return handle()

APP.add_processor(auth_processor)


# Template render with i18n support
render = web.template.render('templates/', globals={'_': vendor_i18n.custom_gettext, 'build': version.BUILD_UID, 'csrf_token':configuration.csrf_token})

configuration.set_authenticator(security.ServerEditionAuthenticator())

class IPOfficeModeResource(object):
    """
    REST resource used to handle IP Office mode.

    Methods:

    GET     -- get current mode
    PUT     -- set new mode (this method should be called only once)
    """
    def GET(self):
        """
        Return IP Office mode packed in JSON format:

            { 'mode':  'Primary'|'Secondary'|'Expansion'|'Appl' }

        """
        web.header("Content-Type", "application/json")
        return json.dumps({'mode': ipoffice.get_mode()})

    @synchronized(ipoffice_lock)
    def PUT(self):
        """
        Sets IP Office and for:

        * Primary   -- configure for Auto-Start and start IP Office, Voicemail and OneX
        * Secondary -- configure for Auto-Start and start IP Office and Voicemail, uninstall OneX
        * Expansion -- configure for Auto-Start and start IP Office, uninstall Voicemail and OneX
        * Appl      -- uninstall Media Server and rename the release file from ABE_release to APPL_release

        For Secondary, Expansion and Appl modes this method will remove
        all the upgrade packages available under UWS repository.

        Warning: This method must be called only once.

        Input:

            mode    -- the mode to set

        Error codes:

            400     -- invalid mode
        """
        mode = web.input(mode=None).mode
        try:
            log.info(TAG, "setting IP Office mode to <%s>" % mode)
            ipoffice.set_mode(mode)
            ipoffice.set_phys_description(mode)
            apps_manager.uninstall_packages(mode)
            if mode == ipoffice.MODE_PRIMARY:
                apps_manager.install_onex_database()
                apps_manager.install_mode_package(mode)
            elif mode == ipoffice.MODE_SECONDARY:
                apps_manager.install_onex_database()
                apps_manager.install_mode_package(mode)
                ipoffice.cleanup_uws_repository()
            elif mode == ipoffice.MODE_EXPANSION:
                apps_manager.install_mode_package(mode)
                ipoffice.cleanup_uws_repository()
            elif mode == ipoffice.MODE_APPL:
                apps_manager.install_mode_package(mode)
                ipoffice.rename_release_file("abe","appl")
                import importlib
                importlib.reload(version)
                importlib.reload(manager)
                importlib.reload(updates_rest_api)
                ipoffice.cleanup_uws_repository()
            apps_manager.change_user_rights(mode)

            #-- in 9.0 upgrade progress should be available only from Web Manager --
            #if mode != ipoffice.MODE_APPL:
            #    log.info(TAG, "server mode set to <%s>: starting IP Office monitoring" % mode)
            #    local_ipoffice_monitor.start()
            #    upgrade_manager.start()

            # reload services database because some it is possible that
            # some services were removed, for example in case of Secondary
            # mode the OneX Portal is removed from system
            services_rest_api.services_manager.load_services()
        except ValueError as e:
            log.error(TAG, "invalid IP Office mode <%s>" % mode)
            web.ctx.status = '400'
            return str(e)
        except (IOError, OSError) as e:
            log.exception(TAG, "unable to set IP Office mode: i/o error")
            web.ctx.status = '400'
            return str(e)
        except Exception as e:
            log.exception(TAG, "unable to set IP Office mode: unexpected error")
            web.ctx.status = '400'
            return str(e)


class IPOfficeVirtualizationResource(object):
    """
    REST resource used to indicated if IP Office is in virtualization mode or not.

    Methods:

    GET     -- get virtualization mode
    """
    def GET(self):
        """
        Return value indicating if IP Office is virtualized:

            { 'virtualization':  true|false }
        """
        web.header("Content-Type", "application/json")
        return json.dumps({'virtualized': ipoffice.is_virtualized()})


class IPOfficeCompandingLawResource(object):
    """
    REST resource used to handle IP Office companding law setting.

    Methods:

    GET     -- get current companding law
    PUT     -- set new companding law
    """
    def GET(self):
        """
        Return IP Office companding law in JSON format:

            { 'companding_law': 'ALaw'|'MULaw'|'' }

        """
        web.header("Content-Type", "application/json")
        return json.dumps({'companding_law' : ipoffice.get_companding_law()})

    @synchronized(ipoffice_lock)
    def PUT(self):
        """
        Sets IP Office companding law.

        Input:

            companding      -- the companding law to set

        Error codes:

            400             -- invalid companding law
        """
        companding = web.input(companding=None).companding
        try:
            log.info(TAG, "setting IP Office companding law to <%s>" % companding)
            ipoffice.set_companding_law(companding)
        except ValueError as e:
            log.error(TAG, "invalid IP Office companding law <%s>" % companding)
            web.ctx.status = '400'
            return str(e)
        except (IOError, OSError) as e:
            log.exception(TAG, "unable to set IP Office companding law: i/o error")
            web.ctx.status = '400'
            return str(e)


class IPOfficeLicenceResource(object):
    """
    REST resource used to control the Licence feature Key
    when the IP Office server type is set to Primary.

    Methods:

    GET --  returns the IP office licence type
    SET --  sets the IP office licence type.
    """
    def GET(self):
        """
        Returns the IP office licence type.

        Data format:

        { 'licence_type': 'USB_Dongle' | 'Internal' }

        """
        licence_type = ipoffice.get_licence_type()
        web.header("Content-Type","application/json")
        return json.dumps({'licence_type' : licence_type})

    @synchronized(ipoffice_lock)
    def PUT(self):
        """
        Sets IP Office BE licence type.

        Input:

            licence      -- the licence type to set

        Error codes:

            400          -- invalid licence type
        """
        type = web.input(type=None).type
        try:
            log.info(TAG, "setting IP Office licencing to <%s>" % type)
            ipoffice.set_licence_type(type)
        except ValueError as e:
            log.error(TAG, "invalid IP Office licencing <%s>" % type)
            web.ctx.status = '400'
            return str(e)
        except (IOError, OSError) as e:
            log.exception(TAG, "unable to set IP Office licencing: i/o error")
            web.ctx.status = '400'
            return str(e)


class HostIdResource(object):
    """
    REST resource used to return the Host ID for the current installation.

    Methods:

    GET --  return the Host ID for current installation
    """
    def GET(self):
        """
        Returns the Host ID for current installation.
        """
        web.header("Content-Type", "application/json")
        return json.dumps({'sysinfo': ipoffice.host_id()})


class HardwareIdResource(object):
    """
    REST resource used to return the SID for the current installation.

    Methods:

    GET --  return the SID for current installation
    """
    def GET(self):
        """
        Returns the SID for current installation.
        """
        web.header("Content-Type","application/json")
        return json.dumps({'SID' : ipoffice.system_id()})


class HardwareProfilesResource(object):
    """
    REST resource to return required hardware information requests.

    Methods:

    GET -- returns required hardware information
    """

    # IP Office configuration
    CONFIGURATION_FILE  = "config/serveredition_hw.ini"

    # Recommended hardware profiles
    HARDWARE_PROFILES         = configobj.ConfigObj(CONFIGURATION_FILE)

    def GET(self):
        """
        Returns json string containing recommended hardware profiles.

        Data format:

        {
            "Primary": {
                "cpu_model": "Intel",
                "cpu_mhz": "2400",
                "disk_total": "500000000",
                "mem_total": "12000000",
                "port_ethernet": "2",
                "dvd" : "1",
                "port_usb" : "1" },
            "Secondary": {
                "cpu_model": "Intel",
                "cpu_mhz": "2400",
                "disk_total": "500000000",
                "mem_total": "4000000",
                "port_ethernet": "2",
                "dvd" : "1",
                "port_usb" : "1" },
            "Expansion": {
                "cpu_model": "Intel",
                "cpu_mhz": "2400",
                "disk_total": "250000000",
                "mem_total": "4000000",
                "port_ethernet": "2",
                "dvd" : "1",
                "port_usb" : "1" },
            "Appl": {
                "cpu_model": "Intel",
                "cpu_mhz": "2400",
                "disk_total": "500000000",
                "mem_total": "8000000" ,
                "port_ethernet": "2",
                "dvd" : "1",
                "port_usb" : "1" }
        }
        """
        web.header("Content-Type","application/json")
        return json.dumps(self.HARDWARE_PROFILES)


class OnexConfigResource(object):
    """
    REST resource to manage one-X Portal settings.

    Methods:

    GET     -- return JSON object with one-X Portal settings
    PUT     -- update one-X Portal settings
    """

    ONEX_CONFIG_FILE = '/opt/Avaya/scripts/config_files/onex_config'

    def GET(self):
        """
        Get one-X Portal url.

        Data format:

            { 'use_local': true/false,
              'onex_ip':  oneX machine IP}

        Error codes:

        400     --  Error occurred while reading one-X Portal settings
                    (one-X Portal configuration file couldn't be read).
        """
        try:
            onex_settings = {'onex_ip': ""}
            local_ip = network.eth0_ip()
            if os.path.isfile(self.ONEX_CONFIG_FILE):
                utils.read_config(self.ONEX_CONFIG_FILE, onex_settings)
                if onex_settings['onex_ip'] == "":
                    onex_settings['onex_ip'] = local_ip
            else:
                onex_settings['onex_ip'] = local_ip
            onex_settings['use_local'] = (onex_settings['onex_ip'] == local_ip)
            web.header("Content-Type","application/json")
            return json.dumps(onex_settings)
        except IOError:
            log.exception(TAG, 'unable to read one-X Portal configuration file')
            web.ctx.status = '400'

    def PUT(self):
        """
        Set the one-X Portal url.

        Input:

        onex_ip     -- The IP of the remote one-X Portal machine

        Error codes:

        400 -- I/O error when writing one-X Portal configuration
        """
        try:
            onex_settings = {}

            onex_ip = web.input(onex_ip=None).onex_ip
            if onex_ip:
                onex_settings['onex_ip'] = onex_ip
            else:
                onex_settings['onex_ip'] = network.eth0_ip()

            utils.update_config(self.ONEX_CONFIG_FILE, onex_settings)
            audit.log(vendor_i18n.custom_gettext("set one-X Portal address to <%s>" % onex_ip))
        except (IOError, OSError):
            log.exception(TAG, 'unable to write one-X Portal configuration file')
            web.ctx.status = '400'


class InstallerResource(object):
    """
    REST resource to return whether ABE installation was manual or scripted.

    Methods:

    GET -- returns ABE installation mode
    """

    # file which presence indicates a scripted install
    FACTORY_FLAG_FILE  = '/opt/Avaya/.factory'

    def GET(self):
        """
        Returns whether the factory flag is set.

        Data format:

        { 'factory_flag': true|false }
        """
        install_info = {'factory_flag': False}
        if os.path.exists(self.FACTORY_FLAG_FILE):
            install_info['factory_flag'] = True

        web.header('Content-Type', 'application/json')
        return json.dumps(install_info)


class Autologin(object):
    """Automatic login page handler"""

    def handle(self):
        referrer = web.input(referrer=None).referrer
        auth = web.input(auth=None).auth
        lang = web.input(lang='en_US').lang
        csrf_token = web.input(csrf_token=None).csrf_token
 
        if auth:
            try:
                username, password = b64decode(auth).decode().split(':')
            except:
                username, password = '', ''
        else:
            username = web.input(username='').username
            password = web.input(password='').password

        return render.autologin(username, password, referrer, lang)

#    @configuration.csrf_protected
    def POST(self):
        return self.handle()

    def GET(self):
        return self.handle()

class OperationStatusResource(object):
    """Handler for retrieving operation (backup, restore, upgrade) status"""

    def get_backup_restore_status(self, ws_client):
        xml_str = ws_client.get_backup_status()
        doc = XML(xml_str)
        tasks = doc.findall('.//task')
        if tasks:
            return tasks[0].text.lower()
        else:
            return ''

    def get_upgrade_status(self, ws_client):
        xml_str = ws_client.get_upgrade_status()
        doc = XML(xml_str)
        tasks = doc.findall('.//task')
        if tasks:
            return tasks[0].text.lower()
        else:
            return ''

    def get_status(self):
        operations = web.input(operations=None).operations
        if not operations:
            return None
        ops = operations.split(',')
        ops_status = dict((el, '') for el in ops)
        ws_client = ipoffice_ws.WSClient(username=web.ctx.session.user, password=web.ctx.session.password)
        for op in ops:
            if op == 'backup' or op == 'restore':
                ops_status[op] = self.get_backup_restore_status(ws_client)
            elif op == 'upgrade':
                ops_status[op] = self.get_upgrade_status(ws_client)
        return ops_status

    def GET(self):
        web.header('Content-Type', 'application/json')
        return json.dumps(self.get_status())
