# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Global configuration
"""

import configobj
import unittest
import glob
import os
from . import log as log 
import getpass
import web
import re
__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

# app configuration file
SHARED_CONFIGURATION_FILE = "config/main.ini"

# global configuration data
SHARED_CONFIGURATION = configobj.ConfigObj(SHARED_CONFIGURATION_FILE)
SHARED_CONFIGURATION['_global_'] = {}

SHARED_CONFIGURATION['applications'] = {}
MAX_USER_SESSION = 5

DATA_IP_VALIDATOR = "^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$"
DATA_PORT_VALIDATOR = "^([0-9]|[1-9][0-9]|[1-9][0-9][0-9]|[1-9][0-9][0-9][0-9]|[1-6][0-9][0-9][0-9][0-9]|)$"
DATA_FILE_VALIDATOR = "^.*(.p7b|.rpm|.pem|.crt|.pkx|.tar.gz|.tar|.tgz|.zip)$"

DATALENGTH_IP_VALIDATOR = 15
DATALENGTH_URL_VALIDATOR = 2083
DATALENGTH_PORT_VALIDATOR = 5
DATALENGTH_HOSTNAME_VALIDATOR = 255

def data_validate(field,validator):
    if not re.match(validator,field):
        raise web.HTTPError(
                "400 Bad request",
                {'content-type':'text/html'},
                'Validation failed for field')
        return False
    return True

def data_length_validate(field,length):
    if len(str(field)) > length :
        raise web.HTTPError(
                "400 Bad request",
                {'content-type':'text/html'},
                'Validation failed for field')
        return False
    return True

def data_validate_type_length(field,validator,length):
    return data_validate(field,validator) and data_length_validate(field,length)

def add_session(user):
    count = 0
    for k, (atime, value) in web.ctx.session.store.mem_store.items():
        if value.get('user') == user:
            count = count + 1
    if count + 1 > MAX_USER_SESSION:
        log.SHARED_LOGGER.info("Maximum number of sessions for user |%s| was reached" % user)
        return False
    else:
        log.SHARED_LOGGER.info("Current number of sessions for user |%s| is %s" % (user,count + 1))
    return True

_AUTHENTICATOR = None

def set_authenticator(authenticator):
    global _AUTHENTICATOR
    _AUTHENTICATOR = authenticator

def get_authenticator():
    return _AUTHENTICATOR

def load_services(path):
    """
    Walk directory tree rooted at <path> and build managed services database.

    This function works by processing ini files found under <path> and
    sub dirs loading from each file the details about a managed service.
    """
    log.SHARED_LOGGER.info("loading managed services from <%s> directory" % path)
    for current_file in glob.glob( os.path.join(path, '*') ):
        if os.path.isdir(current_file):
            load_services(current_file)
        else:
            if os.path.splitext(current_file)[1] == ".ini":
                service_config = configobj.ConfigObj(current_file)
                log.SHARED_LOGGER.info("loading service details from <%s>" % current_file)
                try:
                    app_id = service_config['id']
                    SHARED_CONFIGURATION['applications'][app_id] = service_config
                    log.SHARED_LOGGER.debug('loaded service <%s> from <%s>' % (app_id, current_file))
                except KeyError:
                    log.SHARED_LOGGER.error("service ID is missing from <%s>: service NOT loaded" % current_file)

def set_global_config(ckey, cvalue):
    if ckey:
        SHARED_CONFIGURATION['_global_'][ckey] = cvalue

def get_global_config(ckey):
    if ckey and ckey in SHARED_CONFIGURATION['_global_']:
        return SHARED_CONFIGURATION['_global_'][ckey]
    else:
        return None

def csrf_token():
    """Should be called from the form page's template:
       <form method=post action="">
         <input type=hidden name=csrf_token value="$csrf_token()">
         ...
       </form>"""
    if 'csrf_token' not in web.ctx.session:
        from uuid import uuid4
        web.ctx.session.csrf_token=uuid4().hex
    log.SHARED_LOGGER.info("Created csrf_token %s" % web.ctx.session.csrf_token)
    return web.ctx.session.csrf_token

def csrf_protected(f):
    """Usage:
       @csrf_protected
       def POST(self):
           ..."""
    def decorated(*args,**kwargs):
        inp = web.input()
        if not ('csrf_token' in inp and inp.csrf_token == web.ctx.session.csrf_token):
            log.SHARED_LOGGER.info("Issue %s %s" % (inp.csrf_token, web.ctx.session.csrf_token))
            raise web.HTTPError(
                "400 Bad request",
                {'content-type':'text/html'},
                'Cross-site request forgery (CSRF) attempt (or stale browser form). <a href="">Back to the form</a>.')
        log.SHARED_LOGGER.info("Form executed, decorated %s" % inp.csrf_token)
        return f(*args,**kwargs)
     
    return decorated

log.SHARED_LOGGER.info("loading configuration...")

load_services(SHARED_CONFIGURATION['services']['config_directory'])

# in seconds
DEFAULT_SESSION_TIMEOUT = 600

# The user name under which Web Control runs
USERNAME = getpass.getuser()

WEBCONTROL_LOG_DIR = os.getenv("WEBCONTROL_LOG_DIR", ".")

# Setup default language for Web Control based on LANG settings
# for OS and the list of supported languages
LOCALE_DIR = 'i18n/'
SUPPORTED_LANGUAGES = set()
for name in os.listdir(LOCALE_DIR):
    full_path = os.path.join(LOCALE_DIR, name)
    if os.path.isdir(full_path) and not full_path.startswith('.') and not 'vendor_' in full_path:
        SUPPORTED_LANGUAGES.add(name)

log.SHARED_LOGGER.info("supported languages: %s" % SUPPORTED_LANGUAGES)

DEFAULT_LANGUAGE = 'en_US'

DEFAULT_SESSION_LANGUAGE = os.getenv('OS_LANG', 'en_US')
if '.' in DEFAULT_SESSION_LANGUAGE:
    dot_index = DEFAULT_SESSION_LANGUAGE.index('.')
    DEFAULT_SESSION_LANGUAGE = DEFAULT_SESSION_LANGUAGE[0:dot_index]

if not DEFAULT_SESSION_LANGUAGE in SUPPORTED_LANGUAGES:
    log.SHARED_LOGGER.warn('OS language <%s> is not one of supported languages: %s, failing back to %s' %
                           (DEFAULT_SESSION_LANGUAGE, SUPPORTED_LANGUAGES, DEFAULT_LANGUAGE))
    DEFAULT_SESSION_LANGUAGE = DEFAULT_LANGUAGE

log.SHARED_LOGGER.info("default language is set to <%s>" % DEFAULT_SESSION_LANGUAGE)


class Test(unittest.TestCase):

    def test_global_config(self):
        # test if the top level configuration sections are present
        required_sections = ['services',
                             'webapp',
                             'repositories',
                             'syslog',
                             'network',
                             'disk',
                             'repositories',
                             'logrotate',
                             'snmp',
                             'backup',
                             'archives',
                             'logging',
                             'debug']
        for section in required_sections:
            self.assertTrue(SHARED_CONFIGURATION[section])

if __name__ == '__main__':
    unittest.main()
