# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Common utilities.
"""

import os
import re
import time
import unittest
import datetime
import web
import json
from core.system import shell
from core.common import i18n
from core.common import configuration

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

# common date format
DATE_FORMAT = "%Y-%m-%d %H:%M:%S"

# date formats for localization
L18_DATE_FORMATS = {'zh_CN': '%Y.%m.%d %H:%M:%S',
                    'de_DE': '%d.%m.%Y %H:%M:%S',
                    'ru_RU': '%d.%m.%Y %H:%M:%S',
                    'en_US': '%Y-%m-%d %H:%M:%S'}

# default date format used for localization
L18_DEFAULT_DATE_FORMAT = '%d-%m-%Y %H:%M:%S'

# default chunk size for file buffering
FILE_UPLOAD_BUFFER_SIZE = 10000


def encode_json(obj):
    """
    Helper function used by json module to obtain a serializable version for obj.
    """
    #print(obj)
    if hasattr(obj, '__dict__') :
        return obj.__dict__
    else:
        return obj.decode()
    return obj.__dict__

#    def _try(o): 
#        try: return o.__dict__ 
#        except: return o.decode()

#    def to_JSON(obj): 
#        return json.loads(json.dumps(obj, default=lambda o: _try(o), sort_keys=True, indent=0, separators=(',',':')).replace('\n', ''))
#    return to_JSON(obj)


def update_config(config_file, config_params):
    """
    Utility function for updating standard configuration files.
    
    Input:
    
    config_file   -- config file path
    config_params -- a dictionary containing the parameters that should be 
                     changed (key = parameter name, value = new parameter value)
    """
    with shell.file_open(config_file) as f:
        lines = f.readlines()
    
    params = config_params.copy()
    for i, line in enumerate(lines):
        if line.lstrip().startswith('#'):
            continue
        for param_name in params:
            if param_name in line:
                parts = line.split('=')
                if len(parts) == 1:
                    # line format: export PARAMETER_NAME
                    lines[i] = '%s=%s\n' % (parts[0].strip(), str(params[param_name]))
                elif len(parts) == 2:
                    # line format: export PARAMETER_NAME=PARAMETER_VALUE
                    parts[-1] = str(params[param_name])
                    lines[i] = '='.join(parts) + '\n'
                # remove current param from params map
                # to avoid searching for this parameter again
                # this means that only first occurrence of a parameter in a file is changed
                del params[param_name]
                # line processed
                break
        if not params:
            # all parameters were set
            # there is no point in checking the rest of file, it will be written as it is
            break
    
    with shell.file_open(config_file, 'w') as f:
        f.writelines(lines)


def read_config(config_file, config_params):
    """
    Utility function for retrieving parameter values from standard configuration files.
    
    Input:
    
    config_file   -- config file path
    config_params -- a dictionary containing the parameters that should be 
                     read (key = parameter name, value = will be filled with
                     the value read from the config file)
    """
    with shell.file_open(config_file) as f:
        lines = f.readlines()
    
    params = config_params.copy()
    for i, line in enumerate(lines):
        if line.lstrip().startswith('#'):
            continue
        for param_name in params:
            if param_name in line:
                parts = line.split('=')
                if len(parts) == 1:
                    # line format: export PARAMETER_NAME
                    config_params[param_name] = ''
                elif len(parts) == 2:
                    # line format is: export PARAMETER_NAME=PARAMETER_VALUE
                    config_params[param_name] = parts[-1].strip()
                # remove current param from params map
                # to avoid searching for this parameter again
                # this means that only first occurrence of a parameter in a file is read
                del params[param_name]
                # line processed
                break
        if not params:
            # all parameters were read
            # return now because there is no point in reading the rest of the file
            break


def format_bytes(n):
    """
    Utility function to transform in human readable input bytes value
    """
    K, M, G, T = 1 << 10, 1 << 20, 1 << 30, 1 << 40
    if   n >= T:
        return format_float(float(n) / T, 1, i18n.custom_gettext("T"))
    elif n >= G:
        return format_float(float(n) / G, 1, i18n.custom_gettext("G"))
    elif n >= M:
        return format_float(float(n) / M, 1, i18n.custom_gettext("M"))
    elif n >= K:
        return format_float(float(n) / K, 1, i18n.custom_gettext("K"))
    else:
        return '%d%s' % (n, i18n.custom_gettext("b"))


def format_hz(n):
    """
    Utility function to transform in human readable input bytes value
    """
    K, M, G, T = 1 << 10, 1 << 20, 1 << 30, 1 << 40
    if   n >= T:
        return format_float(float(n) / T, 1, i18n.custom_gettext("THz"))
    elif n >= G:
        return format_float(float(n) / G, 1, i18n.custom_gettext("GHz"))
    elif n >= M:
        return format_float(float(n) / M, 1, i18n.custom_gettext("MHz"))
    elif n >= K:
        return format_float(float(n) / K, 1, i18n.custom_gettext("KHz"))
    else:
        return '%d%s' % (n, i18n.custom_gettext("Hz"))


def format_float(value, decimals_number, unit):
    """
    Utility function that is used to localize a float bytes value
    """
    if "." in str(value):
        i, d = str(value).split(".")
        separator = i18n.custom_gettext(".")
        if int(d[0:decimals_number]) is 0:
            return i + unit
        return i + separator + d[0:decimals_number] + unit
    return value + unit


def format_timestamp(timestamp=None, format=None):
    """
    Convert UNIX timestamp into date string

    Input:

        timestamp     -- the timestamp that needs conversion
        string_format -- the format of the string (defaults to the locale's format)
    """
    if format is None:
        format = date_format()
    return time.strftime(format, time.localtime(timestamp))


def string_to_timestamp(string=None, format=None):
    """
    Converts a date string to UNIX timestamp

    Input:

        date_string   -- the string that needs timestamp conversion
        string_format -- the format of the string (defaults to the locale's format)
    """
    if format is None:
        format = date_format()
    return time.mktime(time.strptime(string , format))


def format_date_string(date_string=None, string_format=DATE_FORMAT):
    """
    Changes the format of a date string

    Input:

        date_string   -- the string that needs format change
        string_format -- the initial format of the string
    """
    if not date_string:
        return date_string
    return time.strftime(date_format(), time.strptime(date_string , string_format))


def date_format():
    """
    Returns the date format required by Web Control UI localization.

    This function will use the web context to detect session language.
    Note that if there is no web context available this function
    returns the default date format used by Web Control defined
    by DATE_FORMAT attribute.
    """
    if hasattr(web.ctx, "session"):
        lang = web.ctx.session.lang
        if not lang in configuration.SUPPORTED_LANGUAGES:
            lang = configuration.DEFAULT_SESSION_LANGUAGE
        return L18_DATE_FORMATS.get(lang, L18_DEFAULT_DATE_FORMAT)
    return DATE_FORMAT


def normalize_path(path):
    """
    Transform input path in UNIX format
    """
    return re.sub('\\\\', '/', path)


def parse_float(s):
    """
    Parse float from string.

    Return 0 if the string doesn't contain a float value.
    """
    match = re.search('[+-]? *(?:\d+(?:\.\d*)?|\.\d+)(?:[eE][+-]?\d+)?', s)
    if match:
        return float(match.group(0))
    else:
        return 0


def parse_int(s):
    """
    Parse int from string.

    Return 0 if the string doesn't contain a int value.
    """
    match = re.search('-?\d+', s)
    if match:
        return int(match.group(0))
    else:
        return 0


def parse_uptime(s):
    """
    Parse process uptime value in [[dd-][hh]:]mm:ss format
    (as returned by ps etime option) into a datetime.timedelta object.
    """
    tokens = [0, 0, 0, 0] # (days, hours, minutes, seconds)

    match = re.search('(\d+-)?(\d{2}:)?(\d{2}):(\d{2})', s)
    if match:
        for i in range(0, len(tokens)):
            parsed_val = match.group(i+1)
            if parsed_val:
                tokens[i] = parse_int(parsed_val)

    return datetime.timedelta(days=tokens[0], hours=tokens[1], minutes=tokens[2], seconds=tokens[3])


def centos_release ():
    """
    Returns the CentOS release number
    """
    output = shell.sudo_execute("cat /etc/redhat-release")[0]
    for i in output:
        if "." in i:
            return i
    return None

def interface_ip(name="eth0"):
    """
    Used to get the ip address of an interface
    """
    ip = None
    out = shell.sudo_execute("ifconfig %s" % name)
    if out:
        ip = out[1][1]
    return ip


def has_content(line):
    """
    Eliminates empty spaces from a string and returns True if the remaining string has content
    """
    return len(line.strip("\t\n\r")) > 0


def copy_file(file_in, file_out):
    """
    I/O utility used to copy file_in contents to file_out.
    This function uses a buffered copy approach using a file chunks generator.
    This function is designed to be used in the REST resources to upload local files.
    """
    with open(file_out, 'wb', FILE_UPLOAD_BUFFER_SIZE) as f:
        for chunk in file_buffer(file_in):
            f.write(chunk)


def file_buffer(f, chunk_size=FILE_UPLOAD_BUFFER_SIZE):
    """
    Generator of buffer file chunks.
    """
    while True:
        chunk = f.read(chunk_size)
        if not chunk: break
        yield chunk


def deflate_archive(archive_file):
    """
    Deflate archive if archive_file exists and return True or False
    whether the deflate operation was executed or not.
    """
    archive_file_lower = archive_file.lower()
    archive_dir  = os.path.dirname(archive_file)
    was_unpacked = False
    if archive_file_lower.endswith('.tar.gz') or \
       archive_file_lower.endswith('.tar') or \
       archive_file_lower.endswith('.tgz'):
        shell.call('tar -xvf %s -c %s' % (archive_file, archive_dir))
        was_unpacked = True
    elif archive_file_lower.endswith('.zip'):
        shell.call('unzip %s -d %s' % (archive_file, archive_dir))
        was_unpacked = True
    return was_unpacked


class OpenStruct(object):
    """
    Open struct class.

    This class allows you to create data objects and set arbitrary attributes:

    person = OpenStruct()
    person.name = "John Smith"
    person.age = 30

    print person.name => John Smith
    """
    def __init__(self, **kwargs):
        self.__dict__.update(kwargs)
        
    def __getattr__(self, k):
        if k in self.__dict__:
            return self.__dict__[k]
        else:
            raise AttributeError( k)
        
    def __setattr__(self,k,v):
        if k in self.__dict__:
            self.__dict__[k] = v
        else:
            self.__dict__.update({k:v})
        return v


class Test(unittest.TestCase):

    def test_set_attr(self):
        obj = OpenStruct()
        obj.test = 1        
        self.assertTrue(obj.test)

    def test_not_set_attr(self):
        obj = OpenStruct()
        obj.a = 1 
        self.assertFalse(hasattr(obj, "test"))

    def test_encode_json(self):
        obj = OpenStruct()
        obj.test = "something"
        self.assertTrue("test" in encode_json(obj))        

    def test_read_config(self):
        config_file = '/etc/sysconfig/watchdog'
        snmp_settings = {'WDOG_SNMP_PROTOCOL': None, 
                         'WDOG_SNMP_ADDRESS_SEND': None,
                         'WDOG_SNMP_PORT': None}
        try:
            read_config(config_file, snmp_settings)
        except (IOError, OSError):
            self.fail('%s could not be read' % config_file)

    def test_read_config_2(self):
        import os
        config_file = '/etc/sysconfig/ipoffice'
        abe_release = '/opt/Avaya/ABE_release'
        ipoffice_be_settings = {'IPOFFICE_BE_MODE': None}
        if os.path.isfile(config_file) and \
           os.path.isfile(abe_release):
            read_config(config_file, ipoffice_be_settings)
            self.assertTrue(ipoffice_be_settings['IPOFFICE_BE_MODE'] is not None)

    def test_format_timestamp(self):
        self.assertTrue(format_timestamp())

    def test_format_bytes(self):
        s1 = format_bytes(1024)
        s2 = format_bytes(512)
        self.assertTrue(s1.endswith('K'))
        self.assertTrue(s2.endswith('b'))

    def test_parse_float(self):
        self.assertAlmostEqual(0.1, parse_float(' abac 0.1 %'))
        self.assertAlmostEqual(-0.1, parse_float('b-0.1 $'))
        self.assertAlmostEqual(120.0, parse_float("120"))
        self.assertAlmostEqual(123.4, parse_float(" 123.4afda"))
        self.assertEqual(0, parse_float('a-bc fdfa %^%$.'))

    def test_parse_int(self):
        self.assertEqual(1, parse_int('afda 1ffda%-'))
        self.assertEqual(-1, parse_int('% -1 %fda '))
        self.assertEqual(0, parse_int('a-b .$A,'))
        self.assertEqual(6222, parse_int('006222'))

    def test_parse_uptime(self):
        td = parse_uptime('1-02:03:04')
        hours, remainder = divmod(td.seconds, 3600)
        minutes, seconds = divmod(remainder, 60)
        self.assertEqual(1, td.days)
        self.assertEqual(2, hours)
        self.assertEqual(3, minutes)
        self.assertEqual(4, seconds)

        td = parse_uptime('02:03:04')
        hours, remainder = divmod(td.seconds, 3600)
        minutes, seconds = divmod(remainder, 60)
        self.assertEqual(0, td.days)
        self.assertEqual(2, hours)
        self.assertEqual(3, minutes)
        self.assertEqual(4, seconds)

        td = parse_uptime('03:04')
        hours, remainder = divmod(td.seconds, 3600)
        minutes, seconds = divmod(remainder, 60)
        self.assertEqual(0, td.days)
        self.assertEqual(0, hours)
        self.assertEqual(3, minutes)
        self.assertEqual(4, seconds)

    def test_has_content(self):
        self.assertTrue(has_content("fsgdshgsd\n\tgdsgfsd fsgdshgs"))
        self.assertFalse(has_content("\n\t\n\n\t\r\n"))

    def test_centos_release(self):
        self.assertTrue(centos_release())

    def test_format_date_string_empty_or_None(self):
        self.assertEqual(None, format_date_string(None))
        self.assertEqual('', format_date_string(''))

    def test_format_date_string(self):
        import web
        web.ctx.session = web.Storage()

        date_str = '22/02/12 10:27:14'
        date_str_format = '%d/%m/%y %H:%M:%S'

        test_data = (('zh_CN', '2012.02.22 10:27:14'), # (lang, expected)
                     ('de_DE', '22.02.2012 10:27:14'))
        for lang, expected in test_data:
            web.ctx.session.lang = lang
            self.assertEqual(expected, format_date_string(date_str, date_str_format))


if __name__ == "__main__":
    unittest.main()
