# Copyright 2011 Avaya Inc. All Rights Reserved.

"""
Provides I18N support for 3rd party vendor plugins.

See http://webpy.org/cookbook/runtime-language-switch
"""

import web
import gettext

LOCALE_DIR = 'i18n/vendor_i18n/'
ALL_TRANSLATIONS = web.storage()


def _get_translations(lang='en_US'):
    # init translations
    if lang in ALL_TRANSLATIONS.keys():
        translation = ALL_TRANSLATIONS[lang]
    elif lang is None:
        translation = gettext.NullTranslations()
    else:
        try:
            translation = gettext.translation(
                    'messages',
                    LOCALE_DIR,
                    languages=[lang],
                    )
        except IOError:
            translation = gettext.NullTranslations()
    return translation


def _load_translations(lang):
    """Return the translations for the locale."""
    lang = str(lang)
    translation  = ALL_TRANSLATIONS.get(lang)
    if translation is None:
        translation = _get_translations(lang)
        ALL_TRANSLATIONS[lang] = translation
        # delete unused translations
        for lk in list(ALL_TRANSLATIONS):
            if lk != lang:
                del ALL_TRANSLATIONS[lk]
    return translation


def custom_gettext(string):
    """
    Translate a given string to the language set for current session.
    Session variable that used to specify language for current session is ``lang``.
    """
    translation = _load_translations(web.ctx.session.get('lang'))
    if translation is None:
        return unicode(string)
    return translation.gettext(string)


def translate(string, lang):
    """
    Translate a given string to the language set for specified language.
    """
    translation = _load_translations(lang)
    if translation is None:
        return unicode(string)
    return translation.gettext(string)
