# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Management of log archives
"""

import os
import stat
import glob
import unittest
import web

from core.common import utils
from core.system import shell
from core.common import configuration

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

class LogArchivesManager(object):
    """
    Create and delete archives with applications logs and debug files.
    """

    def __init__(self, config=configuration.SHARED_CONFIGURATION):
        """
        config -- configuration data
        """
        config_file = config['logging']['archive_location_script']
        self.archives = {}
        for archives_type in config['archives']:
            archive = utils.OpenStruct()
            params = {config['archives'][archives_type]['location_variable']: None}
            utils.read_config(config_file, params)
            archive.dir = params[config['archives'][archives_type]['location_variable']].replace("'", "")
            archive.create_script = config['archives'][archives_type]['archive_script']
            archive.delete_script = config['archives'][archives_type]['delete_script']
            self.archives[archives_type] = archive

    def get_debug_files_info(self):
        """
        Used to detect if there are core dump files available for archiving.

        Return a dictionary:

            {
                'count':   number of core dump files
                'size':    total size for core dump files in human readable format
            }
        """
        total_files = 0
        total_size = 0
        for afile in glob.glob(os.path.join(self.archives['debug'].dir, 'core*')):
            fstats = os.stat(afile)
            total_files += 1
            total_size += fstats[stat.ST_SIZE]
        return {'count': total_files, 'size': utils.format_bytes(total_size)}

    def get_archives(self, archive_type):
        """
        Return log archives, sorted by last modified (most recent first)
        
        archive_type -- archive type (logs or debug)
        """
        archives = []
        for afile in glob.glob(os.path.join(self.archives[archive_type].dir, '*.tar.gz')):
            fstats = os.stat(afile)
            if hasattr(web.ctx.session, "security_rights"):
                if web.ctx.session.security_rights is False and "SEC" in os.path.basename(afile):
                    continue
            finfo = {'name':  os.path.basename(afile),
                     'size':  utils.format_bytes(fstats[stat.ST_SIZE]),
                     'modif': utils.format_timestamp(fstats[stat.ST_MTIME]),
                     'profile': self.get_profile(afile, self.archives[archive_type].dir)}
            archives.append(finfo)
        
        sorted_archives = sorted(archives, key=lambda k: k['modif'], reverse=True)
        return sorted_archives

    def get_profile(self, archive_file, path):
        """
        Return the profile text file associated to an archive

        archive_file -- the .tar.gz archive file
        path         -- folder containing the profile files
        """
        pid = archive_file.split(".")[2]
        for afile in glob.glob(os.path.join(path, '*.txt')):
            if pid in afile:
                return os.path.basename(afile)
        return ""

    def delete_files(self, archive_type, files):
        """
        Delete specified files.

        Args:

        archive_type    -- logs or debug
        files           -- a list of files to delete or None for all files
        """
        if files:
            for afile in files:
                fpath = os.path.join(self.archives[archive_type].dir, afile)
                shell.sudo_call('rm -f %s' % str(fpath))
        else:
            delete_script = self.archives[archive_type].delete_script
            shell.sudo_call(delete_script)

    def create_archive(self, archive_type):
        """
        Create archives for specified type.

        Args:

        archive_type -- logs or debug
        """
        archive_script = self.archives[archive_type].create_script
        shell.sudo_call(archive_script)

class Test(unittest.TestCase):

    def setUp(self):
        self.log_archives = LogArchivesManager()

    def test_logs_archives(self):
        self.log_archives.create_archive('logs')
        self.assertTrue(self.log_archives.get_archives('logs'))

    def test_get_debug_files_info(self):
        info = self.log_archives.get_debug_files_info()
        self.assertTrue(info)
        if info['count'] > 0:
            self.assertTrue(info['size'])

    def test_debug_archives(self):
        info = self.log_archives.get_debug_files_info()
        if info['count'] > 0:
            self.log_archives.create_archive('debug')
            self.assertTrue(self.log_archives.get_archives('debug'))

if __name__ == '__main__':
    unittest.main()
