# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Audit logger

The audit log files are stored under path defined by WEBCONTROL_LOG_DIR environment variable.
If WEBCONTROL_LOG_DIR is not set then current working directory is used.
"""

import os
import glob
import logging
import unittest

from core.common import utils 
from core.common import configuration
import core.common.log

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

TAG = "audit"

class LogEntry(object):
    """
    Audit log entry
    """
    def __init__(self, timestamp=None, user=None, action=None):
        self.timestamp = timestamp
        self.user = user
        self.action = action

class AuditLog(object):
    """
    Audit logger
    """
    def __init__(self, config=configuration.SHARED_CONFIGURATION):
        """
        config -- app configuration data
        """
        self.filename = os.path.join(os.getenv('WEBCONTROL_LOG_DIR', '.'),
                                         config['webapp']['audit_file'])
        self.file_logger = logging.getLogger('Audit')
        self.file_logger.setLevel(logging.INFO)
        handler = logging.FileHandler(self.filename)
        formatter = logging.Formatter("%(asctime)s - %(message)s", utils.DATE_FORMAT)
        handler.setFormatter(formatter)
        self.file_logger.addHandler(handler)

    def log(self, user, action):
        """
        Add entry to audit log

        user    -- username
        action  -- message describing user action
        """
        self.file_logger.info("%s : %s" % (user, action))
        core.common.log.syslog( ("%s|%s" % (user, action) ))

    def log_history(self):
        """
        Return audit log entries
        """
        entries = []
        log_files = glob.glob('%s*' % self.filename)
        for fname in log_files:
            try:
                entries.extend(self._parse(fname))
            except (IOError, OSError):
                core.common.log.exception(TAG, 'unable to parse audit log file %s' % fname)
                pass
        entries.sort(key=lambda entry: entry.timestamp)
        return entries

    def _parse(self, logfile):
        """
        Parse audit log file contents into LogEntries list
        """
        entries = []
        with open(logfile) as f:
            for line in f:
                tokens = line.strip().split(" - ")
                if len(tokens) > 1:
                    tokens2 = tokens[1].split(" : ")
                    if len(tokens2) > 1:
                        entry = LogEntry()
                        entry.timestamp = utils.format_date_string(tokens[0])
                        entry.user = tokens2[0]
                        entry.action = tokens2[1]
                        entries.append(entry)
        return entries
    

class Test(unittest.TestCase):

    def setUp(self):
        self.audit_log = AuditLog()

    def tearDown(self):
        logfiles = glob.glob('%s*' % self.audit_log.filename)
        for file in logfiles:
            os.remove(file)

    def test_audit_log(self):
        self.audit_log.log('me', 'test')
        history = self.audit_log.log_history()
        self.assertEqual(len(history), 1)
        entry = history[0]
        self.assertEqual(entry.user, 'me')
        self.assertEqual(entry.action, 'test')

if __name__ == '__main__':
    unittest.main()
