# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Service performance monitoring
"""

import threading
import time
import copy
import unittest

from core.common import utils
from core.common import configuration

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

class Sample(object):
    """
    Performance data sample
    """
    def __init__(self, timestamp=0, cpu_usage=0, mem_usage=0):
        self.timestamp = timestamp
        self.cpu_usage = cpu_usage
        self.mem_usage = mem_usage

class PerformanceData(object):
    """
    List of samples with performance data for a process
    """
    def __init__(self, max_samples):
        """
        max_samples -- maximum number of samples to hold
        """
        self.max_samples = max_samples
        self.samples = []

    def append(self, sample):
        """
        Append new sample. If the current capacity exceeds max_samples
        the oldest entry is discarded.
        """
        self.samples.append(sample)
        if len(self.samples) > self.max_samples:
            self.samples.pop(0)

class PerformanceMonitor(threading.Thread):
    """
    Performance data monitor for services
    """
    def __init__(self, manager, config=configuration.SHARED_CONFIGURATION):
        """
        manager -- services manager
        config  -- configuration data
        """
        self.manager = manager
        self.max_samples = config['services']['performance'].as_int('max_samples')
        self.poll_interval = config['services']['performance'].as_int('poll_interval')
        self.lock = threading.Lock()

        self.performance_data = {}
        for service in manager:
            self.performance_data[service.id] = PerformanceData(self.max_samples)

        threading.Thread.__init__(self)
        self.setDaemon(True)

    def _get_sample(self, service):
        service.update_run_info()
        sample = Sample(timestamp=utils.format_timestamp())
        if service.is_running():
            sample.cpu_usage = service.run_info.cpu
            sample.mem_usage = service.run_info.mem
        return sample

    def run(self):
        while True:
            self.lock.acquire()
            try:
                for service in self.manager:
                    sample = self._get_sample(service)
                    self.performance_data[service.id].append(sample)
            finally:
                self.lock.release()
            time.sleep(self.poll_interval)

    def get_performance_data(self, id=None):
        """
        id  -- service ID or None for all services
        """
        self.lock.acquire()
        try:
            if id:
                return copy.deepcopy(self.performance_data[id])
            else:
                return copy.deepcopy(self.performance_data)
        finally:
            self.lock.release()

class Test(unittest.TestCase):

    def setUp(self):
        import manager
        self.monitor = PerformanceMonitor(manager.ServicesManager())

    def test_get_performance_sample(self):
        service_id = 'watchdog'
        service = self.monitor.manager[service_id]
        service.update_run_info()
        if service.run_info.state == service.RUNNING:
            sample = self.monitor._get_sample(service)
            self.assertTrue(sample.mem_usage > 0)
        else:
            service.start()
            import time
            time.sleep(10)
            service.update_run_info()
            if service.run_info.state == service.RUNNING:
                sample = self.monitor._get_sample(service)
                self.assertTrue(sample.mem_usage > 0)
                service.stop()
            else:
                self.fail('unable to start %s service' % service_id)

if __name__ == '__main__':
    unittest.main()