# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Utility module for handling software repositories.
"""
import os
import re
import string
import configobj
import unittest

from core.common import configuration
from core.common import log
from core.system import shell
from core.system import rpm

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

TAG = 'repositories'

# Local repositories
LOCAL_USER_REPOSITORY     = configuration.SHARED_CONFIGURATION['repositories']['local']['user']
LOCAL_SYSTEM_REPOSITORY   = configuration.SHARED_CONFIGURATION['repositories']['local']['system']
LOCAL_APPS_REPOSITORY     = configuration.SHARED_CONFIGURATION['repositories']['local']['apps']
LOCAL_WINAPPS_REPOSITORY  = configuration.SHARED_CONFIGURATION['repositories']['local']['winapps']


def _get_repo_config(config, name):
    """
    Returns the configuration for a specific repository from a ConfigObj instance
    """
    url = config[name]['baseurl']
    is_local = url.lstrip().startswith('file://')
    return {'local': is_local, 
            'url': url,
            'default_url': configuration.SHARED_CONFIGURATION['repositories']['default'][name]}


def get_config(config_file, repo_name=None):
    """
    Retrieve repositories configuration. 
    
    If repo_name parameter is provided, only the configuration for the specified
    repository is returned.
    
    For each repository, indicated by its name, a dictionary of two values are returned:
    
    local --- True, if the updates repository is the local one, or False otherwise
    url   --- The URL of remote updates repository, or None if using local repository
    """
    config = configobj.ConfigObj(config_file)

    if repo_name:
        if repo_name in config:
            return {repo_name: _get_repo_config(config, repo_name)}
        else:
            return None
    repo_config = {}
    for repo in config:
        repo_config[repo] = _get_repo_config(config, repo)
    
    return repo_config


def set_config(config_file, repo_config):
    """
    Update repositories configuration. 
    """
    config = configobj.ConfigObj(config_file)
    
    for repo in repo_config:
        config[repo]['baseurl'] = repo_config[repo]['url']
    
    config.write()

    repomd_path = shell.find_file("repomd.xml", "/var/cache/yum/")
    # remove cached repository metadata file
    if repomd_path is not None:
        shell.sudo_call('rm -rf %s' % repomd_path)


def list_local_repository(repo, extension=''):
    """
    Returns a list of repository files, filtered by extension.
    Each item in the returned list is the file base name (not full path).
    """
    file_list = []

    for root, dirs, files in os.walk(repo):
        for file in files:
            if file.endswith(extension):
                file_list.append(file)

    return file_list


def list_unused_rpm_files():
    """
    Returns a list with rpm files from specified repository
    that are not installed on local system.
    """
    repo      = LOCAL_APPS_REPOSITORY
    file_list = []
    rpm_files = [os.path.join(repo, basename) for basename in
                 list_local_repository(repo, extension='.rpm')]

    for rpm_file in rpm_files:
        package_info = rpm.query_rpm(rpm_file)
        if package_info and 'NAME' in package_info:
            rpm_info = rpm.query_package(package_info['NAME'])
            if not rpm_info:
                file_list.append(rpm_file)
            else:
                if package_info['VERSION']!=rpm_info['VERSION'] or package_info['RELEASE']!=rpm_info['RELEASE']:
                    file_list.append(rpm_file)
    return file_list


def list_unused_windows_clients_files(repo, extension='.exe'):
    """
    Returns a list of latest versions for exe repository files.
    """
    file_list = []
    exe_files = list_local_repository(repo, extension)

    if not exe_files:
        return file_list

    exe_files.sort()

    previous_file_name  = ""
    previous_version    = ""
    previous_file       = ""

    for file in exe_files:
        file_to_parse       = file.replace("_",".").replace("-",".")
        current_version     = string.join(re.findall(r'\b\d+\b', file_to_parse),".")
        current_file_name   = re.split(r'\b\d+\b', file_to_parse)[0]

        if previous_file_name == current_file_name:
            if previous_version < current_version:
                file_list.append(shell.find(repo, previous_file)[0])

        previous_file_name = current_file_name
        previous_version = current_version
        previous_file = file

    return file_list


def list_unused_repodata_files():
    """
    Returns a list of repodata files.
    """
    repo            = LOCAL_SYSTEM_REPOSITORY +  'repodata'
    file_list       = []
    repodata_files  = list_local_repository(repo)

    for file in repodata_files:
        file_list.append(shell.find(repo, file)[0])

    return file_list


def list_unused_cached_files():
    """
    Returns a list of files to be deleted from local cache (local repositories).
    """
    files_to_delete = []

    files_to_delete.extend(list_unused_rpm_files())
    files_to_delete.extend(list_unused_windows_clients_files(LOCAL_WINAPPS_REPOSITORY))
    files_to_delete.extend(list_unused_windows_clients_files(LOCAL_USER_REPOSITORY))
    files_to_delete.extend(list_unused_repodata_files())

    log.info(TAG, 'unused cached files: %s' % '\n'.join(files_to_delete))

    return files_to_delete


class Test(unittest.TestCase):

    def test_get_repositories_config(self):
        config_file = '/etc/yum.repos.d/Avaya_Inside.repo'
        if os.path.isfile(config_file):
            # Avaya repo configuration exists
            repo_config = get_config(config_file)
            self.assertTrue(repo_config)

            repo_config = get_config(config_file, 'system')
            self.assertTrue(repo_config)

    def test_list_unused_rpm_files(self):
        self.assertTrue(list_unused_rpm_files() is not None)

    def test_list_unused_windows_clients_files(self):
        self.assertTrue(list_unused_windows_clients_files(LOCAL_WINAPPS_REPOSITORY) is not None)

    def test_list_repository_files(self):
        self.assertTrue(list_local_repository(LOCAL_SYSTEM_REPOSITORY +  'repodata') is not None)


if __name__ == "__main__":
    unittest.main()