#Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Network configuration management
"""
import threading
import unittest
from . import shell
import os
import time
import socketserver

from core.common import i18n
from core.common import configuration
from core.common import log
from socket import inet_ntoa
from struct import pack
from configobj import ConfigObj

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

# delay in seconds between interface down and up operations
INTERFACE_RESTART_DELAY = 5

# network interface management scripts
GET_INTERFACE_SCRIPT        = configuration.SHARED_CONFIGURATION['network']['get_interface_script']
SET_DHCP_INTERFACE_SCRIPT   = configuration.SHARED_CONFIGURATION['network']['set_dhcp_interface_script']
SET_STATIC_INTERFACE_SCRIPT = configuration.SHARED_CONFIGURATION['network']['set_static_interface_script']

# global network configuration files
DNS_CONFIG_FILE         = '/etc/resolv.conf'
NETWORK_CONFIG_FILE     = '/etc/sysconfig/network'
HOSTS_CONFIG_FILE       = '/etc/hosts'
NETWORK_SCRIPTS_DIR     = '/etc/sysconfig/network-scripts/'

# eth0 IP address will also be written in this file
VMPRO_SETTINGS_FILE     = '/etc/vmpro_settings.ini'
# used to manage IP Office LAN settings
IPOFFICE_CONFIG_FILE    = '/etc/sysconfig/ipoffice'

# used to determine if a duplicate IP is in use
DUPLICATE_IP = False

# IP Office  and Linux well known ports table
KNOWN_PORTS =    {22:    'SSH',
                  25:    'SMTP',
                  37:    'TIME',
                  53:    'DNS',
                  67:    'DHCP Server / BOOTP Server',
                  68:    'DHCP Client / BOOTP Client',
                  69:    'TFTP',
                  80:    'HTTP',
                  123:   'NTP',
                  143:   'IP Office / VoicemailPro IMAP Service',
                  161:   'SNMP Agent',
                  162:   'SNMP Trap',
                  389:   'LDAP',
                  443:   'HTTPS',
                  500:   'IKE',
                  514:   'Syslog Client',
                  520:   'RIP',
                  833:   'Network Relay',
                  1433:  'MSSQL',
                  1434:  'MSSQL',
                  1701:  'L2TP',
                  1718:  'H.323 discovery',
                  1719:  'H.323 status',
                  1720:  'H.323 Signalling',
                  4560:  'Log4j',
                  5222:  'XMPP',
                  5269:  'XMPP',
                  5060:  'SIP',
                  5061:  'SIP',
                  7443:  'HTTPS',
                  8000:  'HTTP',
                  8005:  'Tomcat Shutdown',
                  8080:  'HTTP',
                  8135:  'Customer Call Reporter - Wallboard',
                  8443:  'HTTPS - Web client access',
                  8444:  'Mobile Client Authentication',
                  8666:  'JMX - Java Management Extensions',
                  9092:  'JDBC',
                  9094:  'XML RPC',
                  9095:  'OpenFire admin (HTTPS)',
                  9443:  'HTTPS - Web client access',
                  10334: 'acmsComms',
                  42000: 'aSoftMS',
                  50791: 'Voicemail',
                  50792: 'IP Office  VoicemailPro',
                  50793: 'Solomail',
                  50794: 'Monitor',
                  50795: 'Voicenet',
                  50796: 'PC Partner',
                  50797: 'TAPI',
                  50780: 'Dongle server',
                  50801: 'EConf',
                  50802: 'Who Is #2',
                  50804: 'Configuration service - unsecured',
                  50805: 'Configuration service secured TLS',
                  50808: 'SSI service - unsecured',
                  50812: 'Security Settings Service - unsecured',
                  50813: 'Security Settings Service secured TLS',
                  50814: 'EnhTSPI service - unsecured'}


class ConfigFileError(Exception):
    """
    For listing IP Office LAN Configuration errors.
    """
    def __str__(self):
        return self.args[0]


class Interface(object):
    """
    Network interface.
    """
    def __init__(self,
                 name=None,
                 ipaddr=None,
                 netmask=None,
                 gateway=None,
                 use_dhcp=False,
                 auto_dns=True):
        """
        Args:

        name:       interface name
        ipaddr:     IP address
        netmask:    netmask value
        gateway:    gateway IP address
        use_dhcp:   DHCP enabled ?
        auto_dns:   peer DNS enabled ?
        """
        self.name = name
        self.ipaddr = ipaddr
        self.netmask = netmask
        self.gateway = gateway
        self.use_dhcp = use_dhcp
        self.auto_dns = auto_dns


def hostname():
    """
    Return hostname.
    """
    out = shell.execute('hostname')
    if out:
        return out[0][0]


def set_hostname(name):
    """
    Set hostname.
    Return exit code for hostname <name> command.
    """
    out = []
    with shell.file_open(NETWORK_CONFIG_FILE) as infile:
        for line in infile:
            if line.startswith('HOSTNAME'):
                out.append('HOSTNAME=%s\n' % name)
            else:
                out.append(line)
    with shell.file_open(NETWORK_CONFIG_FILE, 'w') as outfile:
        outfile.writelines(out)
    # get current ip address to se in hosts file
    ip_address=eth0_ip()
    if os.path.exists(configuration.SHARED_CONFIGURATION['release_config']['apc']):
        ip_address=interface('eth0.1').ipaddr
    lines = []
    found_ip=False
    with shell.file_open(HOSTS_CONFIG_FILE) as infile:
        for line in infile:
            if line.startswith(ip_address):
                lines.append("%s\t\t%s\n" % (ip_address,name))
                found_ip=True
            elif line.startswith('127.0.0.1'):
                lines.append("127.0.0.1\t\tlocalhost.localdomain localhost\n")
            else:
                lines.append(line)
    if not found_ip:
        lines.append("%s\t\t%s\n" % (ip_address,name))

    with shell.file_open(HOSTS_CONFIG_FILE, 'w') as outfile:
        outfile.writelines(lines)
    shell.sudo_call('hostname %s' % name)
    return shell.sudo_call('hostnamectl set-hostname %s' % name)


def apply_hostname(name):
    """
    Change the hostname and check if it's valid.
    If not, change it back to its previous value.
    """
    old_hostname = shell.sudo_execute("hostname -f", parse=shell.RAW).decode().strip()
    set_hostname(name)
    new_hostname = shell.sudo_execute("hostname -f", parse=shell.RAW).decode().strip()
    if not name == new_hostname:
        log.SHARED_LOGGER.error("Invalid hostname %s. Reverting to the original hostname" % name)
        set_hostname(old_hostname)


def interface(name):
    """
    Return network interface configuration

    name -- interface name (ex: eth0)
    """
    out = shell.sudo_execute("%s %s" % (GET_INTERFACE_SCRIPT, name), parse=shell.RAW)
    return _parse_interface(out)


def _parse_interface(s):
    """
    Parse interface object from the output of the GET_INTERFACE_SCRIPT.
    """
    interf = Interface()
    config = dict([map(_sanitize, kv.split('=')) for kv in s.strip().decode().split('\n')])
    interf.name     = config.get('name', '')
    interf.ipaddr   = config.get('ipaddr', '')
    interf.netmask  = config.get('netmask', '')
    interf.gateway  = config.get('gateway', '')
    interf.use_dhcp = config.get('use_dhcp', 'no') == 'yes'
    interf.auto_dns = config.get('peerdns', 'no') == 'yes'
    return interf


def _sanitize(s):
    return s.strip(' "').lower()


def set_interface(interf):
    """
    Save changes to network interfaces via script
    """
    if interf.use_dhcp:
        shell.sudo_call("%s %s %s" % (SET_DHCP_INTERFACE_SCRIPT,
                                      interf.name,
                                      interf.auto_dns))
    else:
        shell.sudo_call("%s %s %s %s %s %s" % (SET_STATIC_INTERFACE_SCRIPT,
                                               interf.name,
                                               interf.ipaddr,
                                               interf.netmask,
                                               interf.auto_dns,
                                               interf.gateway))


def get_interfaces():
    """
    Return available network interfaces.
    """
    interfaces = []
    output = shell.sudo_execute("ls %s" % NETWORK_SCRIPTS_DIR, parse=shell.LINES)
    for file in output:
        if os.path.isfile(os.path.join(NETWORK_SCRIPTS_DIR, file)) and "ifcfg-eth" in file:
            if not "." in file and file != "lo":
                interfaces.append(file.replace("ifcfg-", ""))
    return interfaces


def get_all_interfaces():
    """
    Return available interfaces and sub-interfaces
    """
    interfaces = []
    output = shell.sudo_execute("ls %s" % NETWORK_SCRIPTS_DIR, parse=shell.LINES)
    for file in output:
        if os.path.isfile(os.path.join(NETWORK_SCRIPTS_DIR, file)) and "ifcfg-eth" in file:
            if file != "lo":
                interfaces.append(file.replace("ifcfg-", ""))
    return interfaces


def restart_interface(interf):
    """
    Restart the network interface.
    Preserve previous IP address when DHCP Client fails.
    """
    shell.sudo_call('ifdown %s' % interf.name)
    time.sleep(INTERFACE_RESTART_DELAY)
    if shell.sudo_call('ifup %s' % interf.name):
        if interf.use_dhcp:
            shell.sudo_call('ifconfig %s inet %s netmask %s' %
                            (interf.name, interf.ipaddr, interf.netmask))


def ping_ip(ip, name="eth0"):
    try:
        output = shell.sudo_execute("arping -I %s -c 2 %s" % (name, ip), parse=shell.LINES)
        global DUPLICATE_IP
        DUPLICATE_IP = False
        for line in output:
            if "reply from" in line:
                shell.sudo_call('logger -t "IPOLError" "IPOffice was forced to restart 3 times due to faulty LAN '
                                'configuration. IPOffice will remain stopped. Please check for duplicate IP address '
                                'in current LAN."')
                DUPLICATE_IP = True
                break
    except: pass

def restart():
    """
    Restart network services.
    Return the exit code for 'service network restart' command.
    """
    return shell.sudo_call('service network restart')


def remove_from_vlans(name):
    """
    Remove a subinterface from the vlan folder and vlan mapping file
    """
    shell.sudo_call("vconfig rem %s" % name)


def eth0_ip():
    """
    Convenience function to get the ip address of the eth0 interface.
    """
    return interface('eth0').ipaddr


def is_port_in_use(port):
    """
    Returns True if a port is opened on local machine, False otherwise
    """
    output = shell.execute('bash -c \'netstat -tlnp | grep %s\'' % port)
    return output is not None


def well_known_port(port):
    """
    Check if specified port is a well known port used either by Linux
    system services or Avaya applications.

    Returns (port, application-name) tuple if port is a well known port,
    None otherwise.
    """
    if port in KNOWN_PORTS:
        return port, KNOWN_PORTS[port]


def dns():
    """
    Return a list with DNS servers
    """
    dns = []
    if os.path.isfile(DNS_CONFIG_FILE):
        with shell.file_open(DNS_CONFIG_FILE) as file:
            for line in file:
                if line.startswith('nameserver') and len(line.split()) > 1:
                    dns.append(line.split()[1])
    return dns


def set_dns(dns):
    """
    Set DNS servers
    """
    existing_dns = []
    if os.path.isfile(DNS_CONFIG_FILE):
        with shell.file_open(DNS_CONFIG_FILE) as file:
            for line in file:
                line = line.strip()
                if line and not line.startswith('nameserver'):
                    existing_dns.append(line)
    with shell.file_open(DNS_CONFIG_FILE, 'w') as file:
        for line in existing_dns:
            file.write("%s\n" % line)
        for server in dns:
            file.write("nameserver %s\n" % server)


def prefix_to_netmask(prefix):
    """
    Returns netmask for the given prefix.

    prefix -- interface prefix (ex: 23)
    """
    bits = 0xffffffff ^ (1 << 32 - int(prefix)) - 1
    return inet_ntoa(pack('>I', bits))


def delete_interface(name):
    """
    Delete interface ifcfg file.

    name:   interface name
    """
    shell.sudo_call("rm %sifcfg-%s" % (NETWORK_SCRIPTS_DIR, name))
    remove_from_vlans(name)


def create_interface(interf):
    """
    Create interface ifcfg file and set interface configuration.

    interf: Interface details
    """
    shell.sudo_call("touch %sifcfg-%s" % (NETWORK_SCRIPTS_DIR, interf.name))
    set_interface(interf)


def get_ipoffice_lan_info():
    """
    Get IP Office LAN data.
    """
    ipoffice_lans = {}
    if os.path.exists(IPOFFICE_CONFIG_FILE):
        with shell.file_open(IPOFFICE_CONFIG_FILE) as file:
            for line in file:
                if line.startswith("#"):
                    continue
                if line.startswith("export IPOFFICE_LAN"):
                    key, value = line.split("=")
                    lan = key.strip().replace("export IPOFFICE_LAN", "")
                    value = value.strip()
                    if value == "":
                        value = "none"
                    ipoffice_lans[lan] = {"value": value}
        return ipoffice_lans
    else:
        raise ConfigFileError(i18n.custom_gettext("The %s file does not exist. Please check if IP Office is installed.")
                              % IPOFFICE_CONFIG_FILE)

def get_traffic_control_info(lan_id):
    """
    Get traffic control data.
    """
    tc = False
    if os.path.exists(IPOFFICE_CONFIG_FILE):
        with shell.file_open(IPOFFICE_CONFIG_FILE) as file:
            for line in file:
                if line.startswith("#"):
                    continue
                if line.startswith("export SYSMON_TC_LAN%s" % lan_id):
                    key, value = line.split("=")
                    if value.strip() == "on":
                        tc = True
        return tc
    else:
        raise ConfigFileError(i18n.custom_gettext("The %s file does not exist. Please check if IP Office is installed.")
        % IPOFFICE_CONFIG_FILE)


def set_ipoffice_lan(lan_id, eth_id, lan_check_id, traffic_control):
    """
    Set IP Office LAN configuration.
    """
    if os.path.exists(IPOFFICE_CONFIG_FILE):
        if eth_id != _get_ipoffice_eth_id(lan_check_id):
            out = []
            with shell.file_open(IPOFFICE_CONFIG_FILE) as file:
                for line in file:
                    if line.startswith("export IPOFFICE_LAN%s" % lan_id):
                        if eth_id != "none":
                            out.append("export IPOFFICE_LAN%s=%s\n" % (lan_id, eth_id))
                        else:
                            out.append("export IPOFFICE_LAN%s=\n" % lan_id)
                    elif line.startswith("export SYSMON_TC_LAN%s" % lan_id):
                        out.append("export SYSMON_TC_LAN%s=%s\n" % (lan_id, traffic_control))
                    else:
                        out.append(line)
            with shell.file_open(IPOFFICE_CONFIG_FILE, 'w') as outfile:
                outfile.writelines(out)
        else:
            raise ConfigFileError(i18n.custom_gettext("LAN%s cannot have the same value as LAN%s.") % (lan_id, lan_check_id))
    else:
        raise ConfigFileError(i18n.custom_gettext("The %s file does not exist. Please check if IP Office is installed.")
                              % IPOFFICE_CONFIG_FILE)


def _get_ipoffice_eth_id(lan_id):
    """
    Get the value of a specific LAN.
    """
    eth_id = ""
    with shell.file_open(IPOFFICE_CONFIG_FILE) as file:
        for line in file:
            if line.startswith("#"):
                continue
            if line.startswith("export IPOFFICE_LAN%s" % lan_id):
                eth_id = line.split("=")[1].strip()
                break
    return eth_id


def set_vmpro_host_ip(ip):
    """
    Update VMPro settings file with host IP address.
    """
    shell.sudo_execute("chmod 777 " + VMPRO_SETTINGS_FILE )
    with shell.file_open(VMPRO_SETTINGS_FILE, 'r+'):
        config = ConfigObj(write_empty_values=True )#, spaces_around_kv_sep=False)
        config.filename = VMPRO_SETTINGS_FILE
        config.reload()
        config['HOSTIPADDRESS'] = ip
        config.write(space_around_delimiters=False)
    shell.sudo_execute("chmod 664 " + VMPRO_SETTINGS_FILE )


class Test(unittest.TestCase):

    def test_get_hostname(self):
        self.assertTrue(hostname())

    def test_parse_interface(self):
        s = """
        name=eth0
        use_dhcp=no
        ipaddr=80.96.82.232
        netmask="255.255.255.0"
        gateway=80.96.82.254
        peerdns=no
        """
        interf = _parse_interface(s)
        self.assertEqual('eth0', interf.name)
        self.assertEqual(False, interf.use_dhcp)
        self.assertEqual('80.96.82.232', interf.ipaddr)
        self.assertEqual('255.255.255.0', interf.netmask)
        self.assertEqual('80.96.82.254', interf.gateway)
        self.assertEqual(False, interf.auto_dns)

    def test_parse_interface_no_config_file(self):
        s = """
        name=eth1
        use_dhcp=no
        ipaddr=
        netmask=
        gateway=80.96.82.254
        peerdns=no
        """
        interf = _parse_interface(s)
        self.assertEqual('eth1', interf.name)
        self.assertEqual(False, interf.use_dhcp)
        self.assertEqual('', interf.ipaddr)
        self.assertEqual('', interf.netmask)
        self.assertEqual('80.96.82.254', interf.gateway)
        self.assertEqual(False, interf.auto_dns)

    def test_get_eth0(self):
        interf = interface("eth0")
        self.assertEqual("eth0", interf.name)
        self.assertTrue(interf.ipaddr)
        self.assertTrue(interf.netmask)
        if not interf.use_dhcp:
            self.assertTrue(interf.gateway)

    def test_get_interfaces(self):
        self.assertTrue(get_interfaces())

    def test_get_eth0_ip(self):
        self.assertTrue(eth0_ip())

    def test_prefix_to_netmask(self):
        self.assertEqual(prefix_to_netmask(23), '255.255.254.0')
        self.assertEqual(prefix_to_netmask(24), '255.255.255.0')
        self.assertEqual(prefix_to_netmask(25), '255.255.255.128')
        self.assertEqual(prefix_to_netmask(26), '255.255.255.192')
        self.assertEqual(prefix_to_netmask(27), '255.255.255.224')
        self.assertEqual(prefix_to_netmask(28), '255.255.255.240')
        self.assertEqual(prefix_to_netmask(29), '255.255.255.248')
        self.assertEqual(prefix_to_netmask(30), '255.255.255.252')
        self.assertEqual(prefix_to_netmask(31), '255.255.255.254')


class TestPorts(unittest.TestCase):

    port = 47821

    def setUp(self):
        self.server  = socketserver.TCPServer(('127.0.0.1', self.port), DummyTCPHandler)
        t = threading.Thread(target=self.server.serve_forever)
        t.start()

    def tearDown(self):
        TestPorts.port += 1
        self.server.shutdown()

    def test_is_port_in_use(self):
        self.assertTrue(is_port_in_use(self.port))

    def test_port_is_not_in_use(self):
        self.assertFalse(is_port_in_use(self.port + 1))

    def test_is_well_known_port(self):
        self.assertNotEqual(well_known_port(80), None)

    def test_is_not_well_known_port(self):
        self.assertEqual(well_known_port(self.port), None)


class DummyTCPHandler(socketserver.StreamRequestHandler):

    def handle(self):
        self.rfile.readline()
        self.wfile.write("test")


class TestDns(unittest.TestCase):

    def setUp(self):
        global DNS_CONFIG_FILE
        self.orig_dns_config = DNS_CONFIG_FILE

        import tempfile
        _, self.tmp_file = tempfile.mkstemp()

    def tearDown(self):
        global DNS_CONFIG_FILE
        DNS_CONFIG_FILE = self.orig_dns_config
        if os.path.isfile(self.tmp_file):
            os.remove(self.tmp_file)

    def test_get_dns(self):
        self.assertTrue(dns() is not None)

    def test_get_dns_no_resolv_conf_file(self):
        global DNS_CONFIG_FILE
        DNS_CONFIG_FILE = self.tmp_file
        self.assertTrue(dns() is not None)


class TestVMPro(unittest.TestCase):

    def setUp(self):
        global VMPRO_SETTINGS_FILE
        self.orig_vmpro_config = VMPRO_SETTINGS_FILE

        import tempfile
        _, self.tmp_file = tempfile.mkstemp()
        VMPRO_SETTINGS_FILE = self.tmp_file

    def tearDown(self):
        global VMPRO_SETTINGS_FILE
        VMPRO_SETTINGS_FILE = self.orig_vmpro_config
        if os.path.isfile(self.tmp_file):
            os.remove(self.tmp_file)

    def test_set_vmpro_set_host_ip(self):
        expected_ip = '192.168.10.23'
        set_vmpro_host_ip(expected_ip)

        saved_config = ConfigObj(VMPRO_SETTINGS_FILE)
        self.assertEqual(expected_ip, saved_config['HOSTIPADDRESS'])

    def test_config_file_does_not_exists(self):
        shell.rm(VMPRO_SETTINGS_FILE)
        self.assertRaises(IOError, set_vmpro_host_ip, '192.168.10.23')


if __name__ == "__main__":
    unittest.main()
