# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
RPM utility module
"""

import os
import re
import tempfile
import unittest

from . import shell
from core.common import utils
from core.common import log

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

TAG = 'rpm'

UPGRADE_SCRIPT = '/opt/Avaya/scripts/upgrade.sh'


class RpmError(Exception):
    """
    For rpm commands errors.
    """
    def __str__(self):
        return repr(self.args[0])


def rpm(args):
    """
    Execute rpm command with given arguments and return command output.
    Raise RpmError for rpm command errors.
    """
    fd_out, temp_out = tempfile.mkstemp()
    fd_err, temp_err = tempfile.mkstemp()
    exit_code = os.system("sudo %s %s %s %s" %
                          (UPGRADE_SCRIPT, temp_out, temp_err, re.escape(args)))

    out = ''
    if os.access(temp_out, os.R_OK):
        out = open(temp_out).read()

    err = ''
    if os.access(temp_err, os.R_OK):
        err = open(temp_err).read()

    os.close(fd_out)
    os.remove(temp_out)
    os.close(fd_err)
    os.remove(temp_err)

    if exit_code:
        if "-e" in args.split(" ") or "-i" in args.split(" "):
            return err or out
        else:
            raise RpmError(err or out)
    elif "-e" in args.split(" ") or "-i" in args.split(" "):
        return ""

    return out


def query_package(name, query_tags=None):
    """
    Return a dictionary {query_tag: value,...}  for specified package
    or None if the package is not installed.

    Args:

    name        -- package name
    querytags   -- a list of RPM query tags
                   if set to None the default query tags are NAME, VERSION and RELEASE
                   see rpm --querytags for a complete list
    """
    if not query_tags:
        query_tags = ["NAME", "VERSION", "RELEASE"]
    args = []
    for tag in query_tags:
        args.append("%%{%s}\\n" % tag)
    try:
        output = rpm('-q --queryformat "%s" %s' % (''.join(args), name))
        return dict(zip(query_tags, output.split('\n')))
    except RpmError as e:
        log.debug(TAG, "unable to query package: %s: %s" % (name, e))


def query_rpm(rpm_url, query_tags=None):
    """
    Return a dictionary {query_tag: value,...}  for specified RPM file
    or None if the RPM url is not accessible.

    Args:

    rpm_url     -- file url
    querytags   -- a list of RPM query tags
                   if set to None the default query tags are NAME, VERSION and RELEASE
                   see rpm --querytags for a complete list
    """    
    if not query_tags:
        query_tags = ["NAME", "VERSION", "RELEASE"]
    args = []
    for tag in query_tags:
        args.append("%%{%s}\\n" % tag)
    try:
        output = rpm('-qp --queryformat "%s" %s' % (''.join(args), rpm_url))
        return dict(zip(query_tags, output.split('\n')))
    except RpmError as e:
        log.warn(TAG, "unable to query rpm file: %s: %s" % (rpm_url, e))


def compare_versions(v1, v2):
    """
    RPM package version comparator.

    Args:

    v1, v2 -- version strings to compare

    The expected format for version strings is:

    version_number-release_number

    ex: 6.1.3-5.0

    This function wil return:

    -1      if ver1 < ver2
     0      if ver1 == ver2
     1      if ver1 > ver2
    """
    for t in zip(v1.split('-'), v2.split('-')):
        diff = _compare_version(t[0], t[1])
        if diff:
            return diff
    return 0


def _compare_version(v1, v2):
    """
    Compare two version strings in dotted format.
    """
    v1_nums = map(utils.parse_int, v1.split('.'))
    v2_nums = map(utils.parse_int, v2.split('.'))

    for t in zip(v1_nums, v2_nums):
        if t[0] < t[1]:
            return -1
        elif t[0] > t[1]:
            return 1

    len_diff = len(list(v1_nums)) - len(list(v2_nums))
    if len_diff > 0:
        return 1
    elif len_diff < 0:
        return -1

    return 0


def upgrade(rpm_url, service_name=None, extra_opts=None):
    """
    Upgrade installed package.

    Args:

    rpm_url         -- new package url
    service_name    -- managed service
                           if this argument is not None then specified service will be
                           stopped before upgrade and started afterwards
    extra_opts      -- extra options to pass to rpm command

    Raise RpmError if <rpm -Uvh rpm_url> command fails.
    """
    if extra_opts is None:
        extra_opts = ''

    if service_name:
        output = shell.sudo_execute("/etc/init.d/%s status" % service_name, parse=shell.RAW)
        if b"stopped" in output:
            rpm('-Uvh %s %s' % (extra_opts, rpm_url))
        else:
            shell.sudo_call('systemctl stop %s' % service_name)
            rpm('-Uvh %s %s' % (extra_opts, rpm_url))
            shell.sudo_call('systemctl start %s' % service_name)
    else:
        rpm('-Uvh %s %s' % (extra_opts, rpm_url))


def downgrade(rpm_url, service_name=None, extra_opts=None):
    """
    Downgrade installed package.

    Args:

    rpm_url         -- new package url
    service_name    -- managed service
                       if set then specified service will be stopped before downgrade and started aftewards
    extra_opts      -- extra options to pass to rpm command

    Raise RpmError if <rpm -Uvh --oldpackage rpm_url> command fails.
    """
    if extra_opts is None:
        extra_opts = ''

    if service_name:
        output = shell.sudo_execute("/etc/init.d/%s status" % service_name, parse=shell.RAW)
        if b"stopped" in output:
            rpm('-Uvh %s --oldpackage %s' % (extra_opts, rpm_url))
        else:
            shell.sudo_call('systemctl stop %s' % service_name)
            rpm('-Uvh %s --oldpackage %s' % (extra_opts, rpm_url))
            shell.sudo_call('systemctl start %s' % service_name)
    else:
        rpm('-Uvh %s --oldpackage %s' % (extra_opts, rpm_url))


def install(rpm_url, extra_opts=None):
    """
    Install package.

    Args:

    rpm_url -- new package url

    Return the exit code for <rpm -i rpm_url> command.

    Raise RpmError if <rpm -i rpm_url> command fails.
    """
    if extra_opts is None:
        extra_opts = ''
    output = rpm('-i %s %s' % (extra_opts, rpm_url))
    if output:
        return output


def uninstall(name, extra_opts=None):
    """
    Uninstall package.

    Args:

    name            -- package name
    extra_opts      -- extra options to pass to rpm command

    Return the exit code for <rpm -e package_name> command.

    Raise RpmError if <rpm -e package_name> command fails.
    """
    if extra_opts is None:
        extra_opts = ''

    try:
        output = rpm('-e %s %s' % (name, extra_opts))
        if output:
            return output
    except: pass



def valid_rpm(rpm_url):
    """
    Check rpm file and return True if it's not corrupted.

    rpm_url -- path to rpm file
    """
    try:
        output = rpm("-K %s" % rpm_url)
        if 'NOT OK' in output:
            return False
        return True
    except RpmError as e:
        log.warn(TAG, "invalid rpm file: %s: %s" % (rpm_url, e))
        return False


class Test(unittest.TestCase):

    def test_query_installed_package(self):
        query_tags = ['NAME', 'VERSION', 'RELEASE']
        pkg_info = query_package('webcontrol', query_tags)
        if pkg_info:
            for tag in query_tags:
                self.assertTrue(pkg_info[tag])

    def test_query_not_installed_package(self):
        self.assertFalse(query_package('non-existing-package'))

    def test_query_existing_rpm(self):
        query_tags = ['NAME', 'VERSION', 'RELEASE']
        # search for RPM files to query
        import glob
        for rpm_file in glob.glob('/var/cache/avaya/apps/*.rpm'):
            pkg_info = query_rpm(rpm_file, query_tags)
            for tag in query_tags:
                self.assertTrue(pkg_info[tag])

    def test_query_non_existing_rpm(self):
        self.assertFalse(query_rpm('non-existing.rpm'))

    def test_equal_versions(self):
        test_data = ('6.1.3-4', '6.1.3-4.0')
        for version in test_data:
            self.assertTrue(compare_versions(version, version) == 0)

    def test_compare_versions_unequal_same_len(self):
        test_data = (('6.1.5-1', '6.1.4-1'),
                     ('6.1.4-7', '6.1.4-6'),
                     ('6.1.4-7.0', '6.1.4-6.1'))
        for newer, older in test_data:
            self.assertTrue(compare_versions(newer, older) > 0)

    def test_compare_versions_unequal_different_len(self):
        test_data = (('6.1.3.1-4', '6.1.3-4'),
                     ('6.1.3-4.1', '6.1.3-4'))
        for newer, older in test_data:
            self.assertTrue(compare_versions(newer, older) > 0)

    def test_compare_versions_no_release_number(self):
        test_data = (('6.1.11.59', '6.1.10.59'),
                     ('6.1.10.59.1', '6.1.10.59'),
                     ('6.2.10.59', '6.1.10.59.2'))
        for newer, older in test_data:
            self.assertTrue(compare_versions(newer, older) > 0)

    def test_compare_versions_alphanum(self):
        test_data = (('6.1.1-b10', '6.1.1-b9'),
                     ('6.1.a.1', '6.1.a.0'),
                     ('6.2-1.x.10', '6.2-1.x.9'))
        for newer, older in test_data:
            self.assertTrue(compare_versions(newer, older) > 0)

    def test_compare_versions_special_build_number(self):
        test_data = (('8.1.7-006222', '8.1.7-3'),
                     ('8.1.6-031', '8.1.6-0027'),
                     ('8.1.7-0042', '8.1.6-101'))
        for newer, older in test_data:
            self.assertTrue(compare_versions(newer, older) > 0)

    def test_invalid_rpm(self):
        invalid_rpm_url = 'non-existing.rpm'
        self.assertFalse(valid_rpm(invalid_rpm_url))

    def test_valid_rpm(self):
        # search for RPM files to validate
        import glob
        for rpm_file in glob.glob('/var/cache/avaya/apps/*.rpm'):
            self.assertTrue(valid_rpm(rpm_file))


if __name__ == '__main__':
    unittest.main()
