# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Shell utilities.
"""
import fnmatch
import unittest
import subprocess
import shlex
import os
import random
import web
import wsgiref.util
import mimetypes
import datetime

from core.common import configuration

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

# constants for execute parse mode
LINES = 0
LINES_AND_COLUMNS = 1
RAW = 2


def execute(command, parse=LINES_AND_COLUMNS):
    """
    Execute command and return output. Returns None if no output is generated.

    command  -- command string to be execute
    parse    -- specify how to split into tokens the command output

    The value for parse parameter is:

    LINES             -- the output is parsed into lines
    LINES_AND_COLUMNS -- the output is parsed into lines, each line is split into tokens
    RAW             -- the output is returned as it is    
    """

    process = subprocess.Popen(shlex.split(command), stdout=subprocess.PIPE)
    output = process.communicate()[0]
    if output:
        if parse == LINES_AND_COLUMNS:
            return [[ token for token in line.split()]
                    for line in output.decode().split('\n')]
        elif parse == LINES:
            return [line for line in output.decode().split('\n')]
        else:
        # assume RAW
            return output


def execute2(cmd):
    """
    Execute command and return a tuple (exit_code, stdout, stderr).

    Args:

    cmd -- command string or string list for multiple commands
    """
    if isinstance(cmd, list):
        args = ';'.join(cmd)
        use_shell = True
    else:
        args = shlex.split(cmd)
        use_shell = False
    process = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=use_shell)
    output = process.communicate()
    return process.returncode, output[0], output[1]


def call(command):
    """
    Run command and return exit code.
    """
    return subprocess.call(shlex.split(command))


def sudo_call(command):
    """
    Execute command with sudo and return exit code.
    """
    return call('sudo %s' % command)


def sudo_calls(commands):
    """
    Execute commands list in a single call and return exit code.
    """
    sudo_commands = map(lambda c: "sudo %s" % c, commands)
    return subprocess.call(';'.join(sudo_commands), shell=True)


def sudo_execute(command, parse=LINES_AND_COLUMNS):
    """
    Execute command with sudo and return output.

    command -- command string to be executed
    parse   -- specify how to split into tokens the command output
               (see execute for details on parse parameter)
    """
    return execute('sudo %s' % command, parse)


def sudo_execute2(cmd):
    """
    Execute command with sudo and return a tuple (exit_code, stdout, stderr).

    Args:

    cmd -- command string or string list for multiple commands
    """
    if isinstance(cmd, list):
        return execute2(map(lambda c: 'sudo %s' % c, cmd))
    return execute2('sudo %s' % cmd)


def ps(name):
    """
    Returns a tuple object containing process status:

    (cpu, mem, uptime, pid)

    where

    cpu      -- percent CPU usage
    mem      -- memory usage in kilobytes
    uptime   -- process uptime value (as string)
    pid      -- process ID

    or None if the process is not running
    """
    output = execute('ps --no-headers -o pcpu,rss,etime,pid -C %s' % name)
    if output:
        print(output)
        return float(str(output[0][0])), int(str(output[0][1])), str(output[0][2]), int(str(output[0][3]))


def chkconfig(name):
    """
    Returns a tuple with runlevels for specified service
    Return empty tuple for non existent or disabled services.

    Ex: chkconfig('webcontrol') => (3, 4, 5)
    """
    runlevels = []
    output = os.popen("systemctl list-unit-files %s.service" % name).read()
    for line in output.split('\n'):
        if line.startswith(name):
            tokens = line.split(' ')
            if tokens[1] == 'enabled':
                runlevels=('3','4','5')
    return tuple(runlevels)


def enable_service(name):
    """
    Enable specified /etc/init.d/<name> service using chkconfig utility.
    Runs the ``chkconfig --add <name>`` and ``chkconfig <name> on`` commands in sequence,
    returns the outcome.
    """
#    add_cmd = 'chkconfig --add %s' % name
    on_cmd = 'systemctl enable %s.service' % name
    return sudo_calls([on_cmd])


def disable_service(name):
    """
    Disable specified /etc/init.d/<name> service using chkconfig utility.
    Returns the exit code of ``chkconfig --del <name>`` command.
    """
    return sudo_call('systemctl disable %s.service' % name)


def tail(filepath, count=10):
    """
    Tail file.

    Args:

    filepath    -- file path
    count       -- how many lines
    """
    with file_open(filepath,'rb') as f:
        f.seek(0, 2)
        bytes = f.tell()
        size = count
        block = -1
        while size > 0 and bytes + block * 1024 > 0:
            if bytes + block * 1024 > 0:
                f.seek(block * 1024, 2)
            else:
                f.seek(0, 0)
            data = f.read(1024)
            lines_found = data.decode().count('\n')
            size -= lines_found
            block -= 1
        try:
            f.seek(block * 1024, 2)
            f.readline()
        except IOError:
        # file is smaller than expected
            f.seek(0, 0)
        last_blocks = list(f.readlines())
        return last_blocks[-count:]


def find_file(file=None, path="."):
    """
    Search for a file in a given path.
    Return full file path or None if file not found
    """
    if file is None:
        return None
    for root, dirnames, filenames in os.walk(path):
        for filename in filenames:
            if filename == file:
                return os.path.join(root, filename)
    return None


def secure_find_file(filepath=None):
    """
    Search for a file path, without modifying access rights.
    Return True if the file is found, False otherwise.
    """
    if filepath is None:
        return False
    else:
        output = sudo_execute("ls -la %s" % filepath, parse=RAW)
        if output is None:
            return False
        else:
            if not "cannot access" in output.decode():
                return True
    return False


def secure_get_file(file_name=None, file_path=None):
    """
    Get a file's data, without modifying access rights on its path.
    """
    if file_name and file_path:

        folder = "%s" % random.randrange(1000, 1000000)
        temp_path = "/tmp/%s" % folder
        sudo_call("mkdir %s" % temp_path)
        sudo_call("chmod 777 %s" % temp_path)
        sudo_call("cp %s%s %s/" % (file_path, file_name, temp_path))
        temp_file_path = "%s/%s" % (temp_path, file_name)
        sudo_call("chmod 777 %s" % temp_file_path)

        block_size = 16384
        mime_type = mimetypes.guess_type(file_name)[0]
        web.header('Content-Type', '%s' % mime_type)
        web.header('Content-Disposition', 'attachment; filename=%s' % file_name)
        stat = os.stat(temp_file_path)
        web.header('Content-Length', '%s' % stat.st_size)
        web.header('Last-Modified', '%s' % web.http.lastmodified(datetime.datetime.fromtimestamp(stat.st_mtime)))
        if web.ctx.protocol.lower() == 'https':
            # add headers to fix issue with IE download over SSL failing when "no-cache" header set
            web.header('Pragma', 'private')
            web.header('Cache-Control', 'private,must-revalidate')
        download_file = wsgiref.util.FileWrapper(open(temp_file_path, 'rb'), block_size)

        sudo_call("rm -rf %s" % temp_path)

        return download_file

    return None




def file_open(filepath, mode='r'):
    """
    Open file according to specified mode. The difference between standard open() library
    call and this function is that this function will make the specified file writable
    and/or readable  by webcontrol user group if required before open it. This function is useful
    when the permissions for a file is changed, for example when a package is upgraded
    and the permissions for its configuration files are changed.

    Args:

    filepath -- file path
    mode     -- one of r, w, r+, w+, a, rw
    """
    if mode in ('r', 'rb', 'r+b'):
        access = os.R_OK
        perm = 'r'
    elif mode in ('w', 'w+', 'a', 'wb', 'w+b'):
        access = os.W_OK
        perm = 'w'
    elif mode in ('r+', 'rw', 'r+w'):
        access = os.W_OK | os.R_OK
        perm = 'rw'
    else:
        # unknown mode, assume file open for read
        access = os.R_OK
        perm = 'r'
    if os.path.isfile(filepath) and not os.access(filepath, access):
        sudo_call('chmod g+%s %s' % (perm, filepath))
    else:
        dir = os.path.dirname(filepath)
        if not os.access(dir, os.R_OK):
            sudo_call('chmod g+%s %s' % ("r", dir))
    return open(filepath, mode)


def shutdown(restart=False):
    """
    Issue a shutdown/reboot command to the operating system.

    Args:

    restart -- if True the system should be rebooted, otherwise a shutdown only is required
    """
    if restart:
        sudo_call('shutdown -hr -t 5 now')
    else:
        sudo_call('shutdown -hP -t 5 now')


def zip(path, zipfile, options=None):
    """
    Create zip archive from specified path.

    This command uses 'sudo zip' shell command instead of python zipfile API
    to avoid problems with file and directory access rights.

    path        -- file or directory or files and directories list
    zipfile     -- output zip file
    options     -- zip command options

    Returns a tuple (exit_code, stdout, stderr).
    """
    opts = options or '-r'

    if type(path) == list:
        files = ' '.join(path)
    else:
        files = path
    return sudo_execute2('zip %s %s %s' % (opts, zipfile, files))


def rm(path):
    """
    Remove (delete) existing files.

    This command uses 'sudo rm -fr' shell command instead of python OS API
    in order to avoid problems with file an directory access rights.

    path -- file or directory or files and directories list

    Returns a tuple (exit_code, stdout, stderr).
    """
    if not path:
        return
    if type(path) == list:
        files = ' '.join(path)
    else:
        files = path
    return sudo_execute2('rm -fr %s' % files)


def find(path, pattern):
    """
    Returns a list of files matching the given pattern.

    path -- file or directory or files and directories list
    pattern -- pattern list
    """
    if path == '':
        return
    if type(path) == list:
        paths = ' '.join(path)
    else:
        paths = path

    file_list = []
    for root, dirnames, filenames in os.walk(paths):
        for filename in fnmatch.filter(filenames, pattern):
            file_list.append(os.path.join(root, filename))

    return file_list


def cat(filename):
    """
    Return the content of the file.

    filename    -- file to be listed
    """
    content = []
    with file_open(filename) as f:
        for line in f:
            content.append(line)
    return ''.join(content)


def touch(path):
    """
    Creates/touch new/existing files

    This function uses 'sudo touch path' command instead of Python OS API
    to avoid problems with file or directory access rights.

    Returns a tuple (exit_code, stdout, stderr).
    """
    if not path:
        return

    if type(path) == list:
        paths = ' '.join(path)
    else:
        paths = path

    return sudo_execute2("touch %s" % paths)


def banner(default=''):
    """
    Return the contents of the banner file.
    Banner file is defined in the global configuration file,
    """
    banner_file = configuration.SHARED_CONFIGURATION['webapp']['banner_file']
    if os.path.exists(banner_file):
        return cat(banner_file)
    return default


def bash_execute(command, parse=LINES_AND_COLUMNS):
    """
    Bash execute command and return output. Returns None if no output is generated.

    command  -- command string to be execute
    parse    -- specify how to split into tokens the command output

    The value for parse paramenter is:

    LINES             -- the ouput is parsed into lines
    LINES_AND_COLUMNS -- the output is parsed into lines, each line is split into tokens
    RAW               -- the output is returned as it is
    """
    return execute("bash -c \"%s\"" % command, parse)


class Test(unittest.TestCase):

    def test_execute(self):
        self.assertTrue(execute('ls -l'))

    def test_call(self):
        self.assertEqual(call('ls -l'), 0)

    def test_sudo_call(self):
        self.assertEqual(sudo_call('nstat'), 0)

    def test_nonexistent_command(self):
        try:
            execute('nonexistentcommand')
            self.fail('nonexistentcommand executed')
        except OSError:
            pass

    def test_execute2(self):
        exit_code, out, err = execute2('pwd')
        self.assertEqual(0, exit_code)
        self.assertTrue(out)
        self.assertFalse(err)

    def test_execute2_multiple(self):
        exit_code, out, err = execute2(['ls -l','pwd'])
        self.assertEqual(0, exit_code)
        self.assertTrue(out)
        self.assertFalse(err)

    def test_sudo_execute2(self):
        exit_code, out, err = sudo_execute2('nstat')
        self.assertEqual(0, exit_code)
        self.assertTrue(out)
        self.assertFalse(err)

    def test_sudo_execute2_multiple(self):
        exit_code, out, err = sudo_execute2(['ls /usr/sbin', 'hwclock', 'nstat'])
        self.assertEqual(0, exit_code)
        self.assertTrue(out)
        self.assertFalse(err)

    def test_ps(self):
        self.assertTrue(ps('python2.6'))

    def test_chkconfig(self):
        sudo_call('chkconfig --del watchdog')
        self.assertFalse(chkconfig('watchdog'))
        sudo_call('chkconfig --add watchdog')
        self.assertTrue(chkconfig('watchdog'))

    def test_enable_service(self):
        service = 'watchdog'
        disable_service(service)
        self.assertFalse(chkconfig(service))
        enable_service(service)
        self.assertTrue(chkconfig(service))

    def test_disable_service(self):
        service = 'watchdog'
        disable_service(service)
        self.assertFalse(chkconfig(service))
        enable_service(service)

    def test_tail(self):
        filepath = '/var/log/messages'
        self.assertEqual(len(tail(filepath, 2)), 2)

    def test_zip(self):
        import tempfile
        test_dir = tempfile.mkdtemp()
        zipfile  = 'test.zip'
        execute2('touch %s/a.txt' % test_dir)
        execute2('touch %s/b.txt' % test_dir)
        zip(test_dir, zipfile)
        self.assertTrue(os.path.exists(zipfile))
        execute2('sudo rm -f %s' % zipfile)

    def test_rm(self):
        import tempfile
        test_dir = tempfile.mkdtemp()
        execute2('touch %s/a.txt' % test_dir)
        execute2('touch %s/b.txt' % test_dir)
        rm(test_dir)
        self.assertFalse(os.path.exists(test_dir))

    def test_cat(self):
        self.assertTrue(cat('/etc/redhat-release'))

    def test_banner(self):
        banner()

    def test_find(self):
        file_list = find('/etc/','*.conf')
        self.assertTrue(file_list)

    def test_rm_find(self):
        import tempfile
        test_dir = tempfile.mkdtemp()
        execute2('touch %s/a.txt' % test_dir)
        execute2('touch %s/b.txt' % test_dir)
        rm(find(test_dir, "*.txt"))
        self.assertFalse(find(test_dir, '*.txt'))
        rm(test_dir)

    def test_touch(self):
        import tempfile
        test_dir = tempfile.mkdtemp()
        test_file = "%s/a.txt" % test_dir
        touch(test_file)
        self.assertTrue(os.path.isfile(test_file))
        rm(test_dir)


if __name__ == "__main__":
    unittest.main()
