# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Utility OS functions.
"""

import unittest
from . import shell
import os
import math
from . import systime
import re

from core.common import utils
from core.common import version
from core.common import configuration
from core.common import i18n
from core.common import log

__author__ = "Avaya Inc."
__copyright__ = "Copyright 2010, Avaya Inc."

# contains the timestamp for last OS update
OS_LAST_UPDATE_FILE = os.path.join(os.getenv('WEBCONTROL_HOME', '.'),
    configuration.SHARED_CONFIGURATION['webapp']['os_last_update_file'])

# this file is generated by system_info.pl at system startup
INV_SYS_INFO_FILE = "/opt/Avaya/system_info/system_info.txt"

# List of Avaya supplied Linux based servers as of R10 release and 
# taken from Henry's master spreadsheet
#
# 000270393	DL360G7 SRVR IPO R8.1+ SE PRIMARY
# 000306961	R620 SRVR IPO SE Primary 
# 000270395	DL120G7 SRVR IPO R8.1+ SE EXP
# 000269810	DL120G7 SRVR IPO UC
# 000270391	DL120G7 SRVR IPO FRANCE TELECOM
# 000302788	R210 II XL SRVR IPO SE EXP
# 000302786	R210 II XL SRVR IPO UC
# 000302787	R210 II XL SRVR IPO FRANCE TELECOM
# 000302788 R210 II XL SERVER IP OFFICE SERVER EDITION EXPANSION
# 000380223 R220 XL SERVER IP OFFICE UNIFIED COMMUNICATIONS
# 000380224 R220 XL SERVER IP OFFICE FRANCE TELECOM
# 000380225 R220 XL SERVER IP OFFICE SERVER EDITION EXPANSION
# 000380226 R220 XL SRVR IPOCC
# 000383542 R630 SRVR IPO SE PRIMARY
# 000390081 R230 XL SRVR IPO UC
# 000390082 R230 XL SRVR IPO FRANCE TELECOM

AVAYA_ASSET_TAGS = ('000270391',
                    '000270393',
                    '000279395',
                    '000269810',
                    '000302786',
                    '000302787',
                    '000302788',
                    '000306961',
                    '000380223',
                    '000380224',
                    '000380225',
                    '000380226',
                    '000383542',
                    '000390081',
                    '000390082')

VIRTUALIZATION_MARKER   = '/opt/Avaya/.ova'

def os_info():
    """
    Returns a tuple containing OS specific info in the form of a tuple:

        (version, kernel, uptime, server_time, last_os_update)

        uptime is an OpenStruct object:

        uptime.days     -- days of uptime (0 for uptime smaller than one day)
        uptime.hours    -- hours of uptime (0 for uptime smaller than one hour)
        uptime.minutes  -- minutes of uptime
        uptime.seconds  -- seconds of uptime
    """
    output = shell.execute('cat /etc/redhat-release')
    version = ' '.join(output[0])

    output = shell.execute('uname -r')
    kernel = ' '.join(output[0])

    up_time = utils.OpenStruct(days=0, hours=0, minutes=0, seconds=0)
    t = uptime()
    up_time.days = t[0]
    up_time.hours = t[1]
    up_time.minutes = t[2]
    up_time.seconds = t[3]

    last_update = None
    if os.path.exists(OS_LAST_UPDATE_FILE):
        with open(OS_LAST_UPDATE_FILE) as f:
            last_update = f.read().strip()

    server_time = utils.format_timestamp(systime.date(), "%H:%M")

    server_type = shell.sudo_execute("dmidecode -s system-product-name", parse=shell.RAW).strip()

    return version, kernel, up_time, server_time, last_update, server_type


def sys_inv_info():
    """
    Returns System inventory information in the form of a tuple.
    The information is read by looking at file specified in INV_SYS_INFO_FILE.
    The file specified in INV_SYS_INFO_FILE is generated at system boot.

        (material_code, model_info, sys_serial_num, sys_type)

    sys_type is set to 'Avaya IP Office' if current system
    is an Avaya supplied server.

    """
    inv_info = {}

    if os.path.isfile(INV_SYS_INFO_FILE):
        with shell.file_open(INV_SYS_INFO_FILE) as file:
            for line in file:
                tokens = line.split('=')
                inv_info[tokens[0]] = ('='.join(tokens[1:])).strip()

    material_code = inv_info.get("MATERIAL_CODE", "")
    model_info = inv_info.get("MODEL_INFO", "")
    sys_serial_num = inv_info.get("SYS_SERIAL_NUM", "")
    if material_code in AVAYA_ASSET_TAGS:
        sys_type = 'Avaya IP Office'
    else:
        sys_type = 'unknown'

    # XXX ugly
    sys_type = i18n.custom_gettext(sys_type)

    return material_code, model_info, sys_serial_num, sys_type


def system_type():
    """
    Returns the type of the system as an object.
    Format: {virtual_system: True/False, cloud_system: True/False}
    """
    is_virtual = os.path.isfile(configuration.SHARED_CONFIGURATION['virtualization']['virtualization_file'])
    auto_ignition_complete = os.path.isfile(configuration.SHARED_CONFIGURATION['autoconfig']['complete_flag'])
    auto_ignition_not_complete = os.path.isfile(configuration.SHARED_CONFIGURATION['autoconfig']['cloud_flag'])
    is_cloud = False
    if is_virtual is True:
        if auto_ignition_complete is True or auto_ignition_not_complete is True:
            is_cloud = True
    return {"virtual_system": is_virtual, "cloud_system": is_cloud}

def uptime():
    """
    Return system uptime in a tuple (days, hours, minutes, seconds).
    Uptime values are computed using /proc/uptime file.
    """
    seconds = float(shell.cat('/proc/uptime').split()[0])
    days = math.floor(seconds / 86400)
    seconds -= days * 86400
    hours = math.floor(seconds / 3600)
    seconds -= hours * 3600
    minutes = math.floor(seconds / 60)
    seconds -= minutes * 60

    return int(days), int(hours), int(minutes), int(seconds)


def mem_info():
    """
    Returns a tuple containing info about system memory usage, in the form of a tuple:

        (total, used, free)

    Values are in kilobytes.
    """
    total, used, free = 0, 0, 0
    output = shell.execute('free -k')
    if output:
        total = int(output[1][1])
        used = int(output[1][2])
        free = total - used
        free = 0 if free < 0 else free
        #free = int(output[1][3])
    return total, used, free


def quota_info():
    """
    Returns a tuple containing info about quota usage, in the form of a tuple:

        (total, used, free, threshold)

    Values are in kilobytes.
    """
    total, used, free, threshold = 0, 0, 0, 0

    vmpro_service_name = configuration.SHARED_CONFIGURATION['applications']['voicemail']['service_name']
    status = shell.sudo_execute("rpm -q %s" % vmpro_service_name, parse=shell.RAW)
    if "not installed" in status.decode():
        threshold = configuration.SHARED_CONFIGURATION['quota']['uninstalled_vmrpo_threshold']
    else:
        threshold = configuration.SHARED_CONFIGURATION['quota']['installed_vmpro_threshold']

    output = shell.sudo_execute(configuration.SHARED_CONFIGURATION['quota']['get_quota_script'], parse=shell.RAW)
    if output and len(output.strip()) is not 0:
        total = int(output.split()[3])
        used = int(output.split()[2])
        free = str(total - used) + "K"
    else:
        log.SHARED_LOGGER.error("Invalid quota output. Please reboot to make sure quota group exists.")
    hdd_size = utils.format_bytes(hardware_info().disk.total * 1024)
    return total, used, free, threshold, hdd_size


def disk_info():
    """
    Returns a tuple containing info about disk partition disk usage, in the form of a tuple: 

        (total, used, free)

    Monitored partition is defined in configuration file. Values are in kilobytes.
    """
    total, used, free = 0, 0, 0
    output = shell.execute('df -k %s' % configuration.SHARED_CONFIGURATION['disk']['partition'])
    if output:
        for line in output:
            if line[-1] == '/':
                used = int(line[-5]) - int(line[-3]) # For SSA consistency used space is total space - free space
                free = int(line[-3])
                total = int(line[-5])
                break
    return total, used, free


def raid_info():
    """
    Returns RAID information in the form of a tuple:

        (supportedLevels, arraysTypes) where
        supportedLevels [raid0, raid1, ... raid5]
        arraysTypes [[drive1, raidType1], [drive2, raidType2], ... [drive5, raidType5]]

    The "Personalities" line tells you what RAID level the kernel currently supports.
    md device line md_d0 : active raid5 sde1[0] sdf1[4] sdb1[5] sdd1[2] sdc1[1]
    It is active or 'started'. An inactive array is usually faulty. Stopped arrays aren't visible here.
    """
    supportedLevels = []
    arraysTypes = []
    output = shell.execute('cat /proc/mdstat', shell.LINES)
    if output:
        d = re.compile("md\w*")
        r = re.compile("raid\w*")

        for line in output:
            if ("Personalities :") in line:
                raidLevels = line.split("Personalities :")[1].strip()
                supportedLevels = re.findall("\[(.*?)\]", raidLevels)
            drive = d.search(str(line))
            raid = r.search(str(line))
            if drive and raid:
                arraysTypes.append([drive.group(0), raid.group(0)])
    return supportedLevels, arraysTypes


def cpu_usage():
    """
    Returns a history of global CPU usage in percents. A maximum of five
    values are sampled at an interval depending on the configuration of 'sar' 
    command (default is 10 min).
    
    Output is a list on the following form:

    [[time1, value1], [time2, value2], ... [time5, value5]]

    where time format is: HH:MM:SS AM/PM
    """

    cpu_usage_history = []
    output = shell.execute('sar -u')
    if output:
        for line in reversed(output):
            if len(line) > 7:
                if line[-1] == '%idle': break
                if line[0].strip().startswith('Average'): continue
                cpu_usage_history.append([line[0][0:-3] + line[1], 100 - float(line[-1].replace(',','.'))])
                if len(cpu_usage_history) > 4:
                    break

    cpu_usage_history.reverse()
    return cpu_usage_history


def cpu_loadavg():
    """
    Returns the average CPU load of the last 1, 5 and 15 minutes period.
    """
    output = shell.execute('cat /proc/loadavg')
    if output and len(output[0]) == 5:
        return output[0][0:3]
    else:
        return [0, 0, 0]

def is_virtualized():
    """
    Convenience method to test whether IP Office runs on virtualized hardware.
    """
    return os.path.isfile(VIRTUALIZATION_MARKER)

def gold_edition():
    if version.RELEASE_TYPE == "abe" or version.RELEASE_TYPE == "appl":
        if not os.path.isfile(configuration.SHARED_CONFIGURATION['gold']['gold_edition_flag']):
            return False
    return True

def cpu_info():
    """
    Returns a OpenStruct object containing information about the cpu:

    info.cpu.model  -- cpu model
    info.cpu.mhz    -- cpu mhz
    info.cpu.cores  -- cpu cores

    Implementation uses dmidecode to extract the maximum speed for the processor.
    """
    dmi_decode_out = shell.execute("sudo dmidecode -t processor", parse=shell.LINES)
    cpu_info_out = shell.execute('cat /proc/cpuinfo', parse=shell.LINES)

    cpu_details = "-"
    cpu_MHz = "-"
    cpu_cores = "-"
    
    if cpu_info_out:
        for line in cpu_info_out:
            cpu_details_info = re.search('model name\\t*:(.*)', line)
            if cpu_details_info:
                cpu_details = cpu_details_info.group(1)

            cpu_MHz_info = re.search('cpu MHz\\t*:(.*)', line)
            if cpu_MHz_info:
                cpu_MHz = cpu_MHz_info.group(1)

    cores = shell.bash_execute("lscpu | grep -i '^CPU(s):'")
    cpu_cores = cores[0][1]

    if is_virtualized():
        cpu_MHz = cpu_MHz.split(".")[0]
        cpu_MHz = cpu_MHz + 'MHz'
        return [cpu_details, cpu_MHz, cpu_cores]

    if dmi_decode_out:
        # if dmidecode returns information then use the Max Frequency value
        r = re.compile('Current Speed:(.*?)MHz')
        for line in dmi_decode_out:
            frequency = r.search(line)
            if frequency:
                cpu_MHz = frequency.group(1).strip()
                if cpu_MHz != 'Unknown':
                    cpu_MHz = utils.format_hz(int(cpu_MHz) * 1024 * 1024)
                    return [cpu_details, cpu_MHz, cpu_cores]
        r = re.compile('Max Speed:(.*?)MHz')
        for line in dmi_decode_out:
            frequency = r.search(line)
            if frequency:
                cpu_MHz = frequency.group(1).strip()
                if cpu_MHz != 'Unknown':
                    cpu_MHz = utils.format_hz(int(cpu_MHz) * 1024 * 1024)
                    return [cpu_details, cpu_MHz, cpu_cores]

    return [cpu_details, cpu_MHz, cpu_cores]

def hardware_info():
    """
    Returns a OpenStruct object containing information about the system:

    info.cpu.model -- cpu model
    info.cpu.mhz   -- cpu mhz
    info.cpu.cores -- cpu cores

    info.disk.total      -- total disk space in KB
    info.disk.raidLevels -- disk supported raid levels
    info.disk.arrayTypes -- disk raid array types

    info.mem.total  -- total memory in KB
    """
    info = utils.OpenStruct()

    #cpu info
    cpu = utils.OpenStruct()
    (cpu.model, cpu.mhz, cpu.cores) = cpu_info()
    info.cpu = cpu

    # disk info
    disk = utils.OpenStruct()
    raid = raid_info()
    (disk.total, disk.supportedLevels, disk.arraysTypes) = (disk_info()[0], raid[0], raid[1])
    info.disk = disk

    # mem info
    mem = utils.OpenStruct()
    mem.total = mem_info()[0]
    info.mem = mem

    return info


def system_info():
    """
    Returns a OpenStruct object containing information about the system:

    info.os.version     -- OS version
    info.os.kernel      -- kernel version
    info.os.uptime      -- system uptime
    info.os.server_time -- system time
    info.os.last_update -- last OS update timestamp

    info.sysinv.inv_code -- System material/inventory code. 
                            This code is only applicable for 
                            avaya provided hardware.
    info.sysinv.model_info -- System Model information (vendor + product name)
                              only applicable for Avaya provided hardware.
    info.sysinv.serial_num -- System Serial Number/asset number
                              only applicable for Avaya provided machines

    info.disk.total -- total disk space in KB
    info.disk.used  -- used disk space in KB
    info.disk.free  -- free disk space in KB

    info.mem.total  -- total memory in KB
    info.mem.used   -- used memory in KB
    info.mem.free   -- free memory in KB

    info.cpu.usage   -- CPU usage history in %
    info.cpu.loadavg -- CPU load average
    """
    info = utils.OpenStruct()

    # OS release info
    os = utils.OpenStruct()
    (os.version, os.kernel, os.uptime, os.server_time, os.last_update, os.server_type) = os_info()
    info.os = os

    # System inventory info
    sysinv = utils.OpenStruct()
    (sysinv.inv_code, sysinv.model_info, sysinv.sys_serial_num, sysinv.sys_type) = sys_inv_info()
    info.sysinv = sysinv

    # disk usage
    disk = utils.OpenStruct()
    (disk.total, disk.used, disk.free) = disk_info()
    info.disk = disk

    # memory usage
    mem = utils.OpenStruct()
    (mem.total, mem.used, mem.free) = mem_info()
    info.mem = mem

    # quota usage
    quota = utils.OpenStruct()
    (quota.total, quota.used, quota.free, quota.threshold, quota.hdd_size) = quota_info()
    info.quota = quota

    # CPU usage history info
    cpu = utils.OpenStruct()
    cpu.usage = cpu_usage()

    # CPU load average info
    cpu.loadavg = cpu_loadavg()
    info.cpu = cpu

    return info


def mark_last_update():
    """
    Mark last OS update date
    """
    with open(OS_LAST_UPDATE_FILE, 'w') as f:
        f.write(utils.format_timestamp())



class Test(unittest.TestCase):

    def test_get_system_info(self):
        info = system_info()
        self.assertTrue(info.os)
        self.assertTrue(info.sysinv)
        self.assertTrue(info.disk)
        self.assertTrue(info.mem)
        self.assertTrue(info.cpu)

    def test_get_hardware_info(self):
        info = hardware_info()
        self.assertTrue(info.cpu)
        self.assertTrue(info.disk)
        self.assertTrue(info.mem)

    def test_uptime(self):
        days, hours, minutes, seconds = uptime()
        self.assertTrue(days or hours or minutes or seconds)


class TestSysInvInfo(unittest.TestCase):

    def setUp(self):
        global INV_SYS_INFO_FILE
        self.orig_inv_sys_info_file = INV_SYS_INFO_FILE
        INV_SYS_INFO_FILE = '/tmp/inv_sys_info_unit_test_file'

        self.expected_material_code = "000270391"
        self.expected_model_info = "Dell R210 II"
        self.expected_sys_serial_num = "BN1223445"
        self.expected_sys_type = 'Avaya IP Office'
        with open(INV_SYS_INFO_FILE, 'w') as outfile:
            outfile.write('MODEL_INFO=' + self.expected_model_info + '\n')
            outfile.write('SYS_SERIAL_NUM=' + self.expected_sys_serial_num + '\n')
            outfile.write('MATERIAL_CODE=' + self.expected_material_code + '\n')

    def tearDown(self):
        global INV_SYS_INFO_FILE
        if os.path.isfile(INV_SYS_INFO_FILE):
            shell.rm(INV_SYS_INFO_FILE)
        INV_SYS_INFO_FILE = self.orig_inv_sys_info_file

    def test_get_sys_inv_info(self):
        material_code, model_info,  sys_serial_number, sys_type = sys_inv_info()
        self.assertEqual(material_code, self.expected_material_code)
        self.assertEqual(model_info, self.expected_model_info)
        self.assertEqual(sys_serial_number, self.expected_sys_serial_num)
        self.assertEqual(sys_type, self.expected_sys_type)

    def test_defaults_sys_inv_info(self):
        shell.rm(INV_SYS_INFO_FILE)
        material_code, model_info,  sys_serial_number, sys_type = sys_inv_info()
        self.assertEqual(material_code, "")
        self.assertEqual(model_info, "")
        self.assertEqual(sys_serial_number, "")
        self.assertEqual(sys_type, 'unknown')


if __name__ == "__main__":
    unittest.main()
