# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Syslog parsing
"""

import os
import unittest
import re
import time
import web

from datetime import datetime

from core.common import configuration
from core.common import utils
from core.common import i18n
from core.system import shell

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

class LogEntry(object):
    """
    Syslog entry
    """
    def __init__(self, timestamp=None, host="localhost",
                 source=None, message="", type=None):
        self.timestamp = timestamp
        self.host = host
        self.source = source
        self.message = message
        self.type = type

class Parser(object):
    """
    Syslog parser
    """
    
    ENTRY_FORMAT               = re.compile("(.+>) (\w+\s+\w+\s+\d+ \d+:\d+:\d+) (.+?) (.+?): (.+)")
    DATE_FORMAT                = "%Y %b %d %H:%M:%S"
    MESSAGE_FORMAT             = re.compile("(.+)\| (\d+) (.+)")
    MESSAGE_TEXT_FORMAT        = re.compile("(.+?) (.+?) (.+)")
    SOURCE_FORMAT              = re.compile("(.+)(\[\d+\])")

    AUDIT_FACILITY             = [13]
    SECURITY_FACILITY          = [4, 10]
    OPERATIONAL_FACILITY       = [0, 1, 2, 3, 5, 6, 7, 8, 9, 11, 12, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23]

    def parse(self, filepath=None, keyword=None, tail_count=0, hosts=None, type=None, source=None):
        """
        Parse input syslog file.

        Args:

        filepath     -- syslog file path
        keyword      -- keyword to filter messages by
        tail_count   -- if > 0 then only last tail_count lines will be parsed
        host         -- host name to filter messages by
        type         -- syslog file type
        """

        logs = []

        if (hosts is None):
            filepath = "/var/log/messages"
            if not os.path.isfile(filepath):
                return logs
            if tail_count > 0:
                for line in shell.tail(filepath, tail_count):
                    self._parse_line(keyword, line, type, logs, source)
            else:
                with open(filepath) as f:
                    for line in f:
                        self._parse_line(keyword, line, type, logs, source)
        else:
            path = configuration.SHARED_CONFIGURATION['syslog']['path']
            if not os.path.isdir(path):
                return logs
            for host in hosts:
                hostpath = path + host + "/"
                if os.path.isdir(hostpath):
                    dates = listDirectories(hostpath)
                    for date in dates:
                        filepath = hostpath + date + "/"
                        if os.path.isdir(filepath):
                            files = listFiles(filepath)
                            for filename in files:
                                if web.ctx.session.security_rights is False and "_security" in filename:
                                    continue
                                file = filepath + filename
                                if os.path.isfile(file):
                                    if tail_count > 0:
                                        for line in shell.tail(file, tail_count):
                                            self._parse_line(keyword, line, type, logs, source, filename)
                                    else:
                                        with open(file) as f:
                                            for line in f:
                                                self._parse_line(keyword, line, type, logs, source, filename)

            logs.sort(key=lambda entry: entry.timestamp, reverse=True)

            if tail_count:
                tail_count = int(tail_count)
                logs = logs[:tail_count]

        return logs


    def _parse_line(self, keyword, line, type, logs, source, file=None):
        """
        Parse syslog line and append it to log entry list.
        """
        decoded_line = line.decode('iso-8859-1')
        if keyword:
            for key in keyword.split(","):
                if key in decoded_line:
                    entry = self._parse_entry(decoded_line, file)
                    if entry:
                        logs.append(entry)
        elif type:
            entry = self._parse_entry(decoded_line, file)
            if entry and str(type) == entry.type:
                if entry and entry.type == type:
                    if source:
                        if entry.source == source:
                            logs.append(entry)
                        elif source == configuration.SHARED_CONFIGURATION['syslog']['system_tag_value'] and file:
                            if file in configuration.SHARED_CONFIGURATION['syslog'].as_list('audit_files') \
                                or file in configuration.SHARED_CONFIGURATION['syslog'].as_list('security_files') \
                                or file in configuration.SHARED_CONFIGURATION['syslog'].as_list('operational_files'):
                                logs.append(entry)
                    else:
                        logs.append(entry)
        elif source:
            if str(source) in decoded_line \
                or (source == configuration.SHARED_CONFIGURATION['syslog']['system_tag_value'] and file):
                entry = self._parse_entry(decoded_line, file)
                if entry and entry.source == source:
                    logs.append(entry)
                elif source == configuration.SHARED_CONFIGURATION['syslog']['system_tag_value'] and file:
                    if file in configuration.SHARED_CONFIGURATION['syslog'].as_list('audit_files')\
                       or file in configuration.SHARED_CONFIGURATION['syslog'].as_list('security_files')\
                    or file in configuration.SHARED_CONFIGURATION['syslog'].as_list('operational_files'):
                        if entry:
                            logs.append(entry)
        else:
            entry = self._parse_entry(decoded_line, file)
            if entry:
                logs.append(entry)

    def _parse_entry(self, line, file=None):
        """
        Parse a single log entry from line
        This method returns None if the line does not match ENTRY_FORMAT.
        """
        now = datetime.now()
        match = self.ENTRY_FORMAT.match(line)

        if match:
            groups = match.groups()
            entry = LogEntry()

            dt = datetime.strptime(groups[1], self.DATE_FORMAT)
            entry.pri = None
            try:
                entry.pri = int(groups[0].replace("<","").replace(">",""))
            except: pass
            entry.timestamp = utils.format_timestamp(time.mktime(dt.timetuple()))
            entry.host = groups[2].split()[0]
            entry.source = groups[3]
            match_source = self.SOURCE_FORMAT.match(entry.source)
            if (match_source):
                entry.source = match_source.group(1)
            entry.message = groups[4]
            entry.type = ""

            match_message = self.MESSAGE_FORMAT.match(entry.message)
            if (match_message):
                g = match_message.groups()
                msg_type = g[1].strip()
                entry.message = g[2].strip()
                if (msg_type != "0"):
                    match_message2 = self.MESSAGE_TEXT_FORMAT.match(entry.message)
                    if (match_message2):
                        g2 = match_message2.groups()
                        entry.type = g2[1].strip()
                        entry.message = g2[2].strip()
                    else:
                        entry.type = self.get_type_from_pri(entry.pri)
                else:
                    entry.type = self.get_type_from_pri(entry.pri)

            if file:
                if file in configuration.SHARED_CONFIGURATION['syslog'].as_list('audit_files'):
                    entry.type = configuration.SHARED_CONFIGURATION['syslog']['audit_type']
                    entry.source = i18n.custom_gettext(configuration.SHARED_CONFIGURATION['syslog']['system_tag_display'])
                elif file in configuration.SHARED_CONFIGURATION['syslog'].as_list('security_files'):
                    entry.type = configuration.SHARED_CONFIGURATION['syslog']['security_type']
                    entry.source = i18n.custom_gettext(configuration.SHARED_CONFIGURATION['syslog']['system_tag_display'])
                elif file in configuration.SHARED_CONFIGURATION['syslog'].as_list('operational_files'):
                    entry.type = configuration.SHARED_CONFIGURATION['syslog']['operational_type']
                    entry.source = i18n.custom_gettext(configuration.SHARED_CONFIGURATION['syslog']['system_tag_display'])

            return entry


    def get_type_from_pri(self, pri=None):
        """
        Parse the PRI value to
        """
        if pri:
            facility = pri // 8
            severity = pri % 8
            if severity == 7:
                # Debug type not yet supported
                return ""
            else:
                if facility in self.AUDIT_FACILITY:
                    return configuration.SHARED_CONFIGURATION['syslog']['audit_type']
                elif facility in self.SECURITY_FACILITY:
                    return configuration.SHARED_CONFIGURATION['syslog']['security_type']
                elif facility in self.OPERATIONAL_FACILITY:
                    return configuration.SHARED_CONFIGURATION['syslog']['operational_type']
        return ""

def listDirectories(path = configuration.SHARED_CONFIGURATION['syslog']['path']):
    hosts = []
    if os.path.isdir(path):
        for o in os.listdir(path):
            if os.path.isdir(os.path.join(path, o)):
                hosts.append(o)
    return hosts


def listFiles(path = configuration.SHARED_CONFIGURATION['syslog']['path']):
    files = []
    if os.path.isdir(path):
        for o in os.listdir(path):
            if os.path.isfile(os.path.join(path, o)):
                files.append(o)
    return files


class Test(unittest.TestCase):

    SYSLOG_FILE = "/var/log/messages"

    def setUp(self):
        self.parser = Parser()

    def test_parse(self):
        entries = self.parser.parse(self.SYSLOG_FILE)
        self.assertTrue(entries)

    def test_tail_parse(self):
        entries = self.parser.parse(self.SYSLOG_FILE, tail_count=2)
        self.assertEqual(len(entries), 2)

if __name__ == "__main__":
    unittest.main()
