# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Provides system date and time management.
"""

import unittest
import time
from . import shell
import re
import os

from core.common import configuration
from core.common import log

__author__ = "Avaya Inc."
__copyright__ = "Copyright 2010, Avaya Inc."

# default timezone database file on a CentOS system
TZ_DATABASE = '/usr/share/zoneinfo/zone.tab'

# timezone configuration file used by older applications
TZ_CONFIG_FILE = '/etc/adjtime'

# configuration file for NTP
NTP_CONFIG_FILE = '/etc/ntp.conf'

# Sync-at-startup configuration file for NTP
NTP_SYNC_FILE = '/etc/ntp/step-tickers'


def date():
    """
    Returns current local time as POSIX timestamp
    """
    return int(time.time())


def set_date(timestamp):
    """
    Sets system date.

    timestamp -- POSIX timestamp
    """
    ts = time.localtime(timestamp)
    year = str(ts.tm_year)
    month = str(ts.tm_mon).rjust(2, '0')
    month_day = str(ts.tm_mday).rjust(2, '0')
    hour = str(ts.tm_hour).rjust(2, '0')
    min = str(ts.tm_min).rjust(2, '0')

    # date command expects time in MMDDhhmmYYYY
    shell.sudo_call("date %s%s%s%s%s" % (month, month_day, hour, min, year))

    # set date to hw clock
    shell.sudo_call("hwclock --systohc")


class Timezone(object):
    """
    Represents time zone details.
    """

    def __init__(self, code, name, comments=None):
        """
        code        -- country code
        name        -- timezone name (used in TZ env variable)
        comments    -- extra info about time zone
        """
        self.code = code
        self.name = name
        self.comments = comments

    def __eq__(self, other):
        if other and isinstance(other, Timezone):
            return self.code == other.code and self.name == other.name
        return False


class TimezoneManager(object):
    """
    Manages timezone settings for CentOS.
    """

    def __init__(self, config=configuration.SHARED_CONFIGURATION):
        """
        config -- app configuration data
        """
        self.get_timezone_script = config['systime']['get_timezone_script']
        self.set_timezone_script = config['systime']['set_timezone_script']
        # load and cache available timezones
        self.TIMEZONES = self.load_timezones(TZ_DATABASE)
        self.DEFAULT_TIMEZONE_NAME = "Europe/London"

    def load_timezones(self, tzdb):
        """
        Loads available timezones from timezone database file.

        The database is a flat text file with following format:

        country_code coordinates timezone_name comments

        all lines starting with # are considered comments
        """
        zones = {}
        with shell.file_open(tzdb) as file:
            for line in file:
                if not line.startswith("#"): # skip comments
                    tokens = line.split()
                    code = tokens[0]
                    name = tokens[2]
                    comments = []
                    if len(tokens) > 3:
                        comments = tokens[3:]
                    tz = Timezone(code, name, ''.join(comments))
                    zones[tz.name] = tz
        return zones

    def get_timezone(self):
        """
        Return system timezone name
        """
        tzname = shell.sudo_execute(self.get_timezone_script, parse=shell.RAW).decode().strip()
        if tzname in self.TIMEZONES:
            return self.TIMEZONES[tzname]
        elif tzname.replace(" ", "_") in self.TIMEZONES:
            return self.TIMEZONES[tzname.replace(" ", "_")]
        else:
            log.SHARED_LOGGER.error("Invalid timezone <%s>. Returning default timezone." % tzname)
            return self.TIMEZONES[self.DEFAULT_TIMEZONE_NAME]

    def set_timezone(self, tzname):
        """
        Set system timezone

        tz -- timezone name
        """
        shell.sudo_call('%s %s' % (self.set_timezone_script, tzname))

    def get_utc(self):
        """
        Indicates if internal clock is set to UTC
        """
        with shell.file_open(TZ_CONFIG_FILE) as f:
            lines = f.readlines()
        for line in lines:
            if line.lstrip().startswith('UTC'):
                return True 
        return False

    def set_utc(self, useUTC):
        """
        Set internal clock to UTC
        """
        utc_found = False
        with shell.file_open(TZ_CONFIG_FILE) as f:
            lines = f.readlines()
        for i, line in enumerate(lines):
            if line.lstrip().startswith('UTC') or line.lstrip().startswith('LOCAL'):
                lines[i] = ('UTC' if useUTC else 'LOCAL') + "\n"
                utc_found = True
                break
        if not utc_found:
            lines.append(('UTC' if useUTC else 'LOCAL') + "\n")
        with shell.file_open(TZ_CONFIG_FILE, 'w') as f:
            f.writelines(lines)


class NTPManager(object):
    """
    Manages NTP configuration for CentOS.
    """
    def __init__(self, config=configuration.SHARED_CONFIGURATION):
        """
        config -- app configuration data
        """
        self.get_ntp_script = config['systime']['get_ntp_script']
        self.start_ntp_script = config['systime']['start_ntp_script']
        self.stop_ntp_script = config['systime']['stop_ntp_script']
        self.is_ova = os.path.isfile(configuration.SHARED_CONFIGURATION['virtualization']['virtualization_file'])

    def get_ntp(self):
        """
        Returns current NTP configuration.
        """
        ntp_config = {'use_ntp': False,
                      'servers': [],
                      'use_local_source': False,
                      'sync': False}

        output = shell.sudo_execute(self.get_ntp_script, parse=shell.LINES)
        for line in output:
            if line.startswith("use_ntp"):
                if line.split("=")[1].strip() == "True":
                    ntp_config["use_ntp"] = True
            if line.startswith("servers"):
                ntp_config["servers"] = line.split("=")[1].strip().split(" ")
            if line.startswith("use_local_source"):
                if line.split("=")[1].strip() == "True":
                    ntp_config["use_local_source"] = True

        ntp_config['sync'] = bool(shell.sudo_execute('cat %s' % NTP_SYNC_FILE))

        return ntp_config

    def set_ntp(self, ntp_config):
        """
        Sets new NTP configuration.
        """
        # setup sync
        with shell.file_open(NTP_SYNC_FILE, 'w') as sfile:
            if ntp_config['sync']:
                sfile.writelines([server + '\n' for server in ntp_config['servers']])
                if self.is_ova is False:
                    sfile.write('127.127.1.0\n')
            else:
                sfile.truncate(0)

        if ntp_config['use_local_source']:
            if self.is_ova is False:
                ntp_config['servers'].extend(['127.127.1.0'])
        ntp_servers = ' '.join(ntp_config['servers'])

        # setup ntpd service
        if ntp_config['use_ntp']:
            shell.sudo_call('%s %s' % (self.start_ntp_script, ntp_servers))
        else:
            shell.sudo_call(self.stop_ntp_script)


class TestDate(unittest.TestCase):

    def setUp(self):
        self.timezone_manager = TimezoneManager()
        self.ntp_manager = NTPManager()

    def test_get_date(self):
        self.assertTrue(date())

    def test_get_ntp(self):
        self.assertTrue(self.ntp_manager.get_ntp())

    def test_set_ntp(self):
        old_ntp = self.ntp_manager.get_ntp()
        new_ntp = {'use_ntp': not old_ntp['use_ntp'],
                   'servers': ['0.ntp.centos.com', '1.ntp.centos.com'],
                   'use_local_source': not old_ntp['use_local_source'],
                   'sync': False}
        self.ntp_manager.set_ntp(new_ntp)
        self.assertEqual(self.ntp_manager.get_ntp(), new_ntp)
        self.ntp_manager.set_ntp(old_ntp)


class TestTimezoneManager(unittest.TestCase):

    def setUp(self):
        self.timezone_manager = TimezoneManager()
        self.original_timezone = self.timezone_manager.get_timezone()

    def tearDown(self):
        self.timezone_manager.set_timezone(self.original_timezone)

    def test_cached_timezone_database(self):
        self.assertTrue(self.timezone_manager.TIMEZONES)

    def test_get_timezone(self):
        self.assertTrue(self.timezone_manager.get_timezone())

    def test_set_timezone(self):
        expected_timezone = self.timezone_manager.TIMEZONES['Europe/Bucharest']
        self.timezone_manager.set_timezone(expected_timezone)
        self.assertEqual(self.timezone_manager.get_timezone(), expected_timezone)


class TestNTPManager(unittest.TestCase):

    def setUp(self):
        self.ntp_manager = NTPManager()
        self.original_ntp_config = self.ntp_manager.get_ntp()

    def tearDown(self):
        self.ntp_manager.set_ntp(self.original_ntp_config)

    def test_get_ntp(self):
        self.assertTrue(self.ntp_manager.get_ntp())

    def test_set_ntp(self):
        expected_ntp_config = {'use_ntp': not self.original_ntp_config['use_ntp'],
                               'servers': ['0.ntp.centos.com', '1.ntp.centos.com'],
                               'use_local_source': not self.original_ntp_config['use_local_source'],
                               'sync': False}
        self.ntp_manager.set_ntp(expected_ntp_config)
        self.assertEqual(self.ntp_manager.get_ntp(), expected_ntp_config)


if __name__ == "__main__":
    unittest.main()
