# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
YUM utility module
"""

import unittest

from core.system import shell
from core.common import i18n

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

# return codes for yum check-update
YUM_UPDATES_AVAILABLE=100
YUM_ERROR=1

class YumError(Exception):
    """
    Used for yum errors.
    """
    def __str__(self):
        return self.args[0]

def updates_available():
    """
    Return True if there are updates available for installed packages.
    Raise Error if yum check-update command fails.
    """
    code = shell.sudo_call('yum -q check-update')
    if code == YUM_UPDATES_AVAILABLE:
        return True
    elif code == YUM_ERROR:
        raise YumError(i18n.custom_gettext("Could not connect to system repository"))
    return False

def list_updates():
    """
    Return a list of packages that need update: [(package_name1, new_version1),...].
    If there are no updates available the returned list is empty.
    """
    updates = []
    output = [line for line in shell.sudo_execute('yum -q check-update', parse=shell.LINES)
              if line.strip()]
    for line in output:
        tokens = line.split()
        # (name, new_version)
        if len(tokens) >= 2:
            updates.append((tokens[0], tokens[1]))
    return updates

class Test(unittest.TestCase):

    def test_list_updates(self):
        if updates_available():
            self.assertTrue(list_updates())

if __name__ == '__main__':
    unittest.main()
