# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
REST API for logs access.
"""
import os
import json
import threading
import web
import wsgiref.util
import mimetypes
import datetime

import audit
import services_rest_api

import core.logs.applications
import core.logs.archives

from core.common import utils
from core.common import log
from core.logs import login
from core.common.decorators import synchronized
from core.system import syslog

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

# Logging tag
TAG = 'log-rest-api'

app_log                 = core.logs.applications.AppLog(services_rest_api.services_manager)
log_archives_manager    = core.logs.archives.LogArchivesManager()

log_archive_lock = threading.Lock()

# block size for windows client file read
BLOCK_SIZE=16384


class AuditLogResource(object):
    """
    REST resource to read audit log entries

    Methods:

    GET -- return audit log entries
    """

    def GET(self):
        """
        Return a json string with audit log entries.

        Data format:

        [ { "timestamp":    "entry timestamp in YYYY-MM-DD HH:MM:SS format",
            "action":       "user action",
            "user":         "user name"},... ]
        """        
        web.header("Content-Type","application/json")        
        return json.dumps(audit.logger.log_history(), default=utils.encode_json)


class ApplicationsLogResource(object):
    """
    REST resource to read applications log

    Methods:

    GET -- return log entries for specified application 
    """

    def GET(self):
        """
        Return a json string with log entries for one or all application

        Data format:

        [ { "application":  "application name",
            "timestamp":    "entry timestamp",
            "message":      "log message",
            "level":        "log level" },... ]

        Input:

        id -- application ID

        If name is missing the all the applications are considered.
        """
        id = str(web.input(id='all').id)
        
        web.header("Content-Type","application/json")
        return json.dumps(app_log.log_history(id), default=utils.encode_json)


class DebugLogsResource(object):
    """
    REST resource to manage debug log files.
    
    Methods:

    GET     -- return JSON object with details for the core/debug files
    """
    def GET(self):
        """
        Returns a dictionary containing details about the available debug files.

        Data format:

        { "count": the number of debug files,
          "size":  the cumulated size of debug files
        }
        """
        web.header("Content-Type","application/json")
        return json.dumps(log_archives_manager.get_debug_files_info(), default=utils.encode_json)


class LoginLogsResource(object):
    """
    REST resource to manage login log file.

    Methods:

    GET     -- return JSON object
    """
    def GET(self):
        """
        Returns a dictionary containing details about the login log file.

        Data format:

        [ { "failed_login_attempts":        "failed login attempts",
            "last_successful_login":        "last successful login timestamp"]
        """
        web.header("Content-Type","application/json")
        return json.dumps(login.info())


class SyslogHostsResource(object):
    """
    REST resource to get syslog hosts directories.

    Methods:

    GET     -- return JSON object
    """
    def GET(self):
        """
        Returns a dictionary containing the syslog hosts directories.

        Data format:

        { "hosts":        "host1, host2, ..." }
        """
        web.header("Content-Type","application/json")
        return json.dumps({'hosts' : syslog.listDirectories()})


class LogArchivesResource(object):
    """
    REST resource to manage log archives.

    Methods:

    GET     -- return JSON object with details about the log archives
    PUT     -- create a new log archive
    """
    def GET(self, id, name):
        """
        Returns a list of details about log archives or statically serve an archive file.

        Data format:

        [ { "modif":    "timestamp in YYYY-MM-DD HH:MM:SS",
            "name":     "file basename",
            "size":     "file size in human readable format"},... ]

        Input:

        id -- log archive type ('debug' or 'logs')

        Error codes:

        404 -- invalid archive type or missing file
        """
        if not id:
            archive_list = {'debug': log_archives_manager.get_archives('debug'),
                            'logs': log_archives_manager.get_archives('logs')}
            web.header("Content-Type","application/json")
            return json.dumps(archive_list, default=utils.encode_json)
        
        if id in ['debug', 'logs']:
            if name:
                try:
                    mime_type = mimetypes.guess_type(name)[0]
                    web.header('Content-Type', '%s' % mime_type)
                    web.header('Content-Disposition', 'attachment; filename=%s' % name)
                    archive_file = os.path.join(log_archives_manager.archives[id].dir, os.path.basename(name))
                    stat = os.stat(archive_file)
                    web.header('Content-Length', '%s' % stat.st_size)
                    web.header('Last-Modified', '%s' %
                        web.http.lastmodified(datetime.datetime.fromtimestamp(stat.st_mtime)))
                    if web.ctx.protocol.lower() == 'https':
                        # add headers to fix issue with IE download over SSL failing when "no-cache" header set
                        web.header('Pragma', 'private')
                        web.header('Cache-Control', 'private,must-revalidate')
                    return wsgiref.util.FileWrapper(open(archive_file, 'rb'), BLOCK_SIZE)
                except (OSError, IOError):
                    log.exception(TAG, "unable to serve archive <%s>" % name)
                    web.ctx.status = '404'
            else:
                web.header("Content-Type","application/json")
                return json.dumps(log_archives_manager.get_archives(id), default=utils.encode_json)
        else:
            log.error(TAG, "invalid archive type <%s>" % id)
            web.ctx.status = '404'

    @synchronized(log_archive_lock)
    def PUT(self, id, name):
        """
        Create a new archive from log/debug files

        Error codes:

        404 -- invalid archive type
        """
        if id in ['debug', 'logs']:
            log_archives_manager.create_archive(id)
        else:
            log.error(TAG, "invalid archive type <%s>" % id)
            web.ctx.status = '404'

    @synchronized(log_archive_lock)
    def DELETE(self, id, name):
        """
        Delete archive files
        
        Error codes:

        404 -- invalid archive type
        """
        if id in ['debug', 'logs']:
            if name:
                log_archives_manager.delete_files(id, name.split(';'))
            else:
                log_archives_manager.delete_files(id, None)
        else:
            log.error(TAG, "invalid archive type <%s>" % id)
            web.ctx.status = '404'
