# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Sub application for REST API.
"""
import sys

import web
import json

import audit

import services_rest_api
import system_rest_api
import settings_rest_api
import updates_rest_api
import i18n_rest_api
import log_rest_api
import vendor_rest_api

from core.auth import pam
from core.logs import login
from core.common import configuration
from core.common import i18n

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."


urls = ('/session',                             'SessionResource',
        '/i18n/lang',                           i18n_rest_api.LanguageResource,
        '/i18n/langs',                          i18n_rest_api.LanguagesResource,
        '/system/changepass',                   system_rest_api.ChangePasswordResource,
        '/system/change_root_pass',             system_rest_api.ChangeRootPasswordResource,
        '/system/password_rules',               system_rest_api.PasswordRulesResource,
        '/system/encrypt_pass',                 system_rest_api.EncryptPasswordResource,
        '/system/status',                       system_rest_api.StatusResource,
        '/system/syslog',                       system_rest_api.SyslogResource,
        '/system/check_hostname',               system_rest_api.CheckHostnameResource,
        '/system/network',                      system_rest_api.NetworkConfigurationResource,
        '/system/network/subinterface',         system_rest_api.SubinterfaceConfigurationResource,
        '/system/network/port',                 system_rest_api.CheckPortResource,
        '/system/date',                         system_rest_api.DateResource,
        '/system/date/standard',                system_rest_api.StandardDateResource,
        '/system/timezone',                     system_rest_api.TimezoneResource,
        '/system/timezones',                    system_rest_api.TimezonesResource,
        '/system/utc',                          system_rest_api.UTCResource,
        '/system/ntp',                          system_rest_api.NTPResource,
        '/system/lan',                          system_rest_api.LANResource,
        '/system/banner',                       system_rest_api.BannerResource,
        '/system/hwinfo',                       system_rest_api.HardwareInfo,
        '/system/newhw',                        system_rest_api.NewHardware,
        '/system/firewall',                     system_rest_api.Firewall,
        '/system/httpserver',                   system_rest_api.HTTPServerResource,
        '/system/vnc',                          system_rest_api.VNC,
        '/system/virtual',                      system_rest_api.VirtualSystemResource,
        '/system/autoconfig',                   system_rest_api.AutoconfigStatusResource,
        '/system/gold_edition',                 system_rest_api.GoldEditionResource,
        '/system/flag',                         system_rest_api.FlagResource,
        '/services',                            services_rest_api.ServicesResource,
        '/services/license',                    services_rest_api.LicenseResource,
        '/services/ws',                         services_rest_api.WebservicesResource,
        '/services/restart',                    services_rest_api.RestartSelectedServicesResource,
        '/services/dependency',                 services_rest_api.DependentServicesResource,
        '/service',                             services_rest_api.ServiceResource,
        '/service/run_at_startup',              services_rest_api.RunAtStartupResource,
        '/service/performance',                 services_rest_api.PerformanceDataResource,
        '/version',                             services_rest_api.VersionResource,
        '/settings/logs',                       settings_rest_api.LogSettingsResource,
        '/settings/syslog',                     settings_rest_api.SyslogSettingsResource,
        '/settings/snmp',                       settings_rest_api.SNMPSettingsResource,
        '/settings/webcontrol',                 settings_rest_api.WebappSettingsResource,
        '/settings/certificate',                settings_rest_api.CertificateResource,
        '/settings/certificate/ca/?(.*)',       settings_rest_api.CaResource,
        '/settings/certificate/generate/?(.*)', settings_rest_api.GenerateCertResource,
        '/settings/certificate/check_distrib',  settings_rest_api.CheckCertDistribInProgress,
        '/settings/repo/(.*)',                  settings_rest_api.RepositoryResource,
        '/settings/backup/(.*)',                settings_rest_api.BackupResource,
        '/settings/restore/(.*)',               settings_rest_api.RestoreResource,
        '/settings/status/restore',             settings_rest_api.RestoreResourceStatus,
        '/settings/mibs/(.*)',                  settings_rest_api.MIBResource,
        '/settings/debug',                      settings_rest_api.DebugSettingsResource,
        '/settings/tcpdumps',                   settings_rest_api.TcpDumpsSettingsResource,
        '/settings/asg',                        settings_rest_api.AsgResource,
        '/settings/asg_status',                 settings_rest_api.AsgStatusResource,
        '/settings/partition_size',             settings_rest_api.PartitionSizeResource,
        '/settings/csipo',                      settings_rest_api.CSIPOResource,
        '/settings/csipo/debug',                settings_rest_api.DebugCsipoResource,
        '/settings/ira/debug',                settings_rest_api.DebugIraResource,
        '/log/audit',                           log_rest_api.AuditLogResource,
        '/log/application',                     log_rest_api.ApplicationsLogResource,
        '/log/debug',                           log_rest_api.DebugLogsResource,
        '/log/login',                           log_rest_api.LoginLogsResource,
        '/log/syslog/hosts',                    log_rest_api.SyslogHostsResource,
        '/log/archive/([a-z]*)/?(.*)',          log_rest_api.LogArchivesResource,
        '/windows/clients/(.*)',                updates_rest_api.WindowsClientsResource,
        '/linux/downloads/?(.*)',               updates_rest_api.LinuxDownloadsResource,
        '/linux/system/downloads/?(.*)',        updates_rest_api.LinuxSystemDownloadsResource,
        '/updates/applications',                updates_rest_api.AppUpdatesResource,
        '/updates/applications/all',            updates_rest_api.AllAppUpdateResource,
        '/updates/applications/all/status',     updates_rest_api.AllAppUpdateStatusResource,
        '/updates/os',                          updates_rest_api.OSUpdatesResource,
        '/updates/os/status',                   updates_rest_api.OSUpdateStatusResource,
        '/updates/cache',                       updates_rest_api.ClearCacheResource,
        '/updates/available',                   updates_rest_api.UpdatesAvailableResource,
        '/updates/dependencies',                updates_rest_api.ServiceDependenciesResource,
        '/vendor/system_info',                  vendor_rest_api.VendorSystemInfoResource,
        '/vendor_tabs/(.*)',                    vendor_rest_api.VendorTabsResource)

# version string, language settings, authentication and change password REST API calls
# must be available in an unsecured (not authenticated) context because they are used
# in the context of the login page
UNSECURED = { 'GET': ('/version', '/i18n/lang', '/i18n/langs', '/system/password_rules', '/system/banner',
                      '/system/virtual', '/system/autoconfig'),
              'PUT': ('/session', '/system/changepass', '/i18n/lang'),
              'POST': (),
              'DELETE': ('/session')}

APP = web.application(urls, locals())
URL = '/api'
LOAD_ORDER = -sys.maxsize    # must be loaded first

def auth_processor(handle):
    """
    If the user is not authenticated the
    client will be redirected to login page.
    """
    if web.ctx.path in UNSECURED[web.ctx.method.upper()]:
        return handle()
    else:
        if not web.ctx.session.get('user'):
            web.ctx.status = '401'
        else:
            return handle()

APP.add_processor(auth_processor)


class DefaultAuthenticator(object):
    def get_referrer(self):
        return None

    def authenticate(self, client_data):
        try:
            user = client_data['username']
            passwd = client_data['password']
            if user in configuration.SHARED_CONFIGURATION['webapp'].as_list('users') and pam.authenticate(user, passwd):
                web.ctx.session.user = user
                web.ctx.session.password = passwd
                auth_ok = True
            else:
                auth_ok = False
        except KeyError:
            auth_ok = False

        if auth_ok:
            audit.log(i18n.custom_gettext('logged in'))
            web.ctx.session.admin_rights = True
            if client_data['username'] == "root":
                web.ctx.session.security_rights = True
            login.mark_success()
            web.ctx.status = '200'
        else:
            login.mark_fail()
            web.ctx.status = '401'

if not configuration.get_authenticator():
    configuration.set_authenticator(DefaultAuthenticator())

class SessionResource(object):
    """
    REST resource to handle authentication requests.

    Methods:

    PUT     -- authenticate and create session
    DELETE  -- logout, destroy session
    GET     -- return json string with authenticate user data
    """
#    @configuration.csrf_protected 
    def PUT(self):
        """
        Authenticate and create session.

        Input:

        username = <user name>
        password = <user password>

        Return:

        401 -- invalid username, password combination
        200 -- OK
        """
        return configuration.get_authenticator().authenticate(web.input())

    def DELETE(self):
        """
        Destroy session, logout.
        """
        try:
            audit.log(i18n.custom_gettext('logged out'))
        except Exception as e:
            pass
        if hasattr(web.ctx, 'session'):
            web.ctx.session.kill()

    def GET(self):
        """
        Return json string with session data:

            {'user': current-logged-in-user}
        """
        web.header("Content-Type","application/json")                
        return json.dumps({'user': web.ctx.session.get("user"),
                           'admin_rights': web.ctx.session.get("admin_rights"),
                           'security_rights': web.ctx.session.get("security_rights"),
                            'ip': web.ctx.host})

