# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
REST API for services management.
"""

import web
import json
import threading
import time

import audit

import core.services.manager
import core.services.monitor

from core.services import license
from core.services import ws
from core.common import utils
from core.common import version
from core.common import log
from core.common import i18n
from core.system import shell
from core.common.decorators import synchronized

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

services_manager = core.services.manager.ServicesManager()
services_monitor = core.services.monitor.PerformanceMonitor(services_manager)

services_monitor.start()

services_lock = threading.Lock()
license_lock  = threading.Lock()

class ServicesResource(object):
    """
    REST resource to manage all services at once.

    Methods:

    GET     -- return json string with registered services and their running information
    PUT     -- start/stop all services
    """

    @synchronized(services_lock)
    def GET(self):
        """
        Return json string with services info.

        Data format:

        { "service_id_1": {
             "id":                      "internal ID"
             "run_at_startup":          true|false,
             "display_name":            "display name in UI",
             "package_name":            "RPM package name",
             "service_name":            "/etc/init.d service name",
             "binary_names":            "executable name list",
             "control_supported":       true|false,
             "restore_supported":       true|false,
             "backup_supported":        true|false,
             "version":                 "version string",
             "run_info": {
                "mem":      integer - memory usage in Kb,
                "state":    enum    - 0 = starting, 1 = running, 2 = stopping, 3 = stopped, 4 = unknown
                "cpu":      float   - cpu usage,
                "uptime":   "service uptime value as returned by ps etime"
              }
             },

          "service_id_2": { ... }
         }
        """
        web.header("Content-Type","application/json")
        services_manager.update_run_info()
        services_manager.update_services_version()
        return json.dumps(services_manager.services, default=utils.encode_json)

    @synchronized(services_lock)
    def PUT(self):
        """
        Start or stop all services.

        Input:

        action = start|stop

        Error codes:

        400 -- invalid action
        """
        action = web.input(action="").action
        if action == "start":
            services_manager.start_all()
            audit.log(i18n.custom_gettext("start all services"))
        elif action == "stop":
            services_manager.stop_all()
            audit.log(i18n.custom_gettext("stop all services"))
        else:
            log.SHARED_LOGGER.error('invalid services action %s' % action)
            web.ctx.status = '400'
        

class ServiceResource(object):
    """
    REST resource to manage an individual service

    Methods:

    GET     -- return json string with service information and running state
    PUT     -- start/stop service
    """

    @synchronized(services_lock)
    def GET(self):
        """
        Return a json string with service information

        Data format:

        {    "id":                      "internal ID"
             "run_at_startup":          true|false,
             "display_name":            "display name in UI",
             "package_name":            "RPM package name",
             "service_name":            "/etc/init.d service name",
             "process_name":            "executable name",
             "control_supported":       true|false,
             "restore_supported":       true|false,
             "backup_supported":        true|false,
             "version":                 "version string",
             "run_info": {
                "mem":      integer - memory usage in Kb,
                "state":    enum    - 0 = starting, 1 = running, 2 = stopping, 3 = stopped, 4 = unknown
                "cpu":      float   - cpu usage,
                "uptime":   "service uptime value as returned by ps etime"
              }
        }

        Input:

        name = <service ID>

        Error codes:

        404 -- unknown service
        """
        try:
            web.header("Content-Type","application/json")
            name = web.input(name="").name
            service = services_manager[name]
            service.update_run_info()
            return json.dumps(service, default=utils.encode_json)
        except KeyError:
            log.SHARED_LOGGER.error('invalid service name %s' % name)
            web.ctx.status = '404'

    @synchronized(services_lock)
    def PUT(self):
        """
        Start service

        Input:

        name    = <service ID>
        action  = start|stop|forcestop

        Error codes:

        404 -- unknown service
        400 -- unknown action
        """
        try:
            name = web.input(name="").name
            service = services_manager[name]
            action = web.input(action="").action
            if action == "start":
                service.start()
                audit.log(i18n.custom_gettext("start %s service") % service.display_name)
            elif action == "stop":
                service.stop()
                audit.log(i18n.custom_gettext("stop %s service") % service.display_name)
            elif action == "forcestop":
                service.force_stop()
                audit.log(i18n.custom_gettext("force stop %s service") % service.display_name)
            else:
                log.SHARED_LOGGER.error('invalid service action %s' % action)
                web.ctx.status = '400'
        except KeyError:
            log.SHARED_LOGGER.error('invalid service name %s' % name)
            web.ctx.status = '404'

class RunAtStartupResource(object):
    """
    REST resource to manage run at startup state for a service

    Methods:

    GET     -- return run at startup state for a service
    PUT     -- configure service to run at startup
    """

    @synchronized(services_lock)
    def GET(self):
        """
        Return run at startup state for service: true|false.

        Input:

        name = <service ID>

        Error codes:

        404 -- unknown service
        """
        try:
            web.header("Content-Type","application/json")
            name = web.input(name="").name
            service = services_manager[name]
            service.update_run_info()
            return json.dumps(service.run_at_startup)
        except KeyError:
            log.SHARED_LOGGER.error('invalid service name %s' % name)
            web.ctx.status = '404'

    @synchronized(services_lock)
    def PUT(self):
        """
        Configure a service to run at startup.

        Input:

        name            = <service ID>
        run_at_startup  = on|off

        Error codes:

        404 -- unknown service
        400 -- invalid value for run_at_startup flag
        """
        name = web.input(name="").name
        try:
            service = services_manager[name]
            run_at_startup = str(web.input(run_at_startup="off").run_at_startup)
            if run_at_startup in ("on", "off"):
                if run_at_startup == "on":
                    shell.enable_service(service.service_name)
                    audit.log(i18n.custom_gettext("change autostart state for %s to on") % service.display_name)
                else:
                    shell.disable_service(service.service_name)
                    audit.log(i18n.custom_gettext("change autostart state for %s to off") % service.display_name)
            else:
                web.ctx.status = '400'
        except KeyError:
            log.SHARED_LOGGER.error('invalid service name %s' % name)
            web.ctx.status = '404'

class RestartSelectedServicesResource(object):
    """
    REST resource that restarts the services selected by the user
    """
    @synchronized(services_lock)
    def PUT(self):
        """
        Input:
        - services -- services ids string , separated by comma .
                      If empty , the resource will restart all services
        """
        services_ids = web.input(services=None).services
        if services_ids:
            services_manager.restart_services(services_ids.split(","))
        else:
            services_manager.restart_all()

class DependentServicesResource(object):
    """
    Returns a list of running services that have a given attribute set to True.
    Used to display warnings in the interface when certain settings are changed.
    """
    def GET(self):
        """
        Input:
        - property -- the attribute to check. If none is provided , it will return all running services
        """
        property = web.input(property='').property
        affected_services = services_manager.dependent_services(property)
        web.header("Content-Type","application/json")
        return json.dumps(affected_services, default=utils.encode_json)

class VersionResource(object):
    """
    REST resource to read release version.

    Methods:

    GET -- return a json string with release version information
    """

    def GET(self):
        """
        Return a json string dictionary containing Web Control release version.

        Data format:

        { "RELEASE":         "release number",
          "VERSION":         "version number",
          "TITLE":           "application title" }

        TITLE is dinamically built according to release type: ISA or APPL.
        """
        web.header("Content-Type","application/json")
        return json.dumps(version.APP_VERSION)

class PerformanceDataResource(object):
    """
    REST resource to access services performance data.

    Methods:

    GET -- return a json string with performance data
    """

    def GET(self):
        """
        Return a json string with performance data for a single service or all services.

        Data format:

        { "max_samples": integer - number of collected performance samples,
          "samples": [
            { "timestamp": "YYYY-MM-DD HH:MM:SS",
              "cpu_usage": float   - cpu usage,
              "mem_usage": integer - memory usage in Kb
            },
            { "timestamp": ...,
              "cpu_usage": ...,
              "mem_usage": ...
            }
            ...
          ]
        }

        'samples' array will be empty if no performance data was yet collected for specified service.

        Input:

        name = <service ID>

        Error codes:

        404 -- service not found
        """
        web.header("Content-Type","application/json")
        name = web.input(name=None).name
        try:
            perf_data = services_monitor.get_performance_data(name)
            return json.dumps(perf_data, default=utils.encode_json)
        except KeyError:
            log.SHARED_LOGGER.error('invalid service name %s' % name)
            web.ctx.status = '404'

class LicenseResource(object):
    """
    REST resource to manage license.

    Methods:

    GET -- get license status
    PUT -- process user input
    """

    def GET(self):
        """
        Return a json string with license status.

        Data format:

        { "type":     "input|text",
          "message":  "license message" }

        If 'type' is 'input' then user must input demo
        password, otherwise is just a warning message.
        """
        web.header("Content-Type","application/json")
        return json.dumps(license.read_website_file(), default=utils.encode_json)

    def PUT(self):
        """
        Process user input.

        Input:

        message  = <user input>

        Error codes:

        400 -- no user input
        """

        message = web.input(message=None).message
        if message:
            audit.log(i18n.custom_gettext('input demo password'))
            license.write_to_password_pipe(message)
        else:
            web.ctx.status = '400'

class WebservicesResource(object):
    """
    REST resource to verify the status of web services.

    Methods:

    GET -- get web services status
    """

    def GET(self):
        """
        Return a json string with port and xml status.

        Data format:

        { "ws_ign":    "true|false",
          "ws_xml":    "true|false",
          "ws_port":   "true|false" }
        """
        web.header("Content-Type","application/json")
        return json.dumps(ws.xml_check(), default=utils.encode_json)
