# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
REST API for settings management
"""

import os
import time
import glob
import threading
import datetime

import web
import wsgiref.util
import mimetypes
import json

from configobj import ConfigObjError

import audit
import services_rest_api
import core.settings.backup_restore

from core.system import shell
from core.system import sshd
from core.system import network
from core.system import rpm
from core.system import sysinfo
from core.common import utils
from core.common import log
from core.common import i18n
from core.common import version
from core.common.decorators import synchronized
from core.common import configuration
from core.settings import applications
from core.settings import repositories
from core.settings import webapp

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

backup_manager = core.settings.backup_restore.BackupManager()
restore_manager = core.settings.backup_restore.restore_manager

log_settings_lock = threading.Lock()
syslog_settings_lock = threading.Lock()
webapp_settings_lock = threading.Lock()
certificate_settings_lock = threading.Lock()
ca_lock = threading.Lock()
snmp_settings_lock = threading.Lock()
repo_settings_lock = threading.Lock()
backup_settings_lock = threading.Lock()
restore_settings_lock = threading.Lock()
mib_files_lock = threading.Lock()
tcpdumps_settings_lock = threading.Lock()
debug_settings_lock = threading.Lock()
debug_csipo_lock = threading.Lock()
debug_ira_lock = threading.Lock()
asg_lock = threading.Lock()
root_partition_lock = threading.Lock()

TAG = "settings-rest-api"

def _init_webapp_config():
    try:
        webapp_config = webapp.read_settings()
        configuration.set_global_config(webapp.PORT_PARAM_NAME, int(webapp_config[webapp.PORT_PARAM_NAME]))
        configuration.set_global_config(webapp.PROTOCOL_PARAM_NAME, webapp_config[webapp.PROTOCOL_PARAM_NAME])
        configuration.set_global_config(webapp.SESSION_TIMEOUT_PARAM_NAME, int(webapp_config[webapp.SESSION_TIMEOUT_PARAM_NAME]))
    except (IOError, OSError, ValueError):
        log.exception(TAG, 'unable to read Web Control configuration from file')

_init_webapp_config()

class LogSettingsResource(object):
    """
    REST resource to manage applications log file.

    Methods:

    GET     -- return JSON object with settings related to applications logs
    PUT     -- update applications log settings
    """

    @synchronized(log_settings_lock)
    def GET(self):
        """
        Return JSON object with settings related to applications logs:
        
        age -- log files age (days)
        
        Error codes:
        
        400 -- Log files age setting couldn't be read; config files are missing or invalid.
        """
        web.header("Content-Type","application/json")        
        config = {}
        for name in configuration.SHARED_CONFIGURATION['logrotate']:
            config_section = configuration.SHARED_CONFIGURATION['logrotate'][name]
            log_config = config_section['log_config_file']
            log_rotate_sections = config_section.as_list('log_rotate_sections')
            config[log_config] = log_rotate_sections
        
        logs_age = applications.get_logs_age(config)
        if logs_age:
            log_settings = {'age': logs_age}
            return json.dumps(log_settings, default=utils.encode_json)
        else:
            log.error(TAG, 'unable to read log age')
            web.ctx.status = '400'

    @synchronized(log_settings_lock)
    def PUT(self):
        """
        Update applications log settings.

        Input:

        age = log files age (days)
        
        Error codes:
        
        400 -- Invalid input value for log files age
        """
        log_age = web.input(age="").age
        try:
            age = int(log_age)
            config = {}
            for name in configuration.SHARED_CONFIGURATION['logrotate']:
                config_section = configuration.SHARED_CONFIGURATION['logrotate'][name]
                log_config = config_section['log_config_file']
                log_rotate_sections = config_section.as_list('log_rotate_sections')
                config[log_config] = log_rotate_sections

            applications.set_logs_age(config, age)

            # SNMP configuration file is actually watchdog sysconfig file
            utils.update_config(configuration.SHARED_CONFIGURATION['snmp']['config_file'],
                        {'WDOG_LOGROTATE_AGE': str(age)})

            audit.log(i18n.custom_gettext('set log age to %s') % age)
        except ValueError:
            log.exception(TAG, 'unable to set log age')
            web.ctx.status = '400'



class SyslogSettingsResource(object):
    """
    REST resource to manage applications syslog file.

    Methods:

    GET     -- return JSON object with settings related to applications syslog
    PUT     -- update applications syslog settings
    """

    @synchronized(syslog_settings_lock)
    def GET(self):
        """
        Return JSON object with settings related to applications syslog. Format:
        {
            age: 5,
            max_size: 30
            ....
        }

        Error codes:

        400 -- Syslog files age and max_size settings couldn't be read; config files are missing or invalid.
        """
        web.header("Content-Type", "application/json")

        try:
            syslog_config_file = configuration.SHARED_CONFIGURATION['syslog']['syslog_config_file']
            syslog_settings = {'age': None,
                      'max_size': None,
                      'security_age': None,
                      'security_max_size': None,
                      'audit_age': None,
                      'audit_max_size': None,
                      'operational_age': None,
                      'operational_max_size': None,
                      'debug_age': None,
                      'debug_max_size': None,
                      'port_tcp': None,
                      'port_tls': None,
                      'port_udp': None,
                      'otherapp1_ip':None,
                      'otherapp1_port':None,
                      'otherapp1_protocol':None,
                      'otherapp2_ip':None,
                      'otherapp2_port':None,
                      'otherapp2_protocol':None,
                      'sources': None}
            utils.read_config(syslog_config_file, syslog_settings)
            return json.dumps(syslog_settings)
        except IOError:
            log.exception(TAG, 'unable to read syslog configuration file %s' % syslog_config_file)
            web.ctx.status = '400'

    @synchronized(syslog_settings_lock)
    def PUT(self):
        """
        Update applications syslog settings.

        Input:

        age                   -- Age of the general log files.
        max_size              -- Maximum size of the general log files.
        security_age          -- Age of the security log files.
        security_max_size     -- Maximum size of the security log files.
        audit_age             -- Age of the audit log files.
        audit_max_size        -- Maximum size of the audit log files.
        operational_age       -- Age of the operational log files.
        operational_max_size  -- Maximum size of the operational log files.
        debug_age             -- Age of the debug log files.
        debug_max_size        -- Maximum size of the debug log files.
        port_tcp              -- Receiver tcp port.
        port_tls              -- Receiver tls port.
        port_udp              -- Receiver udp port.
        send_protocol         -- Receiver protocol.
        otherapp1_ip          -- Forwarding destination 1 ip.
        otherapp1_port        -- Forwarding destination 1 port.
        otherapp1_protocol    -- Forwarding destination 1 protocol.
        otherapp2_ip          -- Forwarding destination 2 ip.
        otherapp2_port        -- Forwarding destination 2 port.
        otherapp2_protocol    -- Forwarding destination 2 protocol.
        sources               -- Selection of log sources

        Error codes:

        400 -- I/O error when writing syslog configuration
        """
        try:
            syslog_settings = {}

            age = web.input(age=None).age
            max_size = web.input(max_size=None).max_size
            security_age = web.input(security_age=None).security_age
            security_max_size = web.input(security_max_size=None).security_max_size
            audit_age = web.input(audit_age=None).audit_age
            audit_max_size = web.input(audit_max_size=None).audit_max_size
            operational_age = web.input(operational_age=None).operational_age
            operational_max_size = web.input(operational_max_size=None).operational_max_size
            debug_age = web.input(debug_age=None).debug_age
            debug_max_size = web.input(debug_max_size=None).debug_max_size
            port_tcp = web.input(port_tcp=None).port_tcp
            port_tls = web.input(port_tls=None).port_tls
            port_udp = web.input(port_udp=None).port_udp
            send_protocol = web.input(send_protocol=None).send_protocol
            otherapp1_ip = web.input(otherapp1_ip=None).otherapp1_ip
            otherapp1_port = web.input(otherapp1_port=None).otherapp1_port
            otherapp1_protocol = web.input(otherapp1_protocol=None).otherapp1_protocol
            otherapp2_ip = web.input(otherapp2_ip=None).otherapp2_ip
            otherapp2_port = web.input(otherapp2_port=None).otherapp2_port
            otherapp2_protocol = web.input(otherapp2_protocol=None).otherapp2_protocol
            sources = web.input(sources=None).sources

            if age:
                syslog_settings['age'] = age
            else:
                syslog_settings['age'] = configuration.SHARED_CONFIGURATION['syslog']['default_age']

            if max_size:
                syslog_settings['max_size'] = max_size
            else:
                syslog_settings['max_size'] = configuration.SHARED_CONFIGURATION['syslog']['default_max_size']

            if security_age:
                syslog_settings['security_age'] = security_age
            else:
                syslog_settings['security_age'] = configuration.SHARED_CONFIGURATION['syslog']['default_age']

            if security_max_size:
                syslog_settings['security_max_size'] = security_max_size
            else:
                syslog_settings['security_max_size'] = configuration.SHARED_CONFIGURATION['syslog']['default_max_size']

            if audit_age:
                syslog_settings['audit_age'] = audit_age
            else:
                syslog_settings['audit_age'] = configuration.SHARED_CONFIGURATION['syslog']['default_age']

            if audit_max_size:
                syslog_settings['audit_max_size'] = audit_max_size
            else:
                syslog_settings['audit_max_size'] = configuration.SHARED_CONFIGURATION['syslog']['default_max_size']

            if operational_age:
                syslog_settings['operational_age'] = operational_age
            else:
                syslog_settings['operational_age'] = configuration.SHARED_CONFIGURATION['syslog']['default_age']

            if operational_max_size:
                syslog_settings['operational_max_size'] = operational_max_size
            else:
                syslog_settings['operational_max_size'] = configuration.SHARED_CONFIGURATION['syslog']['default_max_size']

            if debug_age:
                syslog_settings['debug_age'] = debug_age
            else:
                syslog_settings['debug_age'] = configuration.SHARED_CONFIGURATION['syslog']['default_age']

            if debug_max_size:
                syslog_settings['debug_max_size'] = debug_max_size
            else:
                syslog_settings['debug_max_size'] = configuration.SHARED_CONFIGURATION['syslog']['default_max_size']

            if port_tcp and configuration.data_validate_type_length(port_tcp,configuration.DATA_PORT_VALIDATOR,configuration.DATALENGTH_PORT_VALIDATOR):
                syslog_settings['port_tcp'] = port_tcp
            else:
                syslog_settings['port_tcp'] = ""

            if port_tls and configuration.data_validate_type_length(port_tls,configuration.DATA_PORT_VALIDATOR,configuration.DATALENGTH_PORT_VALIDATOR):
                syslog_settings['port_tls'] = port_tls
            else:
                syslog_settings['port_tls'] = ""

            if port_udp and configuration.data_validate_type_length(port_udp,configuration.DATA_PORT_VALIDATOR,configuration.DATALENGTH_PORT_VALIDATOR):
                syslog_settings['port_udp'] = port_udp
            else:
                syslog_settings['port_udp'] = ""

            if send_protocol:
                syslog_settings['send_protocol'] = send_protocol
            else:
                syslog_settings['send_protocol'] = ""

            if otherapp1_ip and configuration.data_validate_type_length(otherapp1_ip,configuration.DATA_IP_VALIDATOR,configuration.DATALENGTH_IP_VALIDATOR):
                syslog_settings['otherapp1_ip'] = otherapp1_ip
            else:
                syslog_settings['otherapp1_ip'] = ""

            if otherapp1_port and configuration.data_validate_type_length(otherapp1_port,configuration.DATA_PORT_VALIDATOR,configuration.DATALENGTH_PORT_VALIDATOR):
                syslog_settings['otherapp1_port'] = otherapp1_port
            else:
                syslog_settings['otherapp1_port'] = ""

            if otherapp1_protocol:
                syslog_settings['otherapp1_protocol'] = otherapp1_protocol
            else:
                syslog_settings['otherapp1_protocol'] = ""

            if otherapp2_ip and configuration.data_validate_type_length(otherapp1_ip,configuration.DATA_IP_VALIDATOR,configuration.DATALENGTH_IP_VALIDATOR):
                syslog_settings['otherapp2_ip'] = otherapp2_ip
            else:
                syslog_settings['otherapp2_ip'] = ""

            if otherapp2_port  and configuration.data_validate_type_length(otherapp1_port,configuration.DATA_PORT_VALIDATOR,configuration.DATALENGTH_PORT_VALIDATOR):
                syslog_settings['otherapp2_port'] = otherapp2_port
            else:
                syslog_settings['otherapp2_port'] = ""

            if otherapp2_protocol:
                syslog_settings['otherapp2_protocol'] = otherapp2_protocol
            else:
                syslog_settings['otherapp2_protocol'] = ""

            if sources:
                syslog_settings['sources'] = sources
            else:
                syslog_settings['sources'] = ""

            syslog_config_file = configuration.SHARED_CONFIGURATION['syslog']['syslog_config_file']
            utils.update_config(syslog_config_file, syslog_settings)

            syslog_config_script = configuration.SHARED_CONFIGURATION['syslog']['syslog_config_script']
            shell.sudo_call('%s "Administrator" "%s"' % (syslog_config_script,str(web.ctx.session.password)))

        except (IOError, OSError):
            log.exception(TAG, 'unable to write syslog configuration file %s' % syslog_config_file)
            web.ctx.status = '400'


class CSIPOResource(object):
    """
    REST resource to manage CSIPO settings

    Methods:

    GET    -- returns JSON object with CSIPO settings
    PUT    -- update CSIPO settings
    """
    def GET(self):
        """
        Return JSON object with CSIPO settings. Format:
        {
            csipo_path: "C:\Program Files\CSIPO\",
            csipo_username: "user",
            csipo_password: "password",
            csipo_mount: true
        }

        Error codes:

        400 -- Error occurred while reading CSIPO settings (CSIPO configuration file couldn't be read).
        """
        web.header("Content-Type","application/json")
        try:
            csipo_config_file = configuration.SHARED_CONFIGURATION['debug']['csipo_config']
            csipo_settings = {'csipo_path': None,
                              'csipo_username': None,
                              'csipo_password': None,
                              'csipo_mount': None}
            utils.read_config(csipo_config_file, csipo_settings)
            csipo_settings['csipo_mount'] = (csipo_settings['csipo_mount'] in ["True", "true", "t", "T"])
            return json.dumps(csipo_settings)
        except IOError:
            log.exception(TAG, 'unable to read CSIPO configuration file %s' % csipo_config_file)
            web.ctx.status = '400'

    def PUT(self):
        """
        Update CSIPO settings.

        Input:

        csipo_path     -- The path to the shared Windows folder on the CSIPO machine.
        csipo_username -- The username to be used for authentication.
        csipo_password -- The password to be used for authentication.
        csipo_mount    -- True to mount the shared folder , False to unmount it

        Error codes:

        400 -- I/O error when writting CSIPO configuration
        """
        try:
            csipo_settings = {}

            csipo_path = web.input(csipo_path=None).csipo_path
            if csipo_path:
                csipo_settings['csipo_path'] = csipo_path
            else:
                csipo_settings['csipo_path'] = ""

            csipo_username = web.input(csipo_username=None).csipo_username
            if csipo_username:
                csipo_settings['csipo_username'] = csipo_username
            else:
                csipo_settings['csipo_username'] = ""

            csipo_password = web.input(csipo_password=None).csipo_password
            if csipo_password:
                csipo_settings['csipo_password'] = csipo_password
            else:
                csipo_settings['csipo_password'] = ""

            csipo_mount = web.input(csipo_mount=None).csipo_mount
            if csipo_mount:
                csipo_settings['csipo_mount'] = (csipo_mount in ["True", "true", "t", "T"])

            csipo_config_file = configuration.SHARED_CONFIGURATION['debug']['csipo_config']
            utils.update_config(csipo_config_file, csipo_settings)

            # TODO : call script that reads the csipo config file and applies the settings

        except (IOError, OSError):
            log.exception(TAG, 'unable to write CSIPO configuration file %s' % csipo_config_file)
            web.ctx.status = '400'

class SNMPSettingsResource(object):
    """
    REST resource to manage watchdog SNMP settings.

    Methods:

    GET     -- return JSON object with watchdog SNMP settings
    PUT     -- update watchdog SNMP settings
    """

    @synchronized(snmp_settings_lock)
    def GET(self):
        """
        Return JSON object with watchdog SNMP settings.
        
        Error codes:
        
        400 -- Error occurred while reading SNMP settings (watchdog configuration file couldn't be read).
        """
        web.header("Content-Type","application/json")
        try:
            config_file = configuration.SHARED_CONFIGURATION['snmp']['config_file']
            snmp_settings = {'WDOG_SNMP_PROTOCOL': None, 
                             'WDOG_SNMP_ADDRESS_SEND': None,
                             'WDOG_SNMP_PORT': None,
                             'WDOG_SNMP_DEVID': None}
            utils.read_config(config_file, snmp_settings)
            
            snmpd_file = configuration.SHARED_CONFIGURATION['snmp']['snmpd_config']
            snmpd_settings = {'sysdescr': None, 'syslocation': None}
            applications.read_snmpd_settings(snmpd_file, snmpd_settings)
            snmp_settings.update(snmpd_settings)

            snmp_settings['snmpd_enabled'] = False
            if shell.chkconfig(configuration.SHARED_CONFIGURATION['snmp']['service_name']):
                snmp_settings['snmpd_enabled'] = True


            return json.dumps(snmp_settings)
        except IOError:
            log.exception(TAG, 'unable to read SNMP configuration file %s' % config_file)
            web.ctx.status = '400'

    @synchronized(snmp_settings_lock)
    def PUT(self):
        """
        Update watchdog SNMP settings.
        
        Input:
        
        wdog_snmp_protocol     -- protocol used for sending SNMP traps (udp/tcp)
        wdog_snmp_address_send -- IP where SNMP traps should be sent
        wdog_snmp_port         -- port where SNMP traps should be sent
        wdog_snmp_devid        -- trap sender device id
        snmp_sys_desc          -- 'system description' for SNMP agent
        snmp_sys_loc           -- 'system location' for SNMP agent
        snmp_enabled           -- on|off
        
        Error codes:
        
        400 -- validation errors on input data or I/O error when writting SNMP configuration
        """
        try:
            snmp_settings = {}
            snmpd_settings = {}
            validation_ok = True
            
            snmp_protocol = web.input(wdog_snmp_protocol=None).wdog_snmp_protocol
            if snmp_protocol and snmp_protocol in ['udp', 'tcp']:
                snmp_settings['WDOG_SNMP_PROTOCOL'] = snmp_protocol
            else:
                validation_ok = False

            snmp_ip = web.input(wdog_snmp_address_send=None).wdog_snmp_address_send
            if snmp_ip:
                snmp_settings['WDOG_SNMP_ADDRESS_SEND'] = snmp_ip
            
            snmp_port = web.input(wdog_snmp_port=None).wdog_snmp_port
            try:
                int(snmp_port)
                if snmp_port:
                    snmp_settings['WDOG_SNMP_PORT'] = snmp_port                
            except ValueError:
                validation_ok = False

            snmp_device_id = web.input(wdog_snmp_devid='').wdog_snmp_devid
            snmp_settings['WDOG_SNMP_DEVID'] = snmp_device_id

            snmp_desc = web.input(snmp_sys_desc=None).snmp_sys_desc
            if snmp_desc:
                snmpd_settings['sysDescr'] = snmp_desc

            snmp_loc = web.input(snmp_sys_loc=None).snmp_sys_loc
            if snmp_loc:
                snmpd_settings['sysLocation'] = snmp_loc

            snmpd_enabled = str(web.input(snmpd_enabled="off").snmpd_enabled)
            if not snmpd_enabled in ("on", "off"):
                validation_ok = False

            config_file = configuration.SHARED_CONFIGURATION['snmp']['config_file']
            if validation_ok:
                utils.update_config(config_file, snmp_settings)
                
                snmpd_file = configuration.SHARED_CONFIGURATION['snmp']['snmpd_config']
                applications.update_snmpd_config(snmpd_file, snmpd_settings)

                shell.sudo_call('service watchdog restart')

                snmp_service_name = configuration.SHARED_CONFIGURATION['snmp']['service_name']
                snmp_runlevels = configuration.SHARED_CONFIGURATION['snmp']['runlevels']
                if snmpd_enabled == "on":
                    shell.sudo_call('systemctl start %s' % snmp_service_name)
                    shell.sudo_call('systemctl enable %s.service ' %  ( snmp_service_name))
                else:
                    shell.sudo_call('systemctl stop %s' % snmp_service_name)
                    shell.sudo_call('systemctl disable %s.service ' %  ( snmp_service_name))

                audit.log(i18n.custom_gettext('change SNMP settings'))
            else:
                log.error(TAG, 'invalid SNMP configuration data %s' % str(snmpd_settings))
                web.ctx.status = '400'
        except (IOError, OSError):
            log.exception(TAG, 'unable to write SNMP configuration file %s' % config_file)
            web.ctx.status = '400'


class MIBResource(object):
    """
    REST resource to manage IP Office specific MIB files.

    Methods:

    GET  /files     -- return a list with MIB files available for download
    GET  /download  -- download an archive with all MIB files
    """

    MIB_ARCHIVE = '/tmp/ipoffice_mibs.zip'
    BLOCK_SIZE = 16384

    @synchronized(mib_files_lock)
    def GET(self, param):
        """
        Return for:

        /files      -- a simple JSON object {'mib_files': [filepath0, filepath1,...]}
        /download   -- a zip archive with all the MIB files

        Error codes:

        404 -- invalid parameter or no MIB files for download
        """
        if param == 'files':
            web.header("Content-Type", "application/json")
            return json.dumps({'mib_files' : self.mib_files()})
        elif param == 'download':
            files = self.mib_files()
            if files:
                shell.rm(self.MIB_ARCHIVE)
                shell.zip(files, self.MIB_ARCHIVE, '-r')
                try:
                    stat = os.stat(self.MIB_ARCHIVE)
                    mime_type = mimetypes.guess_type(self.MIB_ARCHIVE)[0]
                    web.header('Content-Type', '%s' % mime_type)
                    web.header('Content-Length', '%s' % stat.st_size)
                    web.header('Last-Modified', '%s' %
                        web.http.lastmodified(datetime.datetime.fromtimestamp(stat.st_mtime)))
                    web.header('Content-Disposition',
                               'attachment; filename=%s' % os.path.basename(self.MIB_ARCHIVE))
                    if (web.ctx.protocol.lower() == 'https'):
                        # add headers to fix issue with IE download over SSL failing when "no-cache" header set
                        web.header('Pragma', 'private')
                        web.header('Cache-Control', 'private,must-revalidate')
                    return wsgiref.util.FileWrapper(open(self.MIB_ARCHIVE, 'rb'), self.BLOCK_SIZE)
                except (OSError, IOError):
                    log.exception(TAG, 'unable to serve archive %s' % self.MIB_ARCHIVE)
                    web.ctx.status = '400'
            else:
                log.warn(TAG, 'no MIB files available for download')
                web.ctx.status = '404'
        else:
            log.warn(TAG, 'invalid parameter: %s' % param)
            web.ctx.status = '404'

    def mib_files(self):
        return glob.glob(os.path.join(
                    configuration.SHARED_CONFIGURATION['snmp']['mib_files_dir'], '*'))


class WebappSettingsResource(object):
    """
    REST resource to manage Web Control port settings.

    Methods:

    GET     -- return JSON object containing Web Control port settings
    PUT     -- update Web Control settings
    POST    -- restart Web Control
    """

    @synchronized(webapp_settings_lock)    
    def GET(self):
        """
        Return JSON object containing Web Control settings.

        Error codes:

        400 -- unable to read Web Control configuration
        """
        try:
            web.header("Content-Type", "application/json")
            webapp_config = {'port': configuration.get_global_config(webapp.PORT_PARAM_NAME),
                             'protocol': configuration.get_global_config(webapp.PROTOCOL_PARAM_NAME),
                             'timeout': configuration.get_global_config(webapp.SESSION_TIMEOUT_PARAM_NAME)}
            return json.dumps(webapp_config)
        except KeyError:
            log.exception(TAG, 'unable to read Web Control configuration from file')
            web.ctx.status = '400'

    @synchronized(webapp_settings_lock)
    def PUT(self):
        """
        Update Web Control port settings.

        Error codes:

        400 -- unable to write Web Control configuration
        """
        port = None
        try:
            port = web.input(port=None).port
            protocol = web.input(protocol=None).protocol
            timeout = web.input(timeout=None).timeout
            webapp.write_settings(port, protocol, timeout)
            configuration.data_validate_type_length(port,configuration.DATA_PORT_VALIDATOR,configuration.DATALENGTH_PORT_VALIDATOR)
            audit.log(i18n.custom_gettext('set Web Control configuration: port: %s, protocol: %s, timeout: %s s')
                      % (port, protocol, timeout))
        except (IOError, OSError, ValueError):
            log.exception(TAG, 'unable to save Web Control configuration to file')
            web.ctx.status = '400'

    
    @synchronized(webapp_settings_lock)
    @configuration.csrf_protected
    def POST(self):
        """
        Restart Web Control.
        
        Note that, because it calls a command which resets the REST service, 
        this call will not finish handling the request. That means the call
        should be asynchronous (e.g. for AJAX make sure 'async' param is 'true')
        """
        shell.sudo_call('systemctl restart webcontrol')

class CertificateResource(object):
    """
    REST resource used to manage certificates

    Methods:

    GET     -- return certificate status
    PUT     -- update certificate
    """

    CERTIFICATE_PATH = configuration.SHARED_CONFIGURATION['webapp']['cert_path']
    IPOFFICE_CERTIFICATE_PATH = configuration.SHARED_CONFIGURATION['webapp']['ipoffice_cert_path']
    CURRENT_CERTIFICATE_PATH = configuration.SHARED_CONFIGURATION['webapp']['certificate_file']
    CURRENT_CERTIFICATE_CHAIN_PATH = configuration.SHARED_CONFIGURATION['webapp']['chain_path']
    CERTIFICATE_PARENT_PATH = configuration.SHARED_CONFIGURATION['webapp']['cert_parent_path']
    WCP_DISTRIBUTE_FLAG = configuration.SHARED_CONFIGURATION['webapp']['wcp_distribute_flag']
    WCP_NO_RESTART_FLAG = configuration.SHARED_CONFIGURATION['webapp']['wcp_no_restart_flag']
    DISTRIBUTE_COMPLETE_FLAG = configuration.SHARED_CONFIGURATION['webapp']['distribute_complete_flag']
    DISTRIBUTE_IN_PROGRESS_FLAG = configuration.SHARED_CONFIGURATION['webapp']['distribute_in_progress_flag']
    AUTO_DISTRIBUTE_FLAG = configuration.SHARED_CONFIGURATION['webapp']['auto_generate_flag']
    CERT_GEN_SCRIPT = configuration.SHARED_CONFIGURATION['webapp']['cert_gen_script']
    KEYSTORE_SCRIPT_PATH = "/opt/Avaya/scripts/keystore.sh"
    auto_temp_created = False

    # Constants for certificate status
    UNAVAILABLE = 0
    OUT_OF_DATE = 1
    UP_TO_DATE  = 2

    SSL_SERVICES = ['csipo', 'ipoffice', 'weblm', 'voicemail', 'onexportal', 'webmanager', "webrtcgw"]
    SSL_SYSTEM_SERVICES = ["httpd"]
    STOP_SERVICE_RETRY_COUNT = 40

    def _get_affected_services(self):
        affected_services = []
        for service_id in self.SSL_SERVICES:
            if service_id in services_rest_api.services_manager.services:
                affected_services.append(services_rest_api.services_manager.services[service_id].display_name)
        affected_services.append('webcontrol')
        return affected_services

    def check_distribution_in_progress(self):
        counter = 180
        delay = 1
        sleep = 10
        flag = configuration.SHARED_CONFIGURATION['webapp']['distribute_in_progress_flag']
        time.sleep(sleep)
        while counter > 0:
            if shell.secure_find_file(flag) is True:
                time.sleep(delay)
                counter -= 1
            else:
                break

    def _install_certificate(self):

        shell.sudo_call("touch %s" % self.WCP_NO_RESTART_FLAG)
        shell.sudo_call("chmod 600 %s" % self.WCP_NO_RESTART_FLAG)

        ip = "no_IP"
        if version.RELEASE_TYPE == "apc" :
            try:
                ip = utils.interface_ip("eth0.1")
            except:
                pass
        else:
            try:
                ip = utils.interface_ip("eth0")
            except:
                try:
                    ip = utils.interface_ip("eth1")
                except:
                    pass
        cert_name = "server_%s.pem" % ip
        cert_file = os.path.join(self.CERTIFICATE_PARENT_PATH, cert_name)

        shell.sudo_call("mv %s %s" % (cert_file, self.IPOFFICE_CERTIFICATE_PATH))
        if shell.secure_find_file(self.AUTO_DISTRIBUTE_FLAG) is False:
            time.sleep(10)
            shell.sudo_call("touch %s" % self.AUTO_DISTRIBUTE_FLAG)
            shell.sudo_call("chmod 600 %s" % self.AUTO_DISTRIBUTE_FLAG)
            self.auto_temp_created = True


    def check_distrib_complete(self):
        counter = 18
        interval = 20
        while counter > 0:
            if shell.secure_find_file(self.DISTRIBUTE_COMPLETE_FLAG) is False:
                counter -= 1
                time.sleep(interval)
            else:
                shell.sudo_call("rm %s" % self.DISTRIBUTE_COMPLETE_FLAG)
                shell.sudo_call("rm %s" % self.WCP_NO_RESTART_FLAG)
                if self.auto_temp_created is True:
                    shell.sudo_call("rm %s" % self.AUTO_DISTRIBUTE_FLAG)
                break

    def _stop_services(self, services_to_stop):
        services_running = {}
        for service_id in services_to_stop:
            if service_id in services_rest_api.services_manager.services:
                services_to_stop[service_id] = services_rest_api.services_manager[service_id]
            if services_to_stop[service_id]:
                services_to_stop[service_id].update_run_info()
                if services_to_stop[service_id].is_running():
                    services_running[service_id] = services_to_stop[service_id]
                    services_to_stop[service_id].stop()
                    audit.log(i18n.custom_gettext('updating SSL certificate: stopping %s service...') %
                              services_to_stop[service_id].service_name)
                    count = self.STOP_SERVICE_RETRY_COUNT
                    while services_to_stop[service_id].is_running():
                        time.sleep(2)
                        count -= 1
                        if count < 0:
                            services_to_stop[service_id].force_stop()
                            break
                        services_to_stop[service_id].update_run_info()
        return services_running

    def _start_services(self, services_to_start):
        for service_id in services_to_start:
            services_to_start[service_id].start()

    def _start_stop_system_services(self, action=None):
        for service in self.SSL_SYSTEM_SERVICES:
            output = shell.sudo_execute("rpm -q %s" % service, parse=shell.RAW)
            if not b"not installed" in output:
                if action in ["start", "stop"]:
                    shell.sudo_call("systemctl %s %s.service" % ( action,service))

    @synchronized(certificate_settings_lock)
    def GET(self):
        """
        Returns the certificate status

        Error codes:

        400 -- unable to access Avaya certificate folder
        """
        affected = web.input(affected='').affected
        if affected:
            return ', '.join(self._get_affected_services())
        try:
            if shell.secure_find_file(self.CERTIFICATE_PATH):
                if os.stat(self.CERTIFICATE_PATH).st_mtime > os.stat(self.CURRENT_CERTIFICATE_PATH).st_mtime:
                    return self.OUT_OF_DATE
                else:
                    return self.UP_TO_DATE
            else:
                return self.UNAVAILABLE
        except (IOError, OSError):
            log.exception(TAG, 'unable to access certificate location %s' % self.CERTIFICATE_PATH)
            web.ctx.status = '400'

    @synchronized(certificate_settings_lock)
    def PUT(self):
        """
        Replaces WCP certificate

        Error codes:

        400 -- unable to access Avaya certificate folder
        """
        try:
            self._install_certificate()
            self.check_distrib_complete()
        except IOError as ioerr:
            log.exception(TAG, '%s' % ioerr)
            web.ctx.status = '400'

    @synchronized(certificate_settings_lock)
    @configuration.csrf_protected
    def POST(self):
        shell.sudo_call('systemctl restart webcontrol.service')


class CaResource(object):
    """
    REST resource used to manage certificate imports
    """

    CERT_GEN_SCRIPT = configuration.SHARED_CONFIGURATION['webapp']['cert_gen_script']
    TEMP_CA_PATH = configuration.SHARED_CONFIGURATION['webapp']['ca_temp_file']
    CA_PATH = configuration.SHARED_CONFIGURATION['webapp']['ca_path']
    EXPORT_CA_PATH = configuration.SHARED_CONFIGURATION['webapp']['export_ca_path']
    CERTIFICATE_PARENT_PATH = configuration.SHARED_CONFIGURATION['webapp']['cert_parent_path']
    CURRENT_CERTIFICATE_CHAIN_PATH = configuration.SHARED_CONFIGURATION['webapp']['chain_path']
    BLOCK_SIZE = 16384

    @synchronized(ca_lock)
    def GET(self, name):
        """
        Used to check and decrypt imported CA file, or to download the CA
        """
        if not name:
            # import CA
            password = str(web.input(password="").password)
            shell.sudo_call("chmod 777 %s" % self.TEMP_CA_PATH)
            output = shell.sudo_execute("%s --import-ca %s --pass %s" % (self.CERT_GEN_SCRIPT, self.TEMP_CA_PATH, password),
                                        parse=shell.LINES)

            status = "ok"
            for line in output:
                if int(line.find("invalid password")) > -1:
                    status = "invalid password"
                    break
                if int(line.find("error")) > -1:
                    status = "invalid file"
                    break
            if status == "ok":
                if shell.secure_find_file(self.CURRENT_CERTIFICATE_CHAIN_PATH):
                    shell.sudo_call("rm %s" % self.CURRENT_CERTIFICATE_CHAIN_PATH)

            web.header("Content-Type","application/json")
            return json.dumps({"import_status": status})
        else:
            # download CA
            try:
                return shell.secure_get_file(str(name), str(self.CA_PATH))
            except (OSError, IOError):
                log.exception(TAG, "unable to serve ca <%s>" % name)
                web.ctx.status = '404'

    @synchronized(ca_lock)
    def PUT(self, name):
        """
        Used to create a new CA
        """
        pkcs_12 = web.input(pkcs_12="false").pkcs_12
        password = str(web.input(password="").password)
        path = web.input(path="").path
        regenerate = bool(web.input(regenerate=False).regenerate)
        if pkcs_12 == "true":
            if path == "":
                path = self.EXPORT_CA_PATH
            shell.sudo_call("%s --export-ca %s --pass %s" % (self.CERT_GEN_SCRIPT, path, password))
        else:
            if regenerate is True:
                shell.sudo_call("%s --generate-ca-cert --regenerate" % self.CERT_GEN_SCRIPT)
            else:
                shell.sudo_call("%s --generate-ca-cert" % self.CERT_GEN_SCRIPT)

        if shell.secure_find_file(self.CURRENT_CERTIFICATE_CHAIN_PATH):
            shell.sudo_call("rm %s" % self.CURRENT_CERTIFICATE_CHAIN_PATH)

    @synchronized(ca_lock)
    @configuration.csrf_protected
    def POST(self, name):
        """
        Used to upload CA files to server
        """
        ca_input = web.input(ca_file={})
        file_obj = ca_input['ca_file'].file

        file_out_path = self.TEMP_CA_PATH
        utils.copy_file(file_obj, file_out_path)

    @synchronized(ca_lock)
    def DELETE(self, name):
        """
        Used to delete temporary encrypted CA files
        """
        if name:
            shell.sudo_call("rm %s" % self.CA_PATH + str(name))


class CheckCertDistribInProgress(object):
    """
    REST resource to check if a certificate is being distributed.
    """

    DISTRIBUTE_IN_PROGRESS_FLAG = configuration.SHARED_CONFIGURATION['webapp']['distribute_in_progress_flag']
    CHECK_DELAY = 10

    def GET(self):
        """
        Checks if the distribution in progress flag exists.
        """
        time.sleep(self.CHECK_DELAY)
        web.header("Content-Type","application/json")
        return json.dumps({"distrib_status": shell.secure_find_file(self.DISTRIBUTE_IN_PROGRESS_FLAG)})


class GenerateCertResource(object):
    """
    REST resource to manage certificate creation.
    """

    CERT_GEN_SCRIPT = configuration.SHARED_CONFIGURATION['webapp']['cert_gen_script']
    CERTIFICATE_PARENT_PATH = configuration.SHARED_CONFIGURATION['webapp']['cert_parent_path']
    AUTO_GENERATE_FLAG = configuration.SHARED_CONFIGURATION['webapp']['auto_generate_flag']
    BLOCK_SIZE = 16384

    def create_cert(self, subject=None, alt_subject=None, duration=None, rsa=None, sha=None, auto_generate=None,
                    ip=None, self_signed=None, encrypted=None, password=None):
        """
        Used to call the generate certificate script
        """
        command = "%s --generate-server-cert" % self.CERT_GEN_SCRIPT
        if subject is not None and subject != "None":
            command += " --issued-to '%s'" % subject
        if alt_subject is not None and alt_subject != "None":
            command += " --subject-alt-name '%s'" % alt_subject
        if duration is not None and duration != "None":
            command += " --valid-to '%s'" % duration
        if ip is not None and ip != "None"  and configuration.data_validate_type_length(ip,configuration.DATA_IP_VALIDATOR,configuration.DATALENGTH_IP_VALIDATOR):
            command += " --server-ip '%s'" % ip
        if rsa is not None and rsa != "None":
            if rsa.find("1024") > -1:
                command += " --rsa1024"
        if sha is not None and sha != "None":
            if sha.find("1") > -1:
                command += " --sha1"
        if self_signed is not None and self_signed != "None":
            command += " --self-signed"
        if encrypted is not None and encrypted != "None" and password is not None and password != "None":
            command += " --pkcs12 --pass %s" % password
        output = shell.sudo_execute(command, parse=shell.RAW)
        if output != None:
            output = output.decode().strip()

        if auto_generate:
            if auto_generate == "true":
                shell.sudo_call("touch %s" % self.AUTO_GENERATE_FLAG)
                shell.sudo_call("chmod 600 %s" % self.AUTO_GENERATE_FLAG)
            elif auto_generate == "false":
                shell.sudo_call("rm %s" % self.AUTO_GENERATE_FLAG)

        return output


    @synchronized(certificate_settings_lock)
    def GET(self, name):
        """
        Returns default certificate parameters or downloads the certificate
        """
        if not name:
            # Get defaults
            new_hostname = str(web.input(new_hostname=None).new_hostname)
            if new_hostname is not None and new_hostname != "None":
                lines = shell.sudo_execute("%s --print-default-server-vars --hostname %s"
                                           % (self.CERT_GEN_SCRIPT, new_hostname), parse=shell.LINES)
            else:
                lines = shell.sudo_execute("%s --print-default-server-vars" % self.CERT_GEN_SCRIPT, parse=shell.LINES)
            defaults = {}
            for line in lines:
                if not ": " in line:
                    continue
                key = line.split(": ")[0].strip().replace(" ", "_").replace("-", "_").lower()
                value = line.split(": ")[1].strip()
                defaults[key] = value
            defaults["auto_generate"] = shell.secure_find_file(self.AUTO_GENERATE_FLAG)

            lines = shell.sudo_execute("%s --print-ca-info" % self.CERT_GEN_SCRIPT, parse=shell.LINES)
            if lines is None:
                lines = ""
            for line in lines:
                if line.startswith("IssuedTo"):
                    key = "ca_issued_to"
                    value = ""
                    values = line.split("IssuedTo:")[1].split(",")
                    for item in values:
                        if "CN=" in item:
                            value = item.split("=")[1].strip()
                            break
                    defaults[key] = value
                elif line.startswith("IssuedBy"):
                    key = "ca_issued_by"
                    value = ""
                    values = line.split("IssuedBy:")[1].split(",")
                    for item in values:
                        if "CN=" in item:
                            value = item.split("=")[1].strip()
                            break
                    defaults[key] = value
                else:
                    continue

            web.header("Content-Type","application/json")
            return json.dumps(defaults, default=utils.encode_json)
        else:
            # download cert
            try:
                return shell.secure_get_file(str(name), str(self.CERTIFICATE_PARENT_PATH))
            except (OSError, IOError):
                log.exception(TAG, "unable to serve ca <%s>" % name)
                web.ctx.status = '404'


    @synchronized(certificate_settings_lock)
    def PUT(self, name):
        """
        Creates a certificate with the given parameters
        """
        subject = str(web.input(subject=None).subject)
        alt_subject = str(web.input(alt_subject=None).alt_subject)
        duration = str(web.input(duration=None).duration)
        rsa = str(web.input(rsa=None).rsa)
        sha = str(web.input(sha=None).sha)
        auto_generate = str(web.input(auto_generate=None).auto_generate)
        ip = str(web.input(ip=None).ip)
        self_signed = str(web.input(self_signed=None).self_signed)
        encrypted = str(web.input(encrypted=None).encrypted)
        password = str(web.input(password=None).password)

        output = self.create_cert(subject, alt_subject, duration, rsa, sha, auto_generate, ip, self_signed,
                                  encrypted, password)
        path = None
        filename = None
        if output:
            if "Cannot Sign the Certificate" in output:
                web.ctx.status = '400'
                return i18n.custom_gettext('Error creating certificate. Please make sure the parameters are correct.')
            if "PATH" in output:
                path = output.split("=")[1].strip()
                filename = path.split("/")[-1]
        web.header("Content-Type","application/json")
        return json.dumps({"path": path, "name": filename}, default=utils.encode_json)


    @synchronized(certificate_settings_lock)
    @configuration.csrf_protected
    def POST(self, name):
        """
        Sets auto-generate certificate to on/off
        """
        auto_generate = str(web.input(auto_generate=None).auto_generate)
        if auto_generate:
            if auto_generate == "true":
                shell.sudo_call("touch %s" % self.AUTO_GENERATE_FLAG)
                shell.sudo_call("chmod 600 %s" % self.AUTO_GENERATE_FLAG)
            elif auto_generate == "false":
                shell.sudo_call("rm %s" % self.AUTO_GENERATE_FLAG)


    @synchronized(certificate_settings_lock)
    def DELETE(self, name):
        """
        Deletes a certificate.
        """
        if name:
            shell.sudo_call("rm %s" % self.CERTIFICATE_PARENT_PATH + str(name))


class RepositoryResource(object):
    """
    REST resource to manage update repositories.

    Methods:

    GET     -- return JSON object with update repositories configuration
    PUT     -- modify repositories configuration
    """

    @synchronized(repo_settings_lock)
    def GET(self, id):
        """
        Returns configuration of update repositories.
        
        The result is in the following form:

        { 'repo_name': {'local': True|False, 'url':'file:///..'|'http://...'}, ...}
        
        To get configuration for a specified repository only, add it's name to
        request url (as in '/api/settings/repo/<repo_name>').
        
        If no repository is indicated, the config for all repositories is returned.

        Error codes:

        404 -- invalid repository name
        400 -- error reading repository configuration
        """
        repo_config_file = configuration.SHARED_CONFIGURATION['repositories']['config_file']

        try:
            repo_config = repositories.get_config(repo_config_file, id)
            if repo_config:
                web.header("Content-Type","application/json")
                return json.dumps(repo_config)
            else:
                web.ctx.status = '404'
        except ConfigObjError:
            log.exception(TAG, 'unable to read repository configuration file %s' % repo_config_file)
            web.ctx.status = '400'

    @synchronized(repo_settings_lock)
    def PUT(self, id):
        """
        Modify the configuration of updates repository.

        Error codes:

        404 -- invalid repository name
        400 -- error writting repository configuration
        """
        # known repositories
        repos = ['system', 'apps', 'winapps']
        if id:
            # update configurations for specified repository
            if id in repos:
                repos = [id]
            else:
                web.ctx.status = '404'
                return
        repo_config = {}
        for repo_name in repos:
            try:
                repo_local = web.input()[repo_name + '_repo_local']
                if repo_local == 'true':
                    repo_url = 'file://' + configuration.SHARED_CONFIGURATION['repositories']['local'][repo_name]
                elif repo_local == 'false':
                    if not configuration.data_length_validate( web.input()[repo_name + '_repo_url'],configuration.DATALENGTH_URL_VALIDATOR):
                        repo_url = web.input()[repo_name + '_repo_url']
                else:
                    continue
                repo_config[repo_name] = {'local': repo_local, 'url': repo_url}
            except KeyError:
                continue
            except ValueError:
                continue
        try:
            repo_config_file = configuration.SHARED_CONFIGURATION['repositories']['config_file']
            repositories.set_config(repo_config_file, repo_config)
            audit.log(i18n.custom_gettext('change repositories settings'))
        except ConfigObjError:
            log.exception(TAG, 'unable to save repository configuration file %s' % repo_config_file)
            web.ctx.status = '400'

    @synchronized(repo_settings_lock)
    @configuration.csrf_protected
    def POST(self, id):
        """
        Used to upload files to repository.
        
        If an archive (.tar.gz, .tar, .tgz, .zip) is uploaded, this will be
        deflated first and then the file deleted.

        Error codes:

        404 -- invalid repository name
        """
        repos = ['system', 'apps', 'winapps']
        if id:
            if id in repos:
                repo = id
            else:
                web.ctx.status = '404'
                return
        else:
            # invalid input params: repository name missing
            repo = web.input(repository=None).repository
            if repo not in repos:
                log.error(TAG, 'invalid or missing repository name')
                web.ctx.status = '404'
                return
        
        repo_input = web.input(repo_file={})

        file_name = utils.normalize_path(repo_input['repo_file'].filename)
        file_obj = repo_input['repo_file'].file
        parsed_file_name = os.path.basename(file_name)
        if not configuration.data_validate(parsed_file_name,configuration.DATA_FILE_VALIDATOR):
            log.error(TAG, 'invalid or missing file name')
            web.ctx.status = '404'
            return
 
        rights_changed = False
        if not os.access(configuration.SHARED_CONFIGURATION['repositories']['local'][repo], os.W_OK):
            shell.sudo_call("chmod go+w %s" % configuration.SHARED_CONFIGURATION['repositories']['local'][repo])
            rights_changed = True
        file_out_path = os.path.join(configuration.SHARED_CONFIGURATION['repositories']['local'][repo],
                                     parsed_file_name)
        utils.copy_file(file_obj, file_out_path)
        if rights_changed is True:
            shell.sudo_call("chmod go-w %s" % configuration.SHARED_CONFIGURATION['repositories']['local'][repo])

        if utils.deflate_archive(file_out_path):
            # delete original archive file
            os.remove(file_out_path)
        
        # re-index YUM repository
        if repo == 'system':
            create_repo = 'createrepo %s' % configuration.SHARED_CONFIGURATION['repositories']['local'][repo]
            refresh_yum = 'yum clean expire-cache'
            shell.sudo_calls([create_repo, refresh_yum])
        audit.log(i18n.custom_gettext('upload file to %s repository') % repo)
            
class BackupResource(object):
    """
    REST resource to manage backups.

    Methods:
    POST    -- triggers a new backup operation
    """

    @synchronized(backup_settings_lock)
    def GET(self, id):
        """
        Returns a JSON object with info about available backups:

        {
            'backups_exist':    True|False      - if there is at least one backup created for service,
            'backups':          [backup_info]   - a list with info (specific for each service) about each backup
        }

        For VoiceMail Pro, the backup info has the following structure:

        {
            'name':         name of the backup file,
            'location':     path to backup file (used for restore operation),
            'type':         type of backup (Immediate|Daily|Weekly|Monthly),
            'timestamp':    backup creation time,
            'contents':     a list with backup contents
                            (Voicemails, User Settings and Greetings, Campaigns,
                            Callflows Modules and Conditions, Module Recording, System Settings)
            'size':         size of backup,
            'os':           OS where backup was taken (1-Linux, 2-Windows)
        }

        For IP Office, the response only indicates if a backup is available (IP Office doesn't
        support multiple backups). In this case the 'backups' key has an empty list as value.
        """
        services = services_rest_api.services_manager.services
        if id in services:
            backups = []
            if id == 'voicemail':
                backups = backup_manager.vmpro_backups_list()
                backups_exist = len(backups) > 0
            elif id == 'ipoffice':
                output = backup_manager.list(id)
                backups_exist = output != 'No backup found'
            elif id == 'webrtcgw':
                output = backup_manager.webrtcgw_backups_list()
                backups_exist = len(output) > 0
            else:
                log.error(TAG, 'invalid service name - %s - or operation not supported for service' % id)
                web.ctx.status = '404'
                return

            web.header("Content-Type","application/json")
            backups_response = {'backups_exist': backups_exist,
                                'backups': backups}
            return json.dumps(backups_response, default=utils.encode_json)
        else:
            log.error(TAG, 'invalid service name: %s' % id)
            web.ctx.status = '404'

    @synchronized(backup_settings_lock)
    @configuration.csrf_protected
    def POST(self, id):
        """
        Triggers a backup operation
        """
        services = services_rest_api.services_manager.services
        if id in services:
            if services[id].backup_supported:
                result_msg = backup_manager.backup(id)
                if result_msg != 'Backup complete' and result_msg != 'Backup successful':
                    log.error(TAG, 'backup error: %s' % result_msg)
                    web.ctx.status = '400'
                    return result_msg
            else:
                log.error(TAG, 'backup operation not supported')
                web.ctx.status = '400'
        else:
            log.error(TAG, 'invalid service name: %s' % id)
            web.ctx.status = '404'
        
class RestoreResource(object):
    """
    REST resource to restore application data from a backup.

    Methods:

    POST    -- triggers a new restore operation
    """
    
    @synchronized(restore_settings_lock)
    @configuration.csrf_protected
    def POST(self, id):
        """
        Triggers a restore operation
        
        Input:
        
        id              -- service name (as part of URL)
        backup_source   -- (vmpro only, optional, as URL input parameter)
                            the path to the backup to restore from
        
        Error codes:
        
        400 -- a specific error occurred
        404 -- invalid service name
        """
        services = services_rest_api.services_manager.services
        if id in services:
            if services[id].restore_supported:
                if id == 'voicemail':
                    backup_source = str(web.input(backup_source=None).backup_source)
                    backup_manager.vmpro_restore(backup_source)
                    return
                elif id == "webrtcgw":
                    result_msg = backup_manager.webrtcgw_restore()
                    if result_msg:
                        log.error(TAG, 'restore error: %s' % result_msg)
                        web.ctx.status = '400'
                        return result_msg
                    return
                else:
                    result_msg = backup_manager.restore(id)
                    if result_msg != 'Backup restored':
                        log.error(TAG, 'restore error: %s' % result_msg)
                        web.ctx.status = '400'
                        return result_msg
            else:
                log.error(TAG, 'restore operation not supported')
                web.ctx.status = '400'
        else:
            log.error(TAG, 'invalid service name: %s' % id)
            web.ctx.status = '404'

class RestoreResourceStatus(object):
    """
    REST resource to retrieve information about the status of the restore operation.

    Methods:

    GET -- return the status of the restore operation
    """
    def GET(self):
        """
        Return json string with Application restore status.

        Data format:

        {
            'in_progress':      true if a restore is in progress
        }
        """
        restore_status = restore_manager.get_restore_status()
        web.header("Content-Type","application/json")
        return json.dumps(restore_status)


class AsgStatusResource(object):
    """
    REST resource to retrieve information about the status of the easg.

    Methods:
    GET -- return the status of the easg if installed or uninstalled
    """
    def GET(self):
        """
        Return json string

        Data format:

        {
            'asg_status':      true if asg is installed
        }
        """
        asg_stat = {
               "asg_status" : None
        }

        asg_status_out =  shell.sudo_execute("rpm -q asgsshd asgtools", parse=shell.RAW)
        asg_stat["asg_status"] = "true"
        if "not installed" in asg_status_out.decode():
            asg_stat["asg_status"] = "false"
        web.header("Content-Type","application/json")
        return json.dumps(asg_stat)

class AsgResource(object):
    """
    REST resource to retrieve information about the status of the restore operation.

    Methods:

    GET    -- return the current ASG settings
    PUT    -- change the ASG settings
    POST   -- upload an ASF file
    DELETE -- reset the ASG service
    """

    ASG_SCRIPT = configuration.SHARED_CONFIGURATION["asg"]["asg_script"]
    ASF_LOCATION = configuration.SHARED_CONFIGURATION["asg"]["asf_location"]
    ASG_ROOT = configuration.SHARED_CONFIGURATION["asg"]["asg_root"]

    @synchronized(asg_lock)
    def GET(self):
        """
        Returns the settings of the ASG service.
        """
        asg_settings = {"status": None,
                        "port": None,
                        "listening": None,
                        "easgusers": None,
                        "easgusers_state": None,
                        "enable_status" : None,
                        "asgterms" : None
                        }
        asg_cert_list = {
                        "cert_text" : None
                        }
        password = str(web.input(password=None).password)
        cert_name = str(web.input(cert_name=None).cert_name)
        showcert = str(web.input(showcert=None).showcert)
        acceptterms = str(web.input(acceptterms=None).acceptterms)
        if cert_name is not None and cert_name != "None" and password is not None and password != "None":
            cert_name = utils.normalize_path(cert_name)
            path_to_file = os.path.join(self.ASF_LOCATION, os.path.basename(cert_name).replace('.p7b','.saf'))
            file_temp = os.path.join("/tmp/", os.path.basename(cert_name).replace('.p7b','.saf'))
            with shell.file_open(file_temp, 'w') as f:
                f.writelines(password)
                f.close()
            if os.path.exists(self.ASF_LOCATION):
                shell.sudo_call("chmod g+w %s" % self.ASF_LOCATION)
                shell.sudo_call("\mv %s %s" % (file_temp,path_to_file))
                shell.sudo_call("chown root:root %s" % (path_to_file))
                shell.sudo_call("chmod -R 700 %s" % self.ASF_LOCATION)

        if showcert is not None and showcert != "None":
            if os.path.exists(self.ASG_SCRIPT):
                asg_cert_list["cert_text"] = shell.sudo_execute("%s --show-tech-cert=%s" % (self.ASG_SCRIPT, showcert), parse=shell.RAW).strip()
            web.header("Content-Type","application/json")
            return json.dumps(asg_cert_list, default=utils.encode_json)


        if os.path.exists(self.ASG_SCRIPT):
            status_output = shell.sudo_execute("%s --get-status" % self.ASG_SCRIPT, parse=shell.RAW).decode()
            if status_output.find("enabled") != -1:
                asg_settings["status"] = "enabled"
            elif status_output.find("disabled") != -1:
                asg_settings["status"] = "disabled"
                
            port = shell.sudo_execute("%s --get-port" % self.ASG_SCRIPT, parse=shell.RAW).decode()
            if port and configuration.data_validate_type_length(port,configuration.DATA_PORT_VALIDATOR,configuration.DATALENGTH_PORT_VALIDATOR):
                asg_settings["port"] = port.strip()
            else:
                asg_settings["port"] = ""

            listening =  shell.sudo_execute("%s --get-interfaces" % self.ASG_SCRIPT, parse=shell.RAW)
            if listening:
                asg_settings["listening"] = listening.strip()
            else:
                asg_settings["listening"] = ""

            afs_id = shell.sudo_execute("%s --get-asg-id" % self.ASG_SCRIPT, parse=shell.RAW)
            if afs_id:
                asg_settings["afs_id"] = afs_id.decode().strip()
            else:
                asg_settings["afs_id"] = i18n.custom_gettext("-")
            if acceptterms:
                shell.sudo_execute("%s --accept-terms=%s" % (self.ASG_SCRIPT, acceptterms), parse=shell.RAW)
            output_lines = shell.sudo_execute("%s --list-users" % self.ASG_SCRIPT, parse=shell.RAW).decode()
            asguserlist = []
            asguserstatelist = []
            if output_lines:
                output_lines = output_lines.split("\n")[2:-1]
                for line in output_lines:
                    line_list = line.rsplit()
                    asguserlist.append(line_list[0])
                    if line_list[1] == "yes":
                        asguserstatelist.append("1")
                    elif line_list[1] == "no":
                        asguserstatelist.append("0")
            asgcerts = []
            list_certs = shell.sudo_execute("%s --list-tech-cert" % self.ASG_SCRIPT, parse=shell.RAW)
            if list_certs:
                list_certs = list_certs.decode().split("\n")[1:-1]
                for line in list_certs:
                    list_line = []
                    list_line = line.split()
                    asgcerts.append(list_line[1])
            asg_settings["easgusers"] = asguserlist
            asg_settings["easgusers_state"] = asguserstatelist
            asg_settings["easgcerts"] = asgcerts
            

        web.header("Content-Type","application/json")
        return json.dumps(asg_settings, default=utils.encode_json)

    @synchronized(asg_lock)
    def PUT(self):
        """
        Changes the ASG settings.
        """
        status = str(web.input(status=None).status)
        port = str(web.input(port=None).port)
        listening = str(web.input(listening=None).listening)
        asgusername = str(web.input(asgusername=None).asgusername)
        asguserstate = str(web.input(asguserstate=None).asguserstate)
        deletecert = str(web.input(deletecert=None).deletecert)
        password = str(web.input(password=None).password)
        if os.path.exists(self.ASG_SCRIPT):
            if port is not None and port != "None" and configuration.data_validate_type_length(port,configuration.DATA_PORT_VALIDATOR,configuration.DATALENGTH_PORT_VALIDATOR):
               shell.sudo_call("%s --configure-port=%s" % (self.ASG_SCRIPT, port))
            if listening is not None and listening != "None":
                shell.sudo_call("%s --configure-interfaces=%s" % (self.ASG_SCRIPT, listening))
            if status is not None and status != "None":
                option = None
                if status == "enabled":
                    option = "--enable-service"
                elif status == "disabled":
                    option = "--disable-service"
                if option:
                    output_lines = shell.sudo_execute("%s %s" % (self.ASG_SCRIPT, option), parse=shell.RAW)
                    if output_lines:
                        if option == "--enable-service" and  b"ERROR - Cannot activate service because terms are not accepted" in output_lines:
                            terms = { "asgterms" : None, "enable_status" : "terms issue" }
                            web.header("Content-Type","application/json")
                            terms["asgterms"] = shell.sudo_execute("%s --accept-terms=list" % (self.ASG_SCRIPT), parse=shell.RAW)
                            return json.dumps(terms, default=utils.encode_json)
                        elif b"ERROR - Cannot activate service because terms are not accepted" not in output_lines:
                            terms = { "enable_status" : "successful" }
                            web.header("Content-Type","application/json")
                            return json.dumps(terms, default=utils.encode_json)
                    
            if asgusername is not None and asgusername != "None":
                if asguserstate == "1":
                    shell.sudo_call("%s --enable-user=%s" % (self.ASG_SCRIPT, asgusername))
                elif asguserstate == "0":
                    shell.sudo_call("%s --disable-user=%s" % (self.ASG_SCRIPT, asgusername))
            if deletecert is not None and deletecert != "None":
                shell.sudo_call("%s --delete-tech-cert=%s" % (self.ASG_SCRIPT, deletecert))

    @synchronized(asg_lock)
    @configuration.csrf_protected
    def POST(self):
        """
        Used to upload ASF files to server
        """
        asf_input = web.input(asf_file={})
        file_obj = asf_input['asf_file'].file
        file_name = utils.normalize_path(asf_input['asf_file'].filename)
        parsed_file_name = os.path.basename(file_name)
        shell.sudo_call("chmod 755 %s %s" % (self.ASG_ROOT,self.ASF_LOCATION) )
        if os.path.exists(self.ASF_LOCATION):
            file_out_path = os.path.join(self.ASF_LOCATION, parsed_file_name)
            utils.copy_file(file_obj, file_out_path)
            shell.sudo_call("chown root:root %s" % (file_out_path))
        shell.sudo_call("chmod 700 %s %s" % (self.ASG_ROOT,self.ASF_LOCATION)) 


    @synchronized(asg_lock)
    def DELETE(self):
        """
        Resets the ASG service by reinstalling its packages.
        """
        if os.path.exists(self.ASG_SCRIPT):
            shell.sudo_call("%s --change-asg-id" % (self.ASG_SCRIPT))


class PartitionSizeResource(object):
    """
    REST resource used to manage root partition size

    Methods:

    GET   -- get root partition status
    PUT   -- increase root partition size
    """

    @synchronized(root_partition_lock)
    def GET(self):
        """
        Returns available size for root partition increase
        """
        output = shell.sudo_execute(configuration.SHARED_CONFIGURATION['root_partition']['check_size_script'],
                                    parse=shell.RAW)

        partition_status = {"available_size": output.strip()}

        web.header("Content-Type", "application/json")
        return json.dumps(partition_status, default=utils.encode_json)

    @synchronized(root_partition_lock)
    def PUT(self):
        """
        Increase root partition size to maximum available space
        """
        shell.sudo_call("%s" % configuration.SHARED_CONFIGURATION['root_partition']['increase_size_script'])
        shell.sudo_call("rm %s" % configuration.SHARED_CONFIGURATION["flags"]["expand_in_progress_flag"])


class TcpDumpsSettingsResource(object):
    """
    REST resource to manage package capture settings (tcpdumps).

    Methods:

    GET     -- get the current tcpdumps setting
    PUT     -- start tcpdumps
    DELETE  -- stop tcpdumps
    """
    tcpdumps_config_file = configuration.SHARED_CONFIGURATION['tcpdumps']['config_file']
    tcpdumps_script = configuration.SHARED_CONFIGURATION['tcpdumps']['tcpdumps_script']
    tcpdumps_folder = configuration.SHARED_CONFIGURATION['tcpdumps']['tcpdumps_folder']
    tcpdumps_pid_file = configuration.SHARED_CONFIGURATION['tcpdumps']['tcpdumps_pid_file']
    tcpdumps_percentage = int(configuration.SHARED_CONFIGURATION['tcpdumps']['maximum_size_percentage'])
    min_disk_remaining_space_gb = int(configuration.SHARED_CONFIGURATION['tcpdumps']['minimum_remaining_space_gb_%s'
                                                                                     % version.RELEASE_TYPE])

    def get_available_space(self):

        tcpdumps_folder_size = 0
        if os.path.exists(self.tcpdumps_folder):
            tcpdumps_folder_size = int(shell.sudo_execute("du -s %s" % self.tcpdumps_folder, parse=shell.RAW).split()[0])

        # disk usage
        disk = utils.OpenStruct()
        (disk.total, disk.used, disk.free) = sysinfo.disk_info()

        remaining_free_required = self.min_disk_remaining_space_gb * 1024 * 1024

        # Add space occupied by tcpdumps to available free space, because it will be erased at every start.
        tcpdumps_size_subtraction = disk.free + tcpdumps_folder_size - remaining_free_required
        tcpdumps_size_percentage = int((self.tcpdumps_percentage * disk.total) / 100)

        tcpdumps_space = min(tcpdumps_size_percentage, tcpdumps_size_subtraction)

        return tcpdumps_space

    def is_running(self):

        if os.path.isfile(self.tcpdumps_pid_file):
            tcpdumps_pid = shell.sudo_execute("cat %s" % self.tcpdumps_pid_file, parse=shell.RAW).decode().strip()
            pids = shell.sudo_execute("ps ax --no-headers", parse=shell.LINES)
            for line in pids:
                if tcpdumps_pid in line:
                    return True
        return False

    @synchronized(tcpdumps_settings_lock)
    def GET(self):
        """
        Return tcpdumps settings.
        """
        tcpdumps_settings = {'max_size': None,
                             'max_number': None,
                             'total_size': None,
                             'interfaces': None,
                             'is_running': None}
        utils.read_config(self.tcpdumps_config_file, tcpdumps_settings)

        interfaces = []
        if version.RELEASE_TYPE == "apc":
            interfaces.append("eth0.1")
        else:
            for name in network.get_all_interfaces():
                interfaces.append(name)
        tcpdumps_settings['interfaces'] = interfaces

        tcpdumps_settings['total_size'] = self.get_available_space()
        tcpdumps_settings['is_running'] = self.is_running()

        return json.dumps(tcpdumps_settings)

    @synchronized(tcpdumps_settings_lock)
    def PUT(self):
        """
        Starts tcpdumps using the specified parameters.

        max_number -- maximum number of tcpdump files at any point in time.
        max_size   -- maximum size (MB) of a tcpdump file.
        interface  -- interface being listened.
        """

        max_number = str(web.input(max_number=None).max_number)
        max_size = str(web.input(max_size=None).max_size)
        interface = str(web.input(interface=None).interface)

        if max_number and max_size and interface:
            tcpdumps_settings = {"max_size": max_size, "max_number": max_number}
            utils.update_config(self.tcpdumps_config_file, tcpdumps_settings)
            shell.sudo_call("%s stop" % (self.tcpdumps_script))
            shell.sudo_call("%s start %s %s %s" % (self.tcpdumps_script, interface, max_size, max_number))
            if interface == "all":
                audit.log(i18n.custom_gettext("Started packet capture on all interfaces"))
            else:
                audit.log(i18n.custom_gettext("Started packet capture on %s") % interface)
        else:
            log.error(TAG, "Invalid tcpdumps parameters")

    @synchronized(tcpdumps_settings_lock)
    def DELETE(self):
        """
        Stops tcpdumps.
        """
        shell.sudo_call("%s stop" % self.tcpdumps_script)
        audit.log(i18n.custom_gettext("Stopped packet capture"))


class DebugSettingsResource(object):
    """
    REST resource to manage debug settings for vmpro (and possible other services).

    Methods:

    GET     -- get the current debug setting
    POST    -- update debug level
    """

    @synchronized(debug_settings_lock)
    def GET(self):
        """
        Return JSON object with the value of debug (filter) level for voicemail.

        Error codes:

        404 -- Voicemail is not installed
        400 -- error reading Voicemail debug configuration file
        """
        if 'voicemail' in services_rest_api.services_manager.services:
            debug_settings = {'Filter Level': None}
            try:
                config_file = configuration.SHARED_CONFIGURATION['debug']['vmpro_debug_config']
                utils.read_config(config_file, debug_settings)
            except (IOError, OSError):
                log.exception(TAG, 'error while reading vmpro debug configuration file: %s' % config_file)
                web.ctx.status = '400'
            else:
                web.header("Content-Type","application/json")
                return json.dumps(debug_settings, default=utils.encode_json)
        else:
            web.ctx.status = '404'

    @synchronized(debug_settings_lock)
    def PUT(self):
        """
        Update debug (filter) level for voicemail.

        Input:

        debug_level -- the new debug level value

        Error codes:

        404 -- either the value could not be written to the voicemail debug config file or
               the value is not correct (wrong format or not one of the accepted values)
        """
        input_debug_level = web.input(debug_level=None).debug_level
        try:
            debug_level = int(input_debug_level)
            config_file = configuration.SHARED_CONFIGURATION['debug']['vmpro_debug_config']
            if debug_level in [0, 1, 2, 3, 5, 9]:
                debug_settings = {'Filter Level': debug_level}
                shell.sudo_call("chmod a+w %s" % config_file)
                utils.update_config(config_file, debug_settings)
                audit.log(i18n.custom_gettext('set Voicemail debug level to %s') % debug_level)
            else:
                # invalid value
                log.exception(TAG, 'wrong value for vmpro debug level: %s' % debug_level)
                web.ctx.status = '400'
        except (IOError, OSError):
            log.exception(TAG, 'unable to save vmpro debug level to file: %s' % config_file)
            web.ctx.status = '400'
        except (TypeError, ValueError):
            log.exception(TAG, 'wrong value for vmpro debug level: %s' % input_debug_level)
            web.ctx.status = '400'


class DebugCsipoResource(object):
    """
    REST resource to manage debug settings for vmpro (and possible other services).

    Methods:

    GET     -- get the current debug setting
    POST    -- update debug level
    """

    @synchronized(debug_csipo_lock)
    def GET(self):
        """
        Return JSON object with the value of debug (filter) level for CSIPO.

        Error codes:

        404 -- CSIPO is not installed
        400 -- error reading CSIPO debug configuration file
        """
        if 'csipo' in services_rest_api.services_manager.services:
            debug_settings = {'log.level': None}
            config_file = configuration.SHARED_CONFIGURATION['debug']['csipo_debug_config']
            try:
                utils.read_config(config_file, debug_settings)
            except (IOError, OSError):
                log.exception(TAG, 'error while reading CSIPO debug configuration file: %s' % config_file)
                web.ctx.status = '400'
            else:
                web.header("Content-Type","application/json")
                csipo_settings = {"log_level": debug_settings["log.level"]}
                return json.dumps(csipo_settings, default=utils.encode_json)
        else:
            web.ctx.status = '404'


    @synchronized(debug_csipo_lock)
    def PUT(self):
        """
        Update debug (filter) level for voicemail.

        Input:

        debug_level -- the new debug level value

        Error codes:

        400 -- either the value could not be written to the CSIPO debug config file or
               the value is not correct (wrong format or not one of the accepted values)
        """
        input_debug_level = web.input(debug_level=None).debug_level
        config_file = configuration.SHARED_CONFIGURATION['debug']['csipo_debug_config']
        try:
            if input_debug_level in ["DEBUG", "INFO", "WARN", "ERROR", "FATAL"]:
                debug_settings = {'log.level': input_debug_level}
                shell.sudo_call("chmod o+w %s" % config_file)
                utils.update_config(config_file, debug_settings)
                shell.sudo_call("chmod o-w %s" % config_file)
                shell.sudo_call("chmod g-w %s" % config_file)
                if input_debug_level == "DEBUG":
                    audit.log(i18n.custom_gettext('set Contact Recorder debug level to \"Debug\"'))
                elif input_debug_level == "INFO":
                    audit.log(i18n.custom_gettext('set Contact Recorder debug level to \"Information\"'))
                elif input_debug_level == "WARN":
                    audit.log(i18n.custom_gettext('set Contact Recorder debug level to \"Warning\"'))
                elif input_debug_level == "ERROR":
                    audit.log(i18n.custom_gettext('set Contact Recorder debug level to \"Error\"'))
                elif input_debug_level == "FATAL":
                    audit.log(i18n.custom_gettext('set Contact Recorder debug level to \"Fatal\"'))
            else:
                # invalid value
                log.exception(TAG, 'wrong value for CSIPO debug level: %s' % input_debug_level)
                web.ctx.status = '400'
        except (IOError, OSError):
            log.exception(TAG, 'unable to save CSIPO debug level to file: %s' % config_file)
            web.ctx.status = '400'

class DebugIraResource(object):
    """
    REST resource to manage debug settings for ira(and possible other services).

    Methods:

    GET     -- get the current debug setting
    POST    -- update debug level
    """

    @synchronized(debug_ira_lock)
    def GET(self):
        """
        Return JSON object with the value of debug (filter) level for CSIPO.

        Error codes:

        404 -- CSIPO is not installed
        400 -- error reading CSIPO debug configuration file
        """
        if 'ipora' in services_rest_api.services_manager.services:
            debug_settings = {'log4j.logger.com.avaya': None}
            config_file = configuration.SHARED_CONFIGURATION['debug']['ira_debug_config']
            try:
                utils.read_config(config_file, debug_settings)
            except (IOError, OSError):
                log.exception(TAG, 'error while reading Ira debug configuration file: %s' % config_file)
                web.ctx.status = '400'
            else:
                web.header("Content-Type","application/json")
                ira_settings = {"log_level": debug_settings["log4j.logger.com.avaya"]}
                return json.dumps(ira_settings, default=utils.encode_json)
        else:
            web.ctx.status = '404'



    @synchronized(debug_ira_lock)
    def PUT(self):
        """
        Update debug (filter) level for voicemail.

        Input:

        debug_level -- the new debug level value

        Error codes:

        400 -- either the value could not be written to the CSIPO debug config file or
               the value is not correct (wrong format or not one of the accepted values)
        """
        input_debug_level = web.input(debug_level=None).debug_level
        config_file = configuration.SHARED_CONFIGURATION['debug']['ira_debug_config']
        try:
            if input_debug_level in ["DEBUG", "INFO", "WARN", "ERROR", "FATAL"]:
                debug_settings = {'log4j.logger.com.avaya': input_debug_level}
                shell.sudo_call("chmod o+w %s" % config_file)
                utils.update_config(config_file, debug_settings)
                shell.sudo_call("chmod o-w %s" % config_file)
                shell.sudo_call("chmod g-w %s" % config_file)
                if input_debug_level == "DEBUG":
                    audit.log(i18n.custom_gettext('set Ira debug level to \"Debug\"'))
                elif input_debug_level == "INFO":
                    audit.log(i18n.custom_gettext('set Ira debug level to \"Information\"'))
                elif input_debug_level == "WARN":
                    audit.log(i18n.custom_gettext('set Ira debug level to \"Warning\"'))
                elif input_debug_level == "ERROR":
                    audit.log(i18n.custom_gettext('set Ira debug level to \"Error\"'))
                elif input_debug_level == "FATAL":
                    audit.log(i18n.custom_gettext('set Ira debug level to \"Fatal\"'))
            else:
                # invalid value
                log.exception(TAG, 'wrong value for IRA debug level: %s' % input_debug_level)
                web.ctx.status = '400'
        except (IOError, OSError):
            log.exception(TAG, 'unable to save IRA debug level to file: %s' % config_file)
            web.ctx.status = '400'

