/*
 * Copyright 2010 Avaya Inc. All Rights Reserved.
 */

var TOTAL_MEM = 0;
$(document).ready(function () {

    var home = {};

    home.servicesTableBuilt = false;
    home.notificationsTableBuilt = false;
    home.systemFieldsetBuilt = false;
    home.showOptionalServices = true;

    home.startButtonText        = $("#js_start").text();
    home.stopButtonText         = $("#js_stop").text();
    home.forceStopButtonText    = $("#js_forcestop").text();

    home.tableResizeHandler = null;
    home.fixedHeaderTables = [];

    home.addFixedTableHeader = function(tableId) {
        home.fixedHeaderTables.push(tableId);
    };

    /**
     * Starts or stops a service depending on its status
     * and refreshes the services tables if necessary.
     *
     * @param serviceID       ID of the service to start or stop
     * @param updateTable     true to update services table
     */
    home.startStopService = function (serviceID, updateTable) {
        var run_info_state;
        for (var i = 0; i < home.servicesStatus.length; i++) {
            if (serviceID == home.servicesStatus[i].serviceId) {
                run_info_state = home.servicesStatus[i].serviceStatus;
                break;
            }
        }
        var queryString = "name=" + serviceID + "&action=";
        var action;
        switch(run_info_state) {
            case ServiceState.STARTING:
                action = "forcestop";
                break;
            case ServiceState.RUNNING:
                action = "stop";
                break;
            case ServiceState.STOPPING:
                action = "forcestop";
                break;
            case ServiceState.STOPPED:
                action = "start";
                break;
            default:
                action = "start";
        }
        queryString += action;

        $.ajax({
            type: 'PUT',
            url : "/api/service",
            cache: false,
            data: queryString,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                globals.functions.enableButton(serviceID + "Button", true);
            },
            beforeSend: function (){
                globals.functions.enableButton(serviceID + "Button", false);
            },
            complete: function (xhr) {
                globals.functions.enableButton(serviceID + "Button", true);
                switch (xhr.status) {
                    case 200:
                        if (updateTable) {
                            home.updateServicesTable();
                        }
                        break;
                    case 401:
                    case 404:
                    default:
                        $("#" + serviceID + "Label").removeClass("orangeText");
                        $("#" + serviceID + "Label").text($("#js_stopped").text());
                        $("#" + serviceID + "Label").addClass("accentRedText");
                        break;
                }
            }
        });
    };

    /**
     * Starts or stops one or more services and refreshes the
     * services table after the last start/stop request.
     *
     * @param servicesArray    services to start or stop
     *                         expected data format:
     *                              [{serviceName: "name1", serviceStatus:"status1", displayName:"display1"}]
     */
    home.startStopMultipleServices = function (servicesArray) {
        var doRefresh = false;
        for (var i = 0; i < servicesArray.length; i++) {
            if (i == servicesArray.length-1) {
                doRefresh = true;
            }
            var currentService = servicesArray[i];
            home.startStopService(currentService.serviceId, doRefresh);
        }
    };

    /**
     * Gets the services information and builds the services table and
     * add click events on the services fieldset buttons and links.
     */
    home.buildServicesTable = function () {
        $.manageAjax.add("servicesAjaxQueue" , {
            type: 'GET',
            url : "/api/services",
            dataType: "json",
            cache: false,
            timeout: globals.variables.GLOBAL_AJAX_TIMEOUT,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                $("#servicesFieldset").unmask();
            },
            beforeSend: function () {
                $("#servicesFieldset").mask($("#js_loading").text());
            },
            success : function (data) {
                home.servicesStatus = [];

                $("#servicesList").empty();

                var servicesHTML = "";
                var optionalServicesHTML = "";
                for (var serviceID in data) {
                    var currentService = data[serviceID];
                    if (currentService.control_supported) {
                        if (!globals.variables.goldEdition && currentService.gold_edition_service) {
                            continue;
                        }
                        if (!currentService.optional_service) {
                            servicesHTML += home.addToServicesTable(currentService, serviceID);
                        } else {
                            optionalServicesHTML += home.addToServicesTable(currentService, serviceID);
                        }
                    }
                }

                var optionalServicesShowHide = "";
                optionalServicesShowHide += "<tr id='showHideOptionalRows'>";
                optionalServicesShowHide += "<td id='showHideOptionalRowsIcon' class='hideOptionalRowsBg'></td>";
                optionalServicesShowHide += "<td colspan='5'>";
                optionalServicesShowHide += "<label id='showHideOptionalServicesLabel'>";
                optionalServicesShowHide += $("#js_showOptionalServices").text();
                optionalServicesShowHide += "</label>";
                optionalServicesShowHide += "</td>";
                optionalServicesShowHide += "</tr>";

                $("#servicesList").append(servicesHTML);
                $("#servicesList").append(optionalServicesShowHide);
                $("#servicesList").append(optionalServicesHTML);
                home.formatVersion(data);
                home.servicesTableBuilt = true;

                $("#servicesList a").click( function (e) {
                    e.preventDefault();
                    home.openGraphModal(data[this.name]);
                });

                $("#servicesList button").click( function () {
                    home.startStopService(this.name, true);
                });

                $("#servicesList :checkbox").click( function () {
                    home.changeStartAtStartup(this.name, this.checked);
                });

                $("#showHideOptionalRows").click( function () {
                    if (home.showOptionalServices) {
                        $("#showHideOptionalServicesLabel").text($("#js_hideOptionalServices").text());
                        $("#showHideOptionalRowsIcon").removeClass("hideOptionalRowsBg").addClass("showOptionalRowsBg");
                        $("#servicesList .optionalServiceRow").show();
                    } else {
                        $("#showHideOptionalServicesLabel").text($("#js_showOptionalServices").text());
                        $("#showHideOptionalRowsIcon").removeClass("showOptionalRowsBg").addClass("hideOptionalRowsBg");
                        $("#servicesList .optionalServiceRow").hide();
                    }
                    home.showOptionalServices = !home.showOptionalServices;
                });

                $("#startAllButton").click (function() {
                    var currentService;
                    var stoppedServicesArray = [];
                    var doRefresh = false;
                    for (var i = 0; i < home.servicesStatus.length; i++) {
                        currentService = home.servicesStatus[i];
                        if (currentService.serviceStatus == ServiceState.STOPPED) {
                            stoppedServicesArray.push(currentService);
                        }
                    }
                    home.startStopMultipleServices(stoppedServicesArray);
                });

                $("#stopAllButton").click (function() {
                    var currentService;
                    var runningServicesArray = [];
                    var doRefresh = false;
                    for (var i = 0; i < home.servicesStatus.length; i++) {
                        currentService = home.servicesStatus[i];
                        if (currentService.serviceStatus == ServiceState.RUNNING) {
                            runningServicesArray.push(currentService);
                        }
                    }
                    home.startStopMultipleServices(runningServicesArray);
                });
            },
            complete: function () {
                home.adjustLayout();
                $("#servicesFieldset").unmask();
            }
        });
    };

    /**
     * Composes the html of a Services table row
     *
     * @param currentService    The service json object.
     * @param serviceID         The service ID.
     * @returns {string}        Composed HTML string.
     */
    home.addToServicesTable = function (currentService, serviceID) {

        var startStopInfo = {
            serviceId: currentService.id,
            serviceName: currentService.service_name,
            serviceStatus: currentService.run_info.state,
            displayName:currentService.display_name
        };
        home.servicesStatus.push(startStopInfo);

        var buttonString = "";
        var checkboxString = "";
        var memCpuUsageString = "";
        var statusString;
        var buttonText = "";
        var statusClass = "";

        switch (currentService.run_info.state) {
            case ServiceState.STARTING:
                statusClass = "svcBusy";
                statusString = $("#js_starting").text();
                buttonText = home.forceStopButtonText;
                break;
            case ServiceState.RUNNING:
                statusClass = "svcOn";
                statusString = $("#js_running").text();
                buttonText = home.stopButtonText;
                break;
            case ServiceState.STOPPING:
                statusClass = "svcBusy";
                statusString = $("#js_stopping").text();
                buttonText = home.forceStopButtonText;
                break;
            case ServiceState.STOPPED:
                statusClass = "svcOff";
                statusString = $("#js_stopped").text();
                buttonText = home.startButtonText;
                break;
            case ServiceState.UNKNOWN:
                statusClass = "svcUnknown";
                statusString = $("#js_unknown").text();
                buttonText = home.startButtonText;
                break;
        }
        buttonString = "<button class='avayaWebcontrolButton' id='" + serviceID + "Button' " +
            "name='" + currentService.id + "'>" + buttonText + "</button>";

        if (currentService.chkconfig_supported) {
            if (currentService.run_at_startup) {
                checkboxString = "<input type='checkbox' id='" + currentService.id + "Checkbox' " +
                    "name='" + currentService.id + "' checked='checked'/>";
            } else {
                checkboxString = "<input type='checkbox' id='" + currentService.id + "Checkbox' " +
                    "name='" + currentService.id + "'/>";
            }
        }

        var memUsage;
        if (currentService.run_info.mem == null) {
            memUsage = "0" + $("#js_K").text();
        } else {
            memUsage = currentService.run_info.mem + $("#js_K").text();
        }
        var cpuUsage;
        if (currentService.run_info.cpu == null) {
            cpuUsage = "0 %";
        } else {
            cpuUsage = currentService.run_info.cpu + $("#js_percentage").text();
        }
        memCpuUsageString = "<em>" + $("#js_usageMemCPU").text() + "</em><a href='' id='" + serviceID + "Link' " + "name='" + serviceID + "'>"
            + memUsage + " / " + cpuUsage + "</a>";

        var upTimeString;
        if (currentService.run_info.uptime == null) {
            upTimeString = "<em class='hiddenElement'>" + $("#js_upTime").text() + "</em><label id='" + currentService.id + "UpTimeLabel'>" + statusString + "</label>";
        } else {
            upTimeString = "<em>" + $("#js_upTime").text() + "</em><label id='" + currentService.id + "UpTimeLabel'>" +
                currentService.run_info.uptime + "</label>";
        }

        var versionString = "<label id='"+ currentService.id +"versionLabel'>"
            + currentService.version + "</label>";

        var serviceHTML = "";
        if (currentService.optional_service) {
            serviceHTML += "<tr class='optionalServiceRow hiddenElement " + statusClass + "' id='service-" + currentService.id + "'>";
        } else {
            serviceHTML += "<tr class='" + statusClass + "' id='service-" + currentService.id + "'>";
        }
        serviceHTML += "<td class='svcAuto'>" + checkboxString + "</td>";
        serviceHTML += "<td class='svcStatus'><em title='" + statusString + "'></em></td>";
        serviceHTML += "<td class='svcNameVer'><strong>" + currentService.display_name + "</strong><em>" + versionString + "</em></td>";
        serviceHTML += "<td class='svcUptime'>" + upTimeString + "</td>";
        serviceHTML += "<td class='svcUsage'>" + memCpuUsageString + "</td>";
        serviceHTML += "<td class='svcAction'>" + buttonString + "</td>";
        serviceHTML += "</tr>";

        return serviceHTML;

    };

    /**
     * Rewrites the version in string format if possible.
     *
     * @param data  Object containing services information.
     */
    home.formatVersion  = function (data) {
        for (var id in data) {
            var service = data[id];
            if (service.control_supported) {
                var version = globals.functions.parseVersion(service.version).versionNumber;
                var build = globals.functions.parseVersion(service.version).buildNumber;
                if (version != null && build != null) {
                    var labelId = service.id + "versionLabel";
                    globals.functions.composeText(labelId, "js_versionFormat", [version, build]);
                }
            }
        }
    };

    /**
     * Changes the run_at_startup property of a service depending on the corresponding checkbox.
     *
     * @param service   service name
     * @param checked   checkbox checked property (true or false)
     */
    home.changeStartAtStartup = function (service, checked) {
        var run_at_startup = "off";
        if (checked) {
            run_at_startup = "on";
        }
        $.ajax({
            type: "PUT",
            url : "/api/service/run_at_startup",
            data: "name=" + service + "&run_at_startup=" + run_at_startup
        });
    };

    /**
     * Opens a modal window that displays a service graphs for memory usage and CPU usage history.
     *
     * @param service  service object for which the performance graphs are displayed
     */
    home.openGraphModal = function (service) {
        $.ajax({
            type: 'GET',
            url : "/api/service/performance?name=" + service.id,
            dataType: "json",
            cache: false,
            success : function (data) {
                var cpuUsageData = [];
                var memUsageData = [];
                for (var i=0; i<data.samples.length; i++) {
                    var timeLabel = data.samples[i].timestamp;
                    cpuUsageData[i] = [timeLabel, data.samples[i].cpu_usage];
                    memUsageData[i] = [timeLabel, (data.samples[i].mem_usage * 100) / TOTAL_MEM];
                }
                $('#graphModal').modal({
                    closeHTML: "<a href='' title='" + $("#js_close").text() + "'></a>",
                    overlayId: 'modal-overlay',
                    containerId: 'graphModal-container',
                    onShow: function () {
                        $("#modalHeader").text(service.display_name);
                        $("#closeModal").click(function () {
                            $.modal.close();
                        });

                        try {
                            $(".memoryUsagePopupChartDiv").remove();
                            $(".cpuUsagePopupChartDiv").remove();
                        } catch(e) {}
                        $("#memoryUsagePie").append("<div id='memoryUsagePopupChartDiv' " +
                                "class='memoryUsagePopupChartDiv' width='100%'></div>");
                        $("#cpuUsageGraph").append("<div id='cpuUsagePopupChartDiv' " +
                                "class='cpuUsagePopupChartDiv' width='100%'></div>");

                        try {
                            globals.functions.drawChart(
                                'cpuUsagePopupChartDiv',
                                $("#js_cpuUsageHistory").text(),
                                $("#js_cpuUsageHistory").text(),
                                [cpuUsageData],
                                $("#js_time").text(),
                                $("#js_usage").text(),
                                true
                            );
                        } catch(e) {
                            if (cpuUsageData.length != 1) {
                                $("#cpuUsagePopupChartDiv").addClass("blackOnWhite");
                                $("#cpuUsagePopupChartDiv").append("<br/><br/><br/><label>" +
                                        + $("#js_noCPUUsageHistoryDataAvailable").text() + "</label>");
                            }
                        }
                        try {
                            globals.functions.drawChart(
                                'memoryUsagePopupChartDiv',
                                $("#js_memoryUsageHistory").text(),
                                $("#js_memoryUsageHistory").text(),
                                [memUsageData],
                                $("#js_time").text(),
                                $("#js_usage").text(),
                                true
                            );
                        } catch(e) {
                            if (memUsageData.length != 1) {
                                $("#memoryUsagePopupChartDiv").addClass("blackOnWhite");
                                $("#memoryUsagePopupChartDiv").append("<br/><br/><br/><label>" +
                                        + $("#js_noMemoryUsageHistoryDataAvailable").text() + "</label>");
                            }
                        }
                    }
                });
            }
        });
    };

    /**
     * Gets the services information and updates services table.
     */
    home.updateServicesTable = function () {
        $.manageAjax.add("servicesAjaxQueue" , {
            type: 'GET',
            url : "/api/services",
            dataType: "json",
            cache: false,
            timeout: globals.variables.GLOBAL_AJAX_TIMEOUT,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
            },
            success : function (data) {
                home.servicesStatus = [];
                for (var serviceID in data) {
                    var currentService = data[serviceID];
                    if (currentService.control_supported) {

                        if (!globals.variables.goldEdition && currentService.gold_edition_service) {
                            continue;
                        }

                        home.servicesStatus.push({
                            serviceId: currentService.id,
                            serviceName: currentService.service_name,
                            serviceStatus: currentService.run_info.state,
                            displayName:currentService.display_name
                        });

                        var statusString = "";
                        var buttonText = "";
                        var statusClass = "";
                        switch (currentService.run_info.state) {
                            case ServiceState.STARTING:
                                statusClass = "svcBusy";
                                statusString = $("#js_starting").text();
                                buttonText = home.forceStopButtonText;
                                break;
                            case ServiceState.RUNNING:
                                statusClass = "svcOn";
                                statusString = $("#js_running").text();
                                buttonText = home.stopButtonText;
                                break;
                            case ServiceState.STOPPING:
                                statusClass = "svcBusy";
                                statusString = $("#js_stopping").text();
                                buttonText = home.forceStopButtonText;
                                break;
                            case ServiceState.STOPPED:
                                statusClass = "svcOff";
                                statusString = $("#js_stopped").text();
                                buttonText = home.startButtonText;
                                break;
                            case ServiceState.UNKNOWN:
                                statusClass = "svcUnknown";
                                statusString = $("#js_unknown").text();
                                buttonText = home.startButtonText;
                                break;
                        }
                        if (currentService.optional_service) {
                            statusClass += " optionalServiceRow";
                            if (home.showOptionalServices) {
                                statusClass += " hiddenElement";
                            }
                        }

                        $("#service-" + currentService.id).attr("class", statusClass);
                        $("#service-" + currentService.id + " .svcStatus em").attr("title", statusString);
                        $("#" + currentService.id + "Button").text(buttonText);

                        var memUsage;
                        if (currentService.run_info.mem == null) {
                            memUsage = "0" + $("#js_K").text();
                        } else {
                            memUsage = currentService.run_info.mem + $("#js_K").text();
                        }
                        var cpuUsage;
                        if (currentService.run_info.cpu == null) {
                            cpuUsage = "0 %";
                        } else {
                            cpuUsage = currentService.run_info.cpu + $("#js_percentage").text();
                        }
                        $("#" + serviceID + "Link").html(memUsage + " / " + cpuUsage);

                        var upTime;
                        if (currentService.run_info.uptime == null) {
                            upTime = statusString;
                            $("#service-" + currentService.id + " .svcUptime em").addClass("hiddenElement");
                        } else {
                            upTime = currentService.run_info.uptime;
                            $("#service-" + currentService.id + " .svcUptime em").removeClass("hiddenElement");
                        }
                        $("#" + currentService.id + "UpTimeLabel").text(upTime);
                        $("#" + currentService.id + "versionLabel").text(currentService.version);
                        home.formatVersion(data);
                    }
                }
            }
        });
    };

    /**
    *   Builds the table from the notifications fieldset.
     *
    *   @param sortUpdate   false when building the table
     *                      true when updating the table
    */
    home.buildNotificationsTable = function (sortUpdate) {
        $.manageAjax.add("notificationsAjaxQueue" , {
            type: 'GET',
            url : "/api/system/syslog",
            data: "keyword=IPOfficeWatchdog,IPOLError",
            dataType: "json",
            cache: false,
            timeout: globals.variables.GLOBAL_AJAX_TIMEOUT,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                if (!sortUpdate) {
                    $("#notificationsFieldset").unmask();
                }
            },
            beforeSend: function () {
                if (!sortUpdate) {
                    $("#notificationsFieldset").mask($("#js_loading").text());
                }
            },
            success : function (data) {
                $(".notificationsTbody").remove();
                var notificationsString = "<tbody class='notificationsTbody'>";
                for (var i = 0; i < data.length; i++) {
                    notificationsString = notificationsString + "<tr>";
                    notificationsString = notificationsString + "<td>";
                    notificationsString = notificationsString + data[i].timestamp;
                    notificationsString = notificationsString + "</td>";
                    notificationsString = notificationsString + "<td>";
                    notificationsString = notificationsString + data[i].message;
                    notificationsString = notificationsString + "</td>";
                    notificationsString = notificationsString + "</tr>";
                }
                notificationsString = notificationsString + "</tbody>";
                $("#notificationsTable").append(notificationsString);
                if (sortUpdate) {
                    try {
                        $("#notificationsTable").trigger("update");
                        $("#notificationsTable").trigger("sorton",[[[0,1]]]);
                    } catch (e) {
                    }
                } else {
                    try {
                        $("#notificationsTable").tablesorter({
                            sortList: [[0,1]],
                            widgets: ['zebra'],
                            headers: {
                                0: {
                                    sorter: "wcpDate"
                                }
                            }
                        });
                        $("#notificationsTable thead").height($("#notificationsTable thead").height());
                        globals.functions.makeFixedTableHeader("notificationsTable");
                        home.addFixedTableHeader("notificationsTable");
                    } catch (e) {
                    }
                }
                globals.functions.showNoDataDiv("notificationsTableDiv", "notificationsTable",
                        data.length, $("#js_noNotificationsAvailable").text());
                home.notificationsTableBuilt = true;

                if (typeof globals.variables.REFERRER !== 'undefined') {
                    if (data.length > 5) {
                        $("#notificationsFieldset .groupContainer").height(150);
                    } else if (data.length > 0) {
                        $("#notificationsFieldset .groupContainer").height(100);
                    } else {
                        $("#notificationsFieldset .groupContainer").height(35);
                    }
                }
            },
            complete: function () {
                if (!sortUpdate) {
                    $("#notificationsFieldset").unmask();
                }
            }
        });
    };

    /**
     * Retrieve logon info and show it.
     */
    home.updateLogonInfo = function () {
        $.manageAjax.add("systemAjaxQueue" , {
            type: "GET",
            url: "/api/log/login",
            dataType: "json",
            cache: false,
            timeout: globals.variables.GLOBAL_AJAX_TIMEOUT,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                home.displayLogonInfo("-", "-", true);
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
            },
            success: function (data) {
                home.displayLogonInfo("" + data.last_successful_login, "" + data.failed_login_attempts, "" + true);
            }
        });
    };

    home.displayLogonInfo = function(lastLogon, logonAttempts, defaultPasswordChanged) {
        $("#logonLastInfoLabel").text(lastLogon);
        $("#logonAttemptsInfoLabel").text(logonAttempts);
        if (defaultPasswordChanged.toString().toLowerCase() == "false") {
            $("#changePasswordWarning").removeClass("hiddenElement");
        }
    };

    /**
     * Gets the system related information and updates or redraws
     * the System fieldset graphs and labels.
     */
    home.updateSystem = function () {
        $.manageAjax.add("systemAjaxQueue" , {
            type: "GET",
            url: "/api/system/status",
            dataType: "json",
            cache: false,
            timeout: globals.variables.GLOBAL_AJAX_TIMEOUT,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
            },
            success: function (data) {
                globals.variables.historyChartData = data.cpu.usage;
                globals.variables.memoryUsagePlotData = data.mem;
                globals.variables.diskUsagePlotData = data.disk;
                home.displayOSinfo(data);

                globals.functions.redrawMemoryUsagePie();
                globals.functions.redrawDiskUsagePie();
                globals.functions.redrawHistoryChart();

                // load extra system information from 3rd party vendor
                home.getVendorSystemInfo();
            }
        });
    };

    /**
     * Gets the system information and builds the System fieldset graphs and labels.
     */
    home.buildSystem = function () {
        var systemReady = false;
        var hwReady = false;

        $.manageAjax.add("systemAjaxQueue" , {
            type: "GET",
            url: "/api/system/status",
            dataType: "json",
            cache: false,
            timeout: globals.variables.GLOBAL_AJAX_TIMEOUT,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                systemReady = true;
                if (systemReady && hwReady) {
                    $("#systemContent").unmask();
                }
            },
            beforeSend: function () {
                $("#systemContent").mask($("#js_loading").text());
            },
            success: function (data) {
                globals.variables.historyChartData = data.cpu.usage;
                globals.variables.memoryUsagePlotData = data.mem;
                globals.variables.diskUsagePlotData = data.disk;
                home.displayOSinfo(data);
                home.alignInfoLabels();

                globals.functions.redrawHistoryChart();
                globals.functions.redrawMemoryUsagePie();
                globals.functions.redrawDiskUsagePie();
                TOTAL_MEM = data.mem.used + data.mem.free;

                systemReady = true;
                if (systemReady && hwReady) {
                    // load extra system information from 3rd party vendor
                    home.getVendorSystemInfo();
                    home.systemFieldsetBuilt = true;
                }
            },
            complete: function (xhr) {                
                if (xhr.status != 200) {
                    systemReady = true;
                    if (systemReady && hwReady) {
                        $("#systemContent").unmask();
                    }
                }
            }
        });

        $.ajax({
            type: "GET",
            url: "/api/system/hwinfo",
            dataType: "json",
            cache: false,
            timeout: globals.variables.GLOBAL_AJAX_TIMEOUT,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                hwReady = true;
                if (systemReady && hwReady) {
                    $("#systemContent").unmask();
                }
            },
            beforeSend: function () {
                $("#systemContent").mask($("#js_loading").text());
            },
            success: function (data) {
                home.displayHwInfo(data);
                hwReady = true;
                if (systemReady && hwReady) {
                    home.getVendorSystemInfo();
                    home.systemFieldsetBuilt = true;
                }
            },
            complete: function (xhr) {
                if (xhr.status != 200) {
                    hwReady = true;
                    if (systemReady && hwReady) {
                        $("#systemContent").unmask();
                    }
                }
            }
        });

        $("#shutdownRebootModalButtons button").unbind("click");
        $("#shutdownButton").click( function () {
            home.openShutdownRebootWarning("shutdown");
        });
        $("#rebootButton").click( function () {
            home.openShutdownRebootWarning("reboot");
        });
    };

    /**
     * Displays hardware info in the System widget.
     *
     * @param data
     */
    home.displayHwInfo = function (data) {

        $("#cpuModelLabel").text(data.cpu_model);
        $("#cpuMhzLabel").text(data.cpu_mhz);
        $("#cpuCoresLabel").text(data.cpu_cores);
        $("#diskTotalLabel").text(data.disk_total);
        $("#memTotalLabel").text(data.mem_total);

        if (data.disk_raid_levels.length > 0) {
            $("#diskRaidLevelsLabel").text(data.disk_raid_levels.join($("#js_comma").text() + " "));
        }

        if (data.disk_array_types.length > 0) {
            var arrayTypes = "";
            for (var i = 0; i < data.disk_array_types.length; i++) {
                arrayTypes += data.disk_array_types[i][0] + $("#js_colon").text() + " ";
                arrayTypes += data.disk_array_types[i][1];
                if (i != data.disk_array_types.length - 1) {
                    arrayTypes += $("#js_comma").text() + " ";
                }
            }
            $("#diskArrayTypesLabel").text(arrayTypes);
        }

        if (data.virtualised) {
            $("#virtualisedLabel").text($("#js_Yes").text());
        } else {
            $("#virtualisedLabel").text($("#js_No").text());
        }

    };

    /**
     * Opens a warning modal when the user attempts to reboot or shutdown the server
     *
     * @param action    Action attempted by the user ("reboot" or "shutdown")
     */
    home.openShutdownRebootWarning = function (action) {
        var restart = false;
        var warningString = "";
        if (action == "shutdown") {
            warningString = $("#js_shutdown").text();
        } else if (action == "reboot") {
            restart = true;
            warningString = $("#js_reboot").text();
        }
        warningString += "<br/>" + $("#js_redirectToLogin").text();
        warningString += "<br/>" + $("#js_continue").text();

        $("#shutdownRebootModalText").html(warningString);

        //button click events
        $("#shutdownRebootModalButtons button").unbind("click");
        $("#shutdownRebootModalNo").click(function () {
            $.modal.close();
        });
        $("#shutdownRebootModalYes").click(function () {
            $.modal.close();
            $.ajax({
                type: 'PUT',
                url: '/api/system/status',
                data: 'restart=' + restart,
                timeout: globals.variables.REBOOT_SHUTDOWN_TIMEOUT,
                success: function() {
                    $.ajax({
                        type: 'DELETE',
                        url: '/api/session',
                        complete: function(xhr, statusText) {
                            globals.functions.handleRedirection(globals.variables.REDIRECT_REBOOT_SHUTDOWN);
                        }
                    });
                }
            });
        });
        $('#shutdownRebootModal').modal({
            closeHTML: "<a href='' title='" + $("#js_close").text() + "'></a>",
            overlayId: 'modal-overlay',
            containerId: 'shutdownRebootModal-container'
        });

    };

    /**
     * Get extra system information from 3rd party vendor.
     */
    home.getVendorSystemInfo = function() {
        $.manageAjax.add("vendorAjaxQueue" , {
            type: 'GET',
            url: '/api/vendor/system_info',
            cache: false,
            timeout: globals.variables.GLOBAL_AJAX_TIMEOUT,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                $("#systemContent").unmask();
            },
            success: function(data) {
                if (! $.isEmptyObject(data)) {
                    home.buildVendorSystemInfoDiv(data);    
                } else {
                    $("#vendorInfoFieldset").addClass("hiddenElement");
                }
            },
            complete: function () {
                $("#systemContent").unmask();
            }
        });
    };

    /**
     * Builds and shows 3rd party vendor information
     *
     * @param data  vendor info  
     *              expected data format:
     *                  {vendor_name: "Vendor Name", system_info: [[key1, info1], [key2, info2], ...]}
     */
    home.buildVendorSystemInfoDiv = function (data) {
        $("#vendorInfoFieldsetLegend").text(data.vendor_name);
        $("#vendorInfoContent").empty();
        var vendorInfoString = "";
        for (var i = 0; i < data.system_info.length; i++) {
            vendorInfoString += "<div class='spacedLineDiv'>";
            vendorInfoString += "<div class='vendorLeftDiv'>" + data.system_info[i][0] + ": </div>";
            vendorInfoString += "<div class='vendorRightDiv'>" + data.system_info[i][1] + "</div>";
            vendorInfoString += "</div>";
        }
        $("#vendorInfoContent").append(vendorInfoString);
        $("#vendorInfoFieldset").removeClass("hiddenElement");
    };

    /**
     * Get and display license status.
     */
    home.checkLicense = function() {
        $.manageAjax.add("licenseAjaxQueue" , {
            url: '/api/services/license',
            cache: false,
            timeout: globals.variables.GLOBAL_AJAX_TIMEOUT,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
            },
            success: function(data) {
                if (! $.isEmptyObject(data)) {
                    // demo version
                    if (data.authenticate_message != null) {
                        $("#licenseNeededDiv").addClass("hiddenElement");
                        home.openLicenseModal(data.authenticate_message);
                    }
                    if (data.license_warning_message != null) {
                        $("#licenseNeededLabel").text(data.license_warning_message);
                        $("#licenseNeededDiv").show();
                    } else {
                        $("#licenseNeededDiv").hide();
                    }
                } else {
                    $("#licenseNeededDiv").addClass("hiddenElement");    
                }
            }
        });
    };

    /**
     * Opens a modal that requests the user to enter the license password
     *
     * @param message       The message to display in the window
     */
    home.openLicenseModal = function (message) {
        $("#licensePasswordTextfield").val("");
        $("#licenseModalText").html(message);
        $("#licenseModalHeader").text($("#js_authenticate").text());

        $('#licenseModalPassword').unbind("keydown");
        $('#licenseModalPassword').keydown( function(e) {
            if(e.keyCode == 13 || e.which == 13) {
                home.submitLicensePassword();
            }
        });

        //button click events
        $("#licenseModalButtons button").unbind("click");
        $("#licenseModalCancel").click(function () {
            $.modal.close();
        });
        $("#licenseModalOk").click(home.submitLicensePassword);
        $('#licenseModal').modal({
            closeHTML: "<a href='' title='" + $("#js_close").text() + "'></a>",
            overlayId: 'modal-overlay',
            containerId: 'licenseModal-container',
            escClose: true
        });

    };

    /**
     * Submits the license password
     * Closes the modal
     */
    home.submitLicensePassword = function () {
        var password = $("#licensePasswordTextfield").val();
        $.modal.close();
        $.ajax({
            type: 'PUT',
            url: '/api/services/license',
            data: "message=" + password,
            cache: false
        });
    };

    /**
     * Function that calls all the update functions if the current tab is home.
     * Removes the update interval if the current tab is not home.
     */
    updateTables = function () {
        if (globals.variables.currentTab == "home") {
            try {
                if (home.servicesTableBuilt) {
                    home.updateServicesTable();
                } else {
                    home.buildServicesTable();    
                }
            } catch(e) {
                try {
                    home.buildServicesTable();
                } catch (err) {
                    // ignored
                }
            }
            try {
                home.buildNotificationsTable(home.notificationsTableBuilt);
            } catch(e) {
                try {
                    home.buildNotificationsTable(false);
                } catch (err) {
                    // ignored
                }
            }
            try {
                if (home.systemFieldsetBuilt) {
                    home.updateSystem();
                } else {
                    home.buildSystem();    
                }
            } catch(e) {
                try {
                    home.buildSystem();
                } catch (err) {
                    // ignored
                }
            }
            try {
                home.updateLogonInfo();
            } catch(err) {
                // ignored
            }
            try {
                home.checkLicense();
            } catch (err) {
                // ignored
            }
        }
    };

    /**
     * Displays OS information.
     *
     * @param data  OS info data
     *              expected data format:
     *                  {cpu: { loadavg: [], usage:[["", ""],["", ""],...]},
     *                   disk: { free: "", total: "", used: ""},
     *                   mem: { free: "", total: "", used: ""},
     *                   os: { kernel: "", last_update: "", uptime: {days: "", hours: "", minutes: ""},
     *                                         server_time: "HH:MM", version: ""}}
     */
    home.displayOSinfo = function (data) {

        var uptimeMsg = "";
        if (data.os.uptime.days) {
            if (data.os.uptime.days == 1) {
                uptimeMsg = data.os.uptime.days + " " + $("#js_day").text() + " ";
            } else {
                uptimeMsg = data.os.uptime.days + " " + $("#js_days").text() + " ";
            }
        }
        if (data.os.uptime.hours) {
            if (data.os.uptime.hours == 1) {
                uptimeMsg += data.os.uptime.hours + " " + $("#js_hour").text() + " ";
            } else {
                uptimeMsg += data.os.uptime.hours + " " + $("#js_hours").text() + " ";
            }
        }
        if (data.os.uptime.minutes == 1) {
            uptimeMsg += data.os.uptime.minutes + " " + $("#js_minute").text();
        } else {
            uptimeMsg += data.os.uptime.minutes + " " + $("#js_minutes").text();
        }

        var loadAvgMsg = data.cpu.loadavg[0] + $("#js_oneMin").text()
                         + data.cpu.loadavg[1] + $("#js_fiveMin").text()
                         + data.cpu.loadavg[2] + $("#js_fifteenMin").text();

        try {
            var quotaFree = globals.functions.parseBytes(data.quota.free);
            var hddSize = globals.functions.parseBytes(data.quota.hdd_size);
            var quotaThreshold = globals.functions.parseBytes(data.quota.threshold);
            if (hddSize > quotaThreshold) {
                $("#quotaSizeLabel").text(globals.functions.bytesToSize(quotaFree, 0))
            } else {
                $("#quotaSizeLabel").text($("#js_noneLabel").text());
            }
        } catch (e) {
            // ignored
        }

        $("#systemOsInfoLabel").text(data.os.version);
        $("#systemKernelInfoLabel").text(data.os.kernel);
        $("#systemUpTimeInfoLabel").text(uptimeMsg);
        $("#serverTimeInfoLabel").text(data.os.server_time);
        $("#systemLoadAvgInfoLabel").text(loadAvgMsg);
        $("#serverTypeLabel").text(data.os.server_type);

        if(data.sysinv.inv_code != ""){
            $(".sysInventoryDataFields").removeClass("hiddenElement");
            $("#systemInvCodeInfoLabel").text(data.sysinv.inv_code);
            $("#systemTypeInfoLabel").text(data.sysinv.sys_type);
            $("#systemModelInformationInfoLabel").text(data.sysinv.model_info);
	        $("#systemSysSerialNumInfoLabel").text(data.sysinv.sys_serial_num);
	    }

    };

    /**
     * Aligns the Info labels if necessary
     */
    home.alignInfoLabels = function () {

        $(".vendorLeftDiv label").each( function() {
            var index = this.id.substring(0, this.id.indexOf("Label"));
            var leftLabel = $("#" + index + "Label");
            var rightLabel = $("#" + index + "InfoLabel");
            if (leftLabel.height() > rightLabel.height()) {
                $(rightLabel).css("position", "relative");
                $(rightLabel).css("top", leftLabel.height() - rightLabel.height());
            }
        });

    };

    home.adjustLayout = function () {
        if (typeof globals.variables.REFERRER === 'undefined') {
            $("#notificationsFieldset .groupHeader").height($("#servicesFieldset .groupHeader").height());
            $("#notificationsFieldset .groupContainer").height($("#servicesFieldset .groupContainer").height());
        }
    };

    /**
     * Calls the updateTables function each 15 seconds .
     */
    home.refreshTables = function () {
        globals.variables.homeTabInterval = setInterval("updateTables()", 30000);
    };

    home.chartWidth = 0;
    home.redrawCharts = function () {
        var w = $("#systemFieldset").width();
        if (w != home.chartWidth) {
            home.chartWidth = w;
            globals.functions.redrawHistoryChart();
            globals.functions.redrawMemoryUsagePie();
            globals.functions.redrawDiskUsagePie();
        }
    };

    /**
     * Create ajax queues for the requests that happen at an interval
     */
    home.queueProperties = {
        queue: "clear",
        abortOld: false,
        maxRequests: 1
    };
    $.manageAjax.create('servicesAjaxQueue', home.queueProperties);
    $.manageAjax.create('systemAjaxQueue', home.queueProperties);
    $.manageAjax.create('notificationsAjaxQueue', home.queueProperties);
    $.manageAjax.create('licenseAjaxQueue', home.queueProperties);
    $.manageAjax.create('vendorAjaxQueue', home.queueProperties);

    try {
        home.buildServicesTable();
    } catch (e) {
        // ignored
    }

    try {
        home.buildNotificationsTable(false);
    } catch (e) {
        // ignored
    }

    try {
        home.buildSystem();
    } catch (e) {
        // ignored
    }

    try {
        home.updateLogonInfo();
    } catch (e) {
        // ignored
    }

    try {
        home.checkLicense();
    } catch (e) {
        // ignored
    }

    if (typeof globals.variables.REFERRER !== 'undefined') {
        $("#notificationsFieldset .groupHeader").hide();
    }

    home.refreshTables();

//    $(window).resize(function() {
//        home.redrawCharts();
//    });

    home.adjustTableHeadersOnResize = function() {
        if (globals.variables.currentTab != "home") {
            $(window).unbind("resize", home.adjustTableHeadersOnResize);
            return;
        }
        home.redrawCharts();
        for (var tableIdx in home.fixedHeaderTables) {
            tableId = home.fixedHeaderTables[tableIdx];
            globals.functions.resizeFixedTableHeader(tableId);
        }
    }

    $(window).bind("resize", function() {
        if (home.tableResizeHandler) {
            clearTimeout(home.tableResizeHandler);
        }
        home.tableResizeHandler = setTimeout(home.adjustTableHeadersOnResize, 300);
    });

});
