/*
 * Copyright 2010 Avaya Inc. All Rights Reserved.
 */

$(document).ready(function () {
 
    var logs = {};
    logs.tab = "";

    logs.viewedAuditLogs = false;
    logs.viewedApplicationsLogs = false;
    logs.viewedDownloadLogs = false;
    logs.viewedDownloadDebug = false;
    logs.viewedSyslogViewer = false;

    logs.tableResizeHandler = null;
    logs.fixedHeaderTables = {};

    logs.addFixedTableHeader = function(tab, tableId) {
        if (logs.fixedHeaderTables[tab] === undefined) {
            logs.fixedHeaderTables[tab] = [];
        }
        logs.fixedHeaderTables[tab].push(tableId);
    };

    /**
     * Selects or unselects all checkboxes from a fieldset table when a button is pressed.
     *
     * @param fieldsetName  name of the fieldset
     * @param scope         scope of the button
     */
    logs.selectAll = function (fieldsetName, scope) {
        if (scope.innerHTML == $("#js_selectAll").text()) {
            $("#" + fieldsetName + "FilesTable :checkbox").attr('checked', true);
            scope.innerHTML = $("#js_unselectAll").text();
        } else {
            $("#" + fieldsetName + "FilesTable :checkbox").attr('checked', false);
            scope.innerHTML = $("#js_selectAll").text();
        }
    };

    /**
     * Build combobox for selected applications
     */
    logs.buildApplicationsLogSelect = function () {
        $.ajax({
            type: 'GET',
            url : "/api/services",
            dataType: "json",
            cache: false,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                $("#filterAppicationLog").unmask();
            },
            beforeSend: function () {
                $("#filterAppicationLog").mask($("#js_loading").text());
            },
            success : function (data) {
                var initialValue = $("#applicationsCombo").val();
                var options =  "";
                options += "<option value='all'>" + $("#js_All").text() + "</option>";
                for (var i in data) {
                    var service = data[i];
                    if (service.logging_supported) {
                        options += "<option value='" + service.id + "'>" + service.display_name + "</option>";
                    }
                }
                $("#applicationsCombo").html(options);
                if (initialValue) {
                    $("#applicationsCombo").val(initialValue);
                }
                $("#filterAppicationLog").unmask();
            },
            complete: function (xhr) {
                if (xhr.status != 200) {
                    $("#filterAppicationLog").unmask();
                }
            }
        });
    };

    /**
     * Build comboboxes for syslog filtering
     */
    logs.buildSyslogFilters = function () {
        $("#syslogTagCombo ." + globals.variables.releaseType + "Exclude").hide();
        $("#syslogTagCombo ." + globals.variables.serverType + "Exclude").hide();
        if (globals.variables.isCloud) {
            $("#syslogTagCombo .cloudExclude").hide();
        }
        $("#syslogLastCombo option").each( function (){
            if (this.value.length > 0) {
                globals.functions.composeText(this.id, "js_lastEntries", [this.value]);
            }
        });
        $.ajax({
            type: 'GET',
            url : "/api/log/syslog/hosts",
            dataType: "json",
            cache: false,
            beforeSend: function () {
                $("#filterSyslogDiv").mask($("#js_loading").text());
            },
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                $("#filterSyslogDiv").unmask();
            },
            success : function (data) {
                var hosts_array = data.hosts;
                $.ajax({
                    type: 'GET',
                    url : "/api/system/network",
                    dataType: "json",
                    cache: false,
                    beforeSend: function () {
                        $("#filterSyslogDiv").mask($("#js_loading").text());
                    },
                    success : function (data) {
                        var options =  "";
                        options += "<option value=''>" + $("#js_All").text() + "</option>";
                        for (var i = 0; i < hosts_array.length; i++) {
                            if (hosts_array[i] == "127.0.0.1") {
                                options += "<option value='" + hosts_array[i] + "'>" + data.interfaces[0].ipaddr
                                    + "</option>";
                            } else {
                                options += "<option value='" + hosts_array[i] + "'>" + hosts_array[i]+ "</option>";
                            }
                        }
                        $("#syslogHostCombo").html(options);
                    },
                    complete: function () {
                        $("#filterSyslogDiv").unmask();
                    }
                });
            }
        });
    };

    /**
     * Builds a string containing HTML code for the Application Log table rows.
     *
     * @param applicationLog   application log data from the device
     * @param rowNumber        number the row will have in the table
     *                         expected data format:
     *                              {timestamp: "11:30", message:"message1", application: "IPOffice", level:"1"}
     */
    logs.buildApplicationsLogsRows = function (applicationLog, rowNumber) {
        var returnString = "";
        returnString += "<tr><td>";
        returnString += applicationLog.application;
        returnString += "</td><td class='alignLeft'>";
        returnString += applicationLog.message;
        returnString += "</td></tr>";
        return returnString;
    };

    /**
     * Builds a string containing HTML code for the Audit Log table body.
     *
     * @param auditLog  audit log row data
     *                  expected data format:
     *                      [{timestamp: "11:30", user:"webcontrol", action:"action1"},...]
     */
    logs.buildAuditLogRows = function (auditLog) {
        $(".auditTbody").remove();
        var returnString = "<tbody class='auditTbody'>";
        for (var i = 0; i < auditLog.length; i++) {
            returnString = returnString + "<tr><td>";
            returnString = returnString + auditLog[i].timestamp;
            returnString = returnString + "</td><td>";
            returnString = returnString + auditLog[i].user;
            returnString = returnString + "</td><td>";
            returnString = returnString + auditLog[i].action;
            returnString = returnString + "</td></tr>";
        }
        returnString = returnString + "</tbody>";
        return returnString;
    };

    /**
     * Builds the applications log table depending on the comboboxes value.
     *
     * @param data  applications Log data.
     *              expected data format:
     *                  [{timestamp: "11:30", message:"message1", application: "IPOffice", level:"1"},...]
     */
    logs.buildApplicationsLogsTable = function (data) {
        var applicationsLogsTable = "";
        for (var i = 0; i < data.length; i++) {
            applicationsLogsTable += "<tr><td>";
            applicationsLogsTable += data[i].application;
            applicationsLogsTable += "</td><td class='alignLeft'>";
            applicationsLogsTable += data[i].message;
            applicationsLogsTable += "</td></tr>";
        }
        return applicationsLogsTable;
    };

    /**
     * Reloads the Applications Log table.
     */
    logs.reloadApplicationsTable = function () {
         $.ajax({
             type: "GET",
             url: "/api/log/application",
             dataType: "json",
             data: "id=" + $("#applicationsCombo").val(),
             cache: false,
             error: function (xmlHttpReq, textStatus, errorThrown) {
                 globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                 $("#applicationsLogsTableDiv").unmask();
             },
             beforeSend: function () {
                 $("#applicationsLogsTableDiv").mask($("#js_loading").text());
             },
             success: function (data) {
                 $("#applicationsLogsTable tbody").empty();
                 $("#applicationsLogsTable tbody").append(logs.buildApplicationsLogsTable(data));
                 globals.functions.showNoDataDiv("applicationsLogsTableDiv", "applicationsLogsTable", data.length,
                         $("#js_thereIsNoDataAvailable").text());
                 try {
                     $("#applicationsLogsTable").trigger("update");
                     $("#applicationsLogsTable").trigger("sorton",[[[0,1]]]);
                 } catch (e) {}
             },
             complete: function () {
                 $("#applicationsLogsTableDiv").unmask();
             }
         });
    };

    /**
     * Reloads the Audit Log table.
     */
    logs.reloadAuditTable = function () {
        $.ajax({
            type: "GET",
            url: "/api/log/audit",
            dataType: "json",
            cache: false,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                $("#auditLogDiv").unmask();
            },
            beforeSend: function () {
                $("#auditLogDiv").mask($("#js_loading").text());
            },
            success: function (data) {
                $("#auditLogTable").append(logs.buildAuditLogRows(data));
                globals.functions.showNoDataDiv("auditLogDiv", "auditLogTable", data.length,
                    $("#js_thereIsNoDataAvailable").text());
                try {
                    $("#auditLogTable").trigger("update");
                    $("#auditLogTable").trigger("sorton",[[[0,1]]]);
                } catch (e) {}
            },
            complete: function () {
                $("#auditLogDiv").unmask();
            }
        });
    };

    /**
     * Returns a string containing the HTML code for a table body from the download sub tab.
     *
     * @param tableName     name of the table
     * @param data          data received
     *                      expected data format:
     *                          [{name: "name1", profile: "profile1", modif: "11:30", size: "220k"}, ...]
     */
    logs.buildTableRows = function (tableName, data) {
        var url = "";
        var buildProfileColumn = false;
        if (tableName == "log") {
            url = "/api/log/archive/logs/";
        } else if (tableName == "debug"){
            url = "/api/log/archive/debug/";
            buildProfileColumn = true;
        }
        $("." + tableName + "FilesTbody").remove();
        var returnString = "<tbody class='" + tableName + "FilesTbody'>";
        for (var i = 0; i < data.length; i++) {
            var archiveName = data[i].name;
            var profile = data[i].profile;
            returnString = returnString + "<tr><td>";
            returnString = returnString + "<a href='" + url + archiveName + "'>" + data[i].name + "</a>";
            returnString = returnString + "</td><td>";
            if (buildProfileColumn) {
                if (data[i].profile.length > 0) {
                    returnString = returnString + "<a href='" + url + profile + "'>" + profile + "</a>";
                } else {
                    returnString = returnString + "-";
                }
                returnString = returnString + "</td><td>";
            }
            returnString = returnString + data[i].modif;
            returnString = returnString + "</td><td>";
            returnString = returnString + data[i].size;
            returnString = returnString + "</td><td>";
            returnString = returnString + "<input type='checkbox' id='" + archiveName + "'/>";
            returnString = returnString + "</td></tr>";
        }
        returnString = returnString + "</tbody>";        
        return returnString;
    };

    /**
     * Renders one of the log tabs' sub tabs.
     *
     * @param subTabName    sub tab name
     */
    logs.renderSubtab = function(subTabName) {
        if (subTabName == "viewLogs") {
            if (!logs.viewedApplicationsLogs && !logs.viewedAuditLogs) {
                logs.adjustViewLogsSubtabSize();
            }
            logs.buildAuditLogsFieldset();
            logs.buildApplicationsLogsFieldset();
            logs.buildApplicationsLogSelect();
        } else if (subTabName == "downloadLogs") {
            if (!logs.viewedDownloadLogs && !logs.viewedDownloadDebug) {
                logs.adjustDownloadLogsSubtabSize();
            }
            logs.buildDownloadLogsFieldset();
            logs.buildDownloadDebugFilesFieldset();
        } else if (subTabName == "syslogViewer") {
            if (!logs.viewedSyslogViewer) {
                logs.adjustSyslogViewerSubtabSize();
            }
            logs.buildSyslogViewerFieldset();
            logs.buildSyslogFilters();
        }
    };

    /**
     * Set widget height and add scrolling for the "Debug Logs" subtab.
     */
    logs.adjustViewLogsSubtabSize = function() {
        var tHeight = parseInt(($("#logsContentDiv").height() - 14
            - 2 * $("#applicationLogFieldset .groupHeader").outerHeight()) / 2);
        $("#applicationsLogsTableDiv").height(tHeight);
        $("#applicationsLogsTableDiv").addClass("scroll");
        $("#applicationsLogsTable thead").height($("#applicationsLogsTable thead").height());
        $("#auditLogDiv").height(tHeight);
        $("#auditLogDiv").addClass("scroll");
        $("#auditLogTable thead").height($("#auditLogTable thead").height());
    };

    /**
     * Set widget height and add scrolling for the "Downloads" subtab.
     */
    logs.adjustDownloadLogsSubtabSize = function() {
        var tHeight = parseInt(($("#logsContentDiv").height() - 14 - $("#debugFilesInfo").outerHeight()
            - 2 * $("#debugFilesFieldset .groupHeader").outerHeight()) / 2);
        $("#debugFilesDiv").height(tHeight);
        $("#debugFilesDiv").addClass("scroll");
        $("#debugFilesTable thead").height($("#debugFilesTable thead").height());
        $("#logFilesDiv").height(tHeight);
        $("#logFilesDiv").addClass("scroll");
        $("#logFilesTable thead").height($("#logFilesTable thead").height());
    };

    /**
     * Set widget height and add scrolling for the "Syslog Event Viewer" subtab.
     */
    logs.adjustSyslogViewerSubtabSize = function() {
        var tHeight = parseInt(($("#logsContentDiv").height() - $("#syslogViewerFieldset .groupHeader").outerHeight() + 8));
        $("#syslogViewerTableDiv").height(tHeight);
        $("#syslogViewerTableDiv").addClass("scroll");
        $("#syslogViewerTable thead").height($("#syslogViewerTable thead").height());
    };

    /**
     * Gets log archives from the device and creates the table.
     */
    logs.buildDownloadLogsFieldset = function() {

        $.ajax({
            type: "GET",
            url: "/api/log/archive/logs",
            dataType: "json",
            cache: false,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                $("#logFilesDiv").unmask();
            },
            beforeSend: function () {
                $("#logFilesDiv").mask($("#js_loading").text());
            },
            success: function (data) {
                $("#logFilesTable").append(logs.buildTableRows("log", data));
                globals.functions.showNoDataDiv("logFilesDiv", "logFilesTable", data.length,
                        $("#js_thereIsNoDataAvailable").text());
                if (logs.viewedDownloadLogs) {
                    try {
                        $("#logFilesTable").trigger("update");
                        $("#logFilesTable").trigger("sorton",[[[1,1]]]);
                    } catch (e) {}
                } else {
                    try {
                        $("#logFilesTable").tablesorter({
                            sortList: [[1,1]],
                            widgets: ['zebra'],
                            headers: {
                                0: {
                                    sorter: "text"
                                },
                                1: {
                                    sorter: "wcpDate"
                                },
                                2: {
                                    sorter: "size"
                                }
                            }
                        });
                        globals.functions.makeFixedTableHeader("logFilesTable");
                        logs.addFixedTableHeader("downloadLogs", "logFilesTable");
                    } catch (e) {}
                }

                $("#selectAllLog")[0].innerHTML = $("#js_selectAll").text();

                if (data.length < 1) {
                    $("#selectAllLog").attr("disabled", true);
                    $("#selectAllLog").addClass("ui-state-disabled");
                    $("#deleteSelectedLog").attr("disabled", true);
                    $("#deleteSelectedLog").addClass("ui-state-disabled");
                } else {
                    $("#selectAllLog").attr("disabled", false);
                    $("#selectAllLog").removeClass("ui-state-disabled");
                    $("#deleteSelectedLog").attr("disabled", false);
                    $("#deleteSelectedLog").removeClass("ui-state-disabled");
                }
                if (!logs.viewedDownloadLogs) {
                    $("#selectAllLog").click(function() {
                        logs.selectAll("log", this);
                    });

                    $("#createArchiveLog").click(function() {
                        logs.createArchive("logs");
                    });

                    $("#deleteSelectedLog").click(function() {
                        logs.deleteArchives("logs");
                    });
                }
                logs.viewedDownloadLogs = true;
            },
            complete: function () {
                $("#logFilesDiv").unmask();
            }
        });
    };

    /**
     * Gets debug archives from the device and creates the table
     */
    logs.buildDownloadDebugFilesFieldset = function() {

        $.ajax({
            type: "GET",
            url: "/api/log/archive/debug",
            dataType: "json",
            cache: false,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                $("#debugFilesDiv").unmask();
            },
            beforeSend: function () {
                $("#debugFilesDiv").mask($("#js_loading").text());
            },
            success: function (data) {
                $("#debugFilesTable").append(logs.buildTableRows("debug", data));
                globals.functions.showNoDataDiv("debugFilesDiv", "debugFilesTable", data.length,
                        $("#js_thereIsNoDataAvailable").text());
                if (logs.viewedDownloadDebug) {
                    try {
                        $("#debugFilesTable").trigger("update");
                        $("#debugFilesTable").trigger("sorton",[[[1,1]]]);
                    } catch (e) {}
                } else {
                    try {
                        $("#debugFilesTable").tablesorter({
                            sortList: [[2,1]],
                            widgets: ['zebra'],
                            headers: {
                                2: {
                                    sorter: "wcpDate"
                                }
                            }
                        });
                        globals.functions.makeFixedTableHeader("debugFilesTable");
                        logs.addFixedTableHeader("downloadLogs", "debugFilesTable");
                    } catch (e) {}
                }

                $("#selectAllDebug")[0].innerHTML = $("#js_selectAll").text();

                if (data.length < 1) {
                    $("#selectAllDebug").attr("disabled", true);
                    $("#selectAllDebug").addClass("ui-state-disabled");
                    $("#deleteSelectedDebug").attr("disabled", true);
                    $("#deleteSelectedDebug").addClass("ui-state-disabled");
                } else {
                    $("#selectAllDebug").attr("disabled", false);
                    $("#selectAllDebug").removeClass("ui-state-disabled");
                    $("#deleteSelectedDebug").attr("disabled", false);
                    $("#deleteSelectedDebug").removeClass("ui-state-disabled");
                }
                if (!logs.viewedDownloadDebug) {
                	$("#selectAllDebug").click(function() {
                        logs.selectAll("debug", this);
                    });

                	$("#createArchiveDebug").click(function() {
                        logs.createArchive("debug");
                    });

                    $("#deleteSelectedDebug").click(function() {
                        logs.deleteArchives("debug");
                    });
                }
                logs.viewedDownloadDebug = true;
            },
            complete: function () {
            	$("#debugFilesDiv").unmask();
            }
        });
        $.ajax({
            type: "GET",
            url: "/api/log/debug",
            dataType: "json",
            cache: false,
            success: function (data) {
                switch(data.count) {
                    case 0:
                        $("#debugFilesInfo").text($("#js_noDebugFilesAvailable").text());
                        break;
                    case 1:
                        globals.functions.composeText("debugFilesInfo", "js_debugFileAvailable", [data.count, data.size]);
                        break;
                    default:
                        globals.functions.composeText("debugFilesInfo", "js_debugFilesAvailable", [data.count, data.size]);
                        break;
                }
            }
        });
    };

    /**
     * Gets syslog entries from the device and creates the table
     */
    logs.buildSyslogViewerFieldset = function () {
        $.ajax({
            type: "GET",
            url: "/api/system/syslog",
            dataType: "json",
            data: logs.syslogFilterString(),
            cache: false,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                $("#syslogViewerTableDiv").unmask();
            },
            beforeSend: function () {
                $("#syslogViewerTableDiv").mask($("#js_loading").text());
            },
            success: function (data) {
                $("#syslogViewerTable tbody").empty();
                $("#syslogViewerTable tbody").append(logs.buildSyslogViewerTable(data));
                globals.functions.showNoDataDiv("syslogViewerTableDiv", "syslogViewerTable", data.length,
                    $("#js_thereIsNoDataAvailable").text());

                if (logs.viewedSyslogViewer) {
                    try {
                        $("#syslogViewerTable").trigger("update");
                        $("#syslogViewerTable").trigger("sorton",[[[0,1]]]);
                    } catch (e) {}
                } else {
                    try {
                        $("#filterSyslogDiv select").change(logs.filterSyslog);
                        $("#refreshSyslogTable").click(logs.filterSyslog);
                        $("#syslogViewerTable").tablesorter({
                            sortList: [[0,1]],
                            widgets: ['zebra'],
                            headers: {
                                1: {
                                    sorter: "wcpDate"
                                },
                                3: {
                                    sorter: "text"
                                }
                            }
                        });
                        globals.functions.makeFixedTableHeader("syslogViewerTable");
                        logs.addFixedTableHeader("syslogViewer", "syslogViewerTable");
                    } catch (e) {}
                }

                logs.viewedSyslogViewer = true;
            },
            complete: function () {
                $("#syslogViewerTableDiv").unmask();
            }
        });
    };

    /**
     * Builds the rows of the Syslog Viewer table.
     *
     * @param data          Object array containing syslog entries.
     *                      Format: [{timestamp: "2012-11-20 11:38:38", host: "localhost",
     *                                source: "auditd[8379]" , message: "time reset -0.533940 s"},...]
     * @return {String}     HTML string containing the rows of the table.
     */
    logs.buildSyslogViewerTable = function (data) {
        var returnString = "";
        for (var i = 0; i < data.length; i++) {
            returnString += "<tr><td>";
            returnString += data[i].timestamp;
            returnString += "</td><td>";
            returnString += data[i].host;
            returnString += "</td><td>";
            if (data[i].type.length > 0) {
                returnString += data[i].type;
            } else {
                returnString += "-";
            }
            returnString += "</td><td>";
            returnString += data[i].source;
            returnString += "</td><td>";
            returnString += data[i].message.replace(/</g, "&#60;").replace(/>/g, "&#62;");
            returnString += "</td></tr>";
        }
        return returnString;
    };

    /**
     * Filters the Syslog Viewer table depending on the combobox selection.
     */
    logs.filterSyslog = function () {
        $.ajax({
            type: "GET",
            url: "/api/system/syslog",
            dataType: "json",
            data: logs.syslogFilterString(),
            cache: false,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                $("#syslogViewerTableDiv").unmask();
            },
            beforeSend: function () {
                $("#syslogViewerTableDiv").mask($("#js_loading").text());
            },
            success: function (data) {
                $("#syslogViewerTable tbody").empty();
                $("#syslogViewerTable tbody").append(logs.buildSyslogViewerTable(data));
                globals.functions.showNoDataDiv("syslogViewerTableDiv", "syslogViewerTable", data.length,
                    $("#js_thereIsNoDataAvailable").text());
                try {
                    $("#syslogViewerTable").trigger("update");
                    $("#syslogViewerTable").trigger("sorton",[[[0,1]]]);
                } catch (e) {}
            },
            complete: function () {
                $("#syslogViewerTableDiv").unmask();
            }
        });
    };

    /**
     * Compose the syslog parameter string to be sent to the server , depending on the combobox selection.
     *
     * @return {String}     The parameter string.
     */
    logs.syslogFilterString = function () {
        var dataArray = [];
        $("#filterSyslogDiv select").each( function (){
            if (this.value.length > 0 && this.value.length != null) {
                dataArray.push(this.name + "=" + this.value);
            }
        });
        return dataArray.join("&");
    };

    /**
     * Creates an archive and refreshes the selected table.
     *
     * @param fieldset  fieldset for which to create an archive: logs|debug.
     */
    logs.createArchive = function (fieldset) {
        var buttonId = "";
        var url = "";
        var refreshFunction;
        if (fieldset == "logs") {
            buttonId = "createArchiveLog";
            containerId = "logFilesDiv";
            url = "/api/log/archive/logs";
            refreshFunction = function(){logs.buildDownloadLogsFieldset()};
        } else if (fieldset == "debug") {
            buttonId = "createArchiveDebug";
            containerId = "debugFilesDiv";
            url = "/api/log/archive/debug";
            refreshFunction = function(){logs.buildDownloadDebugFilesFieldset()};
        }
        logs.createDeleteArchive("PUT", url, buttonId, containerId, refreshFunction);
    };

    /**
     * Deletes the selected archives from a fieldset table.
     *
     * @param fieldset  fieldset name: logs|debug.
     */
    logs.deleteArchives = function (fieldset) {
        var buttonId = "";
        var url = "";
        var tableId = "";
        var refreshFunction;
        if (fieldset == "logs") {
            buttonId = "deleteSelectedLog";
            url = "/api/log/archive/logs";
            tableId = "logFilesTable";
            containerId = "logFilesDiv";
            refreshFunction = function(){logs.buildDownloadLogsFieldset()};
        } else if (fieldset == "debug") {
            buttonId = "deleteSelectedDebug";
            url = "/api/log/archive/debug";
            tableId = "debugFilesTable";
            containerId = "debugFilesDiv";
            refreshFunction = function(){logs.buildDownloadDebugFilesFieldset()};
        }
        var allSelected = true;
        var noneSelected = true;
        var parametersString = "";
        $("#" + tableId + " :checkbox").each( function () {
            if (this.checked){
                noneSelected = false;
                parametersString += this.id + ";";
            } else {
                allSelected = false;    
            }
        });
        if (!noneSelected) {
            if (!allSelected) {
                parametersString = parametersString.substring(0, parametersString.length-1);
                url = url + "/" + parametersString;
            }
            logs.createDeleteArchive("DELETE", url, buttonId, containerId, refreshFunction);
        }
    };

    /**
     * Deletes or creates archives
     *
     * @param operation         Type of ajax request (DELETE to delete archives or PUT to create archives)
     * @param url               URL to the create/delete archive resource ( for debug or log files )
     * @param buttonId          Button to disable during the operation
     * @param containerId       Div to mask during the operation
     * @param refreshFunction   Refresh function to call after the operation
     */
    logs.createDeleteArchive = function (operation, url, buttonId, containerId, refreshFunction) {

        $.ajax({
            type: operation,
            url: url,
            timeout: globals.variables.CREATE_ARCHIVE_TIMEOUT,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                globals.functions.enableButton(buttonId, true);
                $("#" + containerId).unmask();
            },
            beforeSend: function () {
                globals.functions.enableButton(buttonId, false);
                $("#" + containerId).mask($("#js_loading").text());
            },
            complete: function () {
                refreshFunction();
                globals.functions.enableButton(buttonId, true);
            }
        });

    };
    
    /**
     * Requests the audit log data and populates the respective table.
     */
    logs.buildAuditLogsFieldset = function() {
        $.ajax({
            type: "GET",
            url: "/api/log/audit",
            dataType: "json",
            cache: false,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                $("#auditLogDiv").unmask();
            },
            beforeSend: function () {
                $("#auditLogDiv").mask($("#js_loading").text());
            },
            success: function (data) {
                $("#auditLogTable").append(logs.buildAuditLogRows(data));
                globals.functions.showNoDataDiv("auditLogDiv", "auditLogTable", data.length,
                        $("#js_thereIsNoDataAvailable").text());
                if (logs.viewedAuditLogs) {
                    try {
                        $("#auditLogTable").trigger("update");
                        $("#auditLogTable").trigger("sorton",[[[0,1]]]);
                    } catch (e) {}
                } else {
                    $("#refreshAuditTable").click(logs.reloadAuditTable);
                    try {
                        $("#auditLogTable").tablesorter({
                            sortList: [[0,1]],
                            widgets: ['zebra'],
                            headers: {
                                0: {
                                    sorter: "wcpDate"
                                }
                            }
                        });
                        globals.functions.makeFixedTableHeader("auditLogTable");
                        logs.addFixedTableHeader("viewLogs", "auditLogTable");
                    } catch (e) {}
                }
                logs.viewedAuditLogs = true;

            },
            complete: function () {
                $("#auditLogDiv").unmask();
            }
        });
    };

    /**
     *  Requests the applications log data and populates the respective table.
     */
    logs.buildApplicationsLogsFieldset = function () {
        var parameters = "";
        if ($("#applicationsCombo").val() != null) {
            parameters = "id=" + $("#applicationsCombo").val();
        }
        $.ajax({
            type: "GET",
            url: "/api/log/application",
            dataType: "json",
            data: parameters,
            cache: false,
            error: function (xmlHttpReq, textStatus, errorThrown) {
                globals.functions.ajaxError(xmlHttpReq, textStatus, errorThrown);
                $("#applicationsLogsTableDiv").unmask();
            },
            beforeSend: function () {
                $("#applicationsLogsTableDiv").mask($("#js_loading").text());
            },
            success: function (data) {
                $("#applicationsLogsTable tbody").empty();
                $("#applicationsLogsTable tbody").append(logs.buildApplicationsLogsTable(data));
                globals.functions.showNoDataDiv("applicationsLogsTableDiv", "applicationsLogsTable", data.length,
                        $("#js_thereIsNoDataAvailable").text());

                if (logs.viewedApplicationsLogs) {
                    try {
                        $("#applicationsLogsTable").trigger("update");
                        $("#applicationsLogsTable").trigger("sorton",[[[0,1]]]);
                    } catch (e) {}
                } else {
                    try {
                        $("#refreshApplicationsTable").click(logs.reloadApplicationsTable);
                        $("#applicationsCombo").change(function(){logs.reloadApplicationsTable()});
                        $("#applicationsLogsTable").tablesorter({
                            sortList: [[0,1]],
                            widgets: ['zebra'],
                            headers: {
                                1: {
                                    sorter: "text"
                                }
                            }
                        });
                        globals.functions.makeFixedTableHeader("applicationsLogsTable");
                        logs.addFixedTableHeader("viewLogs", "applicationsLogsTable");
                    } catch (e) {}
                }
                
                logs.viewedApplicationsLogs = true;
            },
            complete: function () {
                $("#applicationsLogsTableDiv").unmask();
            }
        });
    };

    //Internet Explorer hacks in order to set height and enable table scroll
    $("#logsContentDiv").height(($(window).height() - $("#viewLogsDiv").offset().top - 24));

    /**
     * Makes the selected subtab content visible and calls the
     * render sub tab function when the sub tab is clicked.
     */
    $("#logsTabs li a").click(function (e) {
    
        e.preventDefault();
        if (logs.tab != this.name) {
            $("#logsTabs li a").removeClass("selected");
            $(this).addClass("selected");
            $("#logsContentDiv .logsSubtab").hide();
            $("#" + this.name + "Div").show();
            logs.renderSubtab(this.name);
            logs.tab = this.name;
        }
                
    }).filter(':first').click();

    logs.adjustTableHeadersOnResize = function() {
        if (globals.variables.currentTab != "logs") {
            $(window).unbind("resize", logs.adjustTableHeadersOnResize);
            return;
        }
        for (var tableIdx in logs.fixedHeaderTables[logs.tab]) {
            tableId = logs.fixedHeaderTables[logs.tab][tableIdx];
            globals.functions.resizeFixedTableHeader(tableId);
        }
    }

    $(window).bind("resize", function() {
        if (logs.tableResizeHandler) {
            clearTimeout(logs.tableResizeHandler);
        }
        logs.tableResizeHandler = setTimeout(logs.adjustTableHeadersOnResize, 300);
    });

});