# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
REST API for system management.
"""

import web
import json
import threading
import time
import configobj
import os
import subprocess
import urllib
import audit

import core.system.systime
import services_rest_api
import settings_rest_api

from core.auth import passwd
from core.logs import login
from core.system import sysinfo
from core.system import syslog
from core.system import network
from core.system import vnc
from core.system import systime
from core.system import shell
from core.system import rpm
from core.services import manager
from core.services import monitor
from core.common import utils
from core.common import log
from core.common import configuration
from core.common import version
from core.common import i18n
from core.common.decorators import synchronized

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."


password_lock               = threading.Lock()
encrypt_lock                = threading.Lock()
network_lock                = threading.Lock()
date_lock                   = threading.Lock()
firewall_lock               = threading.Lock()
http_server_lock            = threading.Lock()
ntp_lock                    = threading.Lock()
ipoffice_lan_config_lock    = threading.Lock()
subinterface_config_lock    = threading.Lock()
new_hw_lock                 = threading.Lock()

timezone_manager                 = core.system.systime.TimezoneManager()
ntp_manager                      = core.system.systime.NTPManager()
services_manager                 = manager.ServicesManager()
generate_certificate_manager     = settings_rest_api.GenerateCertResource()
distribute_certificate_manager   = settings_rest_api.CertificateResource()

TAG                     = "system-rest-api"

SERVICE_STOP_RETRY_COUNT = 65
SERVICE_RETRY_DELAY = 2


def shutdown_reboot(restart=False):
    """
    Stops the applications and calls the shutdown/reboot function.

    Args:
    restart -- True to reboot, False to shutdown
    """

    log.SHARED_LOGGER.info("Stopping services")

    # stop tcpdumps
    shell.sudo_call("%s stop" % configuration.SHARED_CONFIGURATION['tcpdumps']['tcpdumps_script'])
    log.SHARED_LOGGER.info("Stopped tcpdumps")

    # stop managed services
    for index in configuration.SHARED_CONFIGURATION["applications"]:
        if configuration.SHARED_CONFIGURATION["applications"][index].as_bool('control_supported'):
            if index in services_manager.services:
                log.SHARED_LOGGER.info("Current service: %s" % index)
                service = services_manager[index]
                service.update_run_info()
                if service.is_running():
                    log.SHARED_LOGGER.info("%s is running, sending stop command" % index)
                    service.stop()
                    count = SERVICE_STOP_RETRY_COUNT
                    while service.is_running():
                        time.sleep(SERVICE_RETRY_DELAY)
                        log.SHARED_LOGGER.info("Waiting for %s to stop, remaining retires: %i" % (index, count))
                        count = count - 1
                        if count < 0:
                            log.SHARED_LOGGER.info("force stop %s" % index)
                            service.force_stop()
                            break
                        service.update_run_info()
                else:
                    log.SHARED_LOGGER.info("%s is stopped" % index)
    output = shell.sudo_execute("rpm -q MediaManager", parse=shell.RAW)
    if output.decode().startswith("MediaManager-11.0.4") and not os.path.exists("/opt/MediaManager/backup/mmgr_bk.tar.gz"):
        if os.path.exists("/opt/MediaManager/run_backup.sh"):
            shell.sudo_call("/opt/MediaManager/run_backup.sh")
        if not os.path.exists("/opt/MediaManager/backup/mmgr_bk.tar.gz"):
            log.SHARED_LOGGER.info("Media manager Backup archive was not created,remove the flag to stop upgrade")
            if  os.path.exists("/opt/Avaya/ABE_release" ):
                shell.sudo_call("cp /opt/Avaya/ABE_release /opt/Avaya/ABE_release_tmp")
                shell.sudo_call("sed -i 's/ABE_VERSION=11.0/ABE_VERSION=10.0/g' /opt/Avaya/ABE_release");
            elif  os.path.exists("/opt/Avaya/APPL_release" ):
                shell.sudo_call("cp /opt/Avaya/APPL_release /opt/Avaya/APPL_release_tmp")
                shell.sudo_call("sed -i 's/ABE_VERSION=11.0/ABE_VERSION=10.0/g' /opt/Avaya/APPL_release");
    shell.shutdown(restart)

class ChangePasswordResource(object):
    """
    REST resource to change user password.

    Methods:

    PUT -- change user password
    """

    @synchronized(password_lock)
    def PUT(self):
        """
        Change user password.

        Input:

        oldpass = <old password>
        newpass = <new password>

        Error codes:

        401 -- invalid password
        """
        try:
            old_pass = web.input(oldpass="").oldpass
            new_pass = web.input(newpass="").newpass
            passwd.change_pass(configuration.USERNAME, old_pass, new_pass)
            audit.log(i18n.custom_gettext('change password'), username=configuration.USERNAME)
            login.mark_default_password_changed(new_pass)
        except passwd.AuthError:
            log.exception(TAG, 'unable to change password')
            web.ctx.status = '401'

class ChangeRootPasswordResource(object):
    """
    REST resource to change root password.

    Methods:

    GET -- check root password
    PUT -- change root password
    """

    @synchronized(password_lock)
    def GET(self):
        """
        Checks if the new password is correct.
        Returns an error message from the server if the new password is incorrect

        Input:

        oldpass = <old password>
        newpass = <new password>
        """
        try:
            old_pass = web.input(oldpass="").oldpass
            new_pass = web.input(newpass="").newpass
            user = web.input(user="root").user
            error_text = passwd.check_root_pass(old_pass, new_pass, user)
            audit.log(i18n.custom_gettext('check root password'), user)
            log.info(TAG, 'check root password %s' % error_text)
            return error_text
        except UnicodeEncodeError:
            web.ctx.status = '400'

    @synchronized(password_lock)
    def PUT(self):
        """
        Changes the root password if the new password is correct.

        Input:

        oldpass = <old password>
        newpass = <new password>
        """
        old_pass = "%s" % web.input(oldpass="").oldpass
        new_pass = "%s" % web.input(newpass="").newpass
        user = "%s" % web.input(user="root").user
        command = "%s %s %s %s" % (configuration.SHARED_CONFIGURATION['password']['change_password_script'],
                                new_pass.replace(" ", "\ "), old_pass.replace(" ", "\ "), user)
        text = shell.sudo_execute(command, parse=shell.RAW)
        password_changed = False
        if text.decode().find("error") == -1:
            password_changed = True
        audit.log(i18n.custom_gettext('change root password') + " " + str(password_changed), user)
        log.info(TAG, 'change %s password %s %s' % (user, password_changed, text))

        web.header("Content-Type","application/json")
        return json.dumps({"password_changed" : password_changed})


class PasswordRulesResource(object):
    """
    REST resource to manage password rules settings

    Methods:

    GET    -- returns JSON object with password rules settings
    PUT    -- update password rules settings
    """
    @synchronized(password_lock)
    def GET(self):
        """
        Return JSON object with password rules settings. Format:
        {
            min_lengt: 8,
            min_uppercase: 0,
            min_lowercase: 0,
            min_numeric: 0,
            min_special: 0,
            allow_sequence: true,
            max_allowed_sequence: 0
        }

        Error codes:

        400 -- Error occurred while reading password rules settings (password rules configuration file couldn't be read).
        """
        password_rules_file = configuration.SHARED_CONFIGURATION['password']['password_rules_file']
        try:
            password_rules_settings = {'min_length': None,
                                       'min_uppercase': None,
                                       'min_lowercase': None,
                                       'min_numeric': None,
                                       'min_special': None,
                                       'allow_sequence': None,
                                       'max_allowed_sequence': None}
            utils.read_config(password_rules_file, password_rules_settings)
            password_rules_settings['allow_sequence'] = (password_rules_settings['allow_sequence'] in ["True", "true", "t", "T"])
            web.header("Content-Type","application/json")
            return json.dumps(password_rules_settings)
        except IOError:
            log.exception(TAG, 'unable to read password rules configuration file %s' % password_rules_file)
            web.ctx.status = '400'

    @synchronized(password_lock)
    def PUT(self):
        """
        Update password rules settings.

        Input:

        min_length              -- Minimum password length.
        min_uppercase           -- Minimum number of uppercase characters to be used in the password.
        min_lowercase           -- Minimum number of lowercase characters to be used in the password.
        min_numeric             -- Minimum number of numeric characters to be used in the password.
        min_special             -- Minimum number of special characters to be used in the password.
        allow_sequence          -- True to allow character sequences in the password.
        max_allowed_sequence    -- The maximum allowed character sequence length in the password.

        Error codes:

        400 -- I/O error when writting password rules configuration
        """
        try:
            password_rules_settings = {}

            min_length = web.input(min_length=None).min_length
            if min_length:
                password_rules_settings['min_length'] = min_length

            min_uppercase = web.input(min_uppercase=None).min_uppercase
            if min_uppercase:
                password_rules_settings['min_uppercase'] = min_uppercase

            min_lowercase = web.input(min_lowercase=None).min_lowercase
            if min_lowercase:
                password_rules_settings['min_lowercase'] = min_lowercase

            min_numeric = web.input(min_numeric=None).min_numeric
            if min_numeric:
                password_rules_settings['min_numeric'] = min_numeric

            min_special = web.input(min_special=None).min_special
            if min_special:
                password_rules_settings['min_special'] = min_special

            allow_sequence = web.input(allow_sequence=None).allow_sequence
            if allow_sequence:
                password_rules_settings['allow_sequence'] = (allow_sequence in ["True", "true", "t", "T"])

            max_allowed_sequence = web.input(max_allowed_sequence=None).max_allowed_sequence
            if max_allowed_sequence:
                password_rules_settings['max_allowed_sequence'] = max_allowed_sequence

            password_rules_file = configuration.SHARED_CONFIGURATION['password']['password_rules_file']
            utils.update_config(password_rules_file, password_rules_settings)

            # TODO : call script that reads the password rules config file and applies the settings

        except (IOError, OSError):
            log.exception(TAG, 'unable to write password rules configuration file %s' % password_rules_file)
            web.ctx.status = '400'


class EncryptPasswordResource(object):
    """
    REST resource to manage password encryption

    Methods:

    PUT    -- encrypt passwords
    """
    @synchronized(encrypt_lock)
    def PUT(self):
        """
        Writes the passwords in a file and calls the encryption script.
        """
        root_pass = "%s" % web.input(root_pass="").root_pass
        admin_pass = "%s" % web.input(admin_pass="").admin_pass
        system_pass = "%s" % web.input(system_pass="").system_pass
        password_file = configuration.SHARED_CONFIGURATION['password']['encrypt_password_file']

        shell.sudo_call("touch %s" % password_file)
        shell.sudo_call("chmod go+w %s" % password_file)
        file_content = "Administrator=%s\n" % admin_pass
        file_content += "security=%s\n" % root_pass
        file_content += "System=%s\n" % system_pass
        try:
            with shell.file_open(password_file, "wb") \
                    as outfile:
                outfile.write(file_content.encode("utf-16"))
        except IOError:
            log.SHARED_LOGGER.exception("unable to write password file")
        shell.sudo_call("chmod go-w %s" % password_file)
        shell.sudo_call(configuration.SHARED_CONFIGURATION['password']['encrypt_password_script'])


class StatusResource(object):
    """
    REST resource to handle system information/control requests.

    Methods:

    GET -- returns system status information.
    PUT -- issues system shutdown or reboot command
    """

    def GET(self):
        """
        Returns json string containing system information.

        Data format:

        {
            "os": {
                "kernel": "2.6.18-164.el5",
                "uptime": "1:16",
                "version": "CentOS release 4.7" },
            "sysinv": {
                "inv_code": "1234567",
                "model_info": "Dell R210",
                "sys_serial_num": "BN123344"},
            "mem": {
                "total": 1035140,
                "free": 424112,
                "used": 611028 },
            "disk": {
                "total": 17219128,
                "free": 14093520,
                "used": 3125608 },
            "cpu": {
                "usage": [["02:10PM", 0.49], ["02:20PM", 2.30], ["02:30PM", 1.10]],
                "loadavg": ["1.5", "7.8", "3.50"] }
        }
        """
        web.header("Content-Type","application/json")
        return json.dumps(sysinfo.system_info(), default=utils.encode_json)

    @synchronized(services_rest_api.services_lock)
    def PUT(self):
        """
        Issue a shutdown/reboot command to the operating system.
        Before issuing the command , it stops the running services

        Input:

        restart = true|false (optional - if it exists and is 'true' indicates that a reboot is required)
        """
        restart = web.input(restart='false').restart.lower() == 'true'
        if restart:
            audit.log(i18n.custom_gettext('reboot the operating system'))
        else:
            audit.log(i18n.custom_gettext('shutdown the operating system'))
        shutdown_reboot(restart)


class SyslogResource(object):
    """
    REST resource to read syslog entries

    Methods:

    GET -- return syslog entries
    """
    parser = syslog.Parser()

    def GET(self):
        """
        Return a json string with syslog entries.

        Data format:

        [ { "timestamp":    "YYYY-MM-DD HH:MM:SS",
            "host":         "hostname",
            "message_type": "event type",
            "message":      "syslog message"
            "source":       "message source"},
          ... ]

        Input:

        keyword     = word to filter messages by (optional, defaults to None)
        tail_count  = number of last lines from file to be considered (optional,
                      defaults to [syslog] tail_count configuration entry - see main.ini)
        host        = host name to filter messages by (optional, defaults to None)
        type        = syslog file type, can be one of [syslog] audit_type, security_type,
                      operational_type entries - see main.ini (defaults to None)
        source      = syslog source(tag field): ipoffice, ipowatchdog, ipomediaserver, ipowebcontrol,
                      ipovmpro, ipoonexportal

        Ex: to get all syslog entries from the watchdog service from last 100 lines:
            GET /system/syslog?keyword=IPOfficeWatchdog&tail_count=100
        """
        web.header("Content-Type","application/json")
        key         = web.input(keyword=None).keyword
        tail_count  = web.input(tail_count=None).tail_count
        hosts       = web.input(host=None).host
        type        = web.input(type=None).type
        source      = web.input(source=None).source

        log.debug(TAG, "key %s" % key)
        log.debug(TAG, "tail_count %s" % tail_count)
        log.debug(TAG, "hosts %s" % hosts)
        log.debug(TAG, "type %s" % type)
        log.debug(TAG, "source %s" % source)

        if not tail_count:
            tail_count = configuration.SHARED_CONFIGURATION['syslog'].as_int('tail_count')
        tail_count = int(tail_count)

        shell.sudo_call("chmod -R 777 %s" % configuration.SHARED_CONFIGURATION['syslog']['path'])
        if not hosts:
            hosts = syslog.listDirectories()
        else:
            hosts = [hosts]

        if type:
            type = str(type)

        if source:
            source = str(source)

        path = ""
        if key and key in "IPOfficeWatchdog,IPOLError":
            path = configuration.SHARED_CONFIGURATION['syslog']['file']
            hosts = None


        log.debug(TAG, "key %s" % key)
        log.debug(TAG, "tail_count %s" % tail_count)
        log.debug(TAG, "hosts %s" % hosts)
        log.debug(TAG, "type %s" % type)
        log.debug(TAG, "source %s" % source)

        logs = self.parser.parse(filepath=path,
                                keyword=key,
                                tail_count=tail_count,
                                hosts=hosts,
                                type=type,
                                source=source)
        return json.dumps(logs, default=utils.encode_json)


class NetworkConfigurationResource(object):
    """
    REST resource to manage network configuration

    Methods:

    GET -- return network configuration
    PUT -- change network configuration
    """

    STOP_SERVICE_RETRY_COUNT = 40

    @synchronized(network_lock)
    def GET(self):
        """
        Return a json string with network configuration.

        Data format:

        { "interfaces":
            [
                { "use_dhcp":   true|false,
                  "netmask":    "netmask",
                  "ipaddr":     "IPv4 address",
                  "name":       "interface name",
                  "gateway":    "gateway address" },
                   ...
            ],
          "hostname":   "host name",
          "dns":        ["dns_ip_address_1",...]
        }

        """
        web.header("Content-Type","application/json")
        
        netconfig = utils.OpenStruct()

        netconfig.interfaces = []
        if version.RELEASE_TYPE == "apc" :
            netconfig.interfaces.append(network.interface("eth0.1"))
        else:
            for name in network.get_all_interfaces():
                netconfig.interfaces.append(network.interface(name))

        netconfig.hostname = network.hostname()
        netconfig.dns = network.dns()
        netconfig.use_prefix = (float(".".join(str(utils.centos_release()).split('.')[0:1])) >= 6.0)

        return json.dumps(netconfig, default=utils.encode_json)

    @synchronized(network_lock)
    def PUT(self):
        """
        Configure network parameters. Also restarts Avaya services if needed.

        Input:

        name        -- interface name
        ipaddr      -- interface IP address
        netmask     -- interface netmask
        prefix      -- interface prefix
        gateway     -- interface gateway
        use_dhcp    -- True|False
        auto_dns    -- yes|no (if automatically obtain DNS information from provider)
        hostname    -- host name
        dns         -- comma separated list of DNS servers
        start_stopped_services  -- True to start all the services after the network settings have been applied
        """
        interf = network.Interface()
        interf.name = str(web.input(name='').name)
        interf.ipaddr = str(web.input(ipaddr='').ipaddr)
        interf.netmask = str(web.input(netmask='').netmask)
        interf.gateway = str(web.input(gateway='').gateway)
        interf.use_dhcp = bool(web.input(use_dhcp=False).use_dhcp)
        interf.auto_dns = str(web.input(auto_dns='yes').auto_dns)
        if float(".".join(str(utils.centos_release()).split('.')[0:1])) >= 6.0:
            interf.prefix = str(web.input(prefix='').prefix)
        hostname = str(web.input(hostname='').hostname)
        dns = web.input(dns='').dns.split(",")

        create_cert = str(web.input(create_cert="false").create_cert)
        subject = str(web.input(subject=None).subject)
        alt_subject = str(web.input(alt_subject=None).alt_subject)
        duration = str(web.input(duration=None).duration)
        rsa = str(web.input(rsa=None).rsa)
        sha = str(web.input(sha=None).sha)
        auto_generate = str(web.input(auto_generate=None).auto_generate)
        self_signed = str(web.input(self_signed=None).self_signed)

        start_stopped_services = bool(web.input(start_stopped_services=False).start_stopped_services)
        start_selected_services = web.input(start_selected_services=None).start_selected_services
        if interf.use_dhcp == False and not configuration.data_validate_type_length(interf.ipaddr,configuration.DATA_IP_VALIDATOR,configuration.DATALENGTH_IP_VALIDATOR):
            log.exception(TAG, 'Issues when setting configuration')
        if not configuration.data_length_validate(hostname,configuration.DATALENGTH_HOSTNAME_VALIDATOR):
            log.exception(TAG, 'Issues when setting configuration')

        called_by_ignition = False
        if start_selected_services:
            start_selected_services = start_selected_services.split(",")
            called_by_ignition = True
        if start_stopped_services is True:
            called_by_ignition = True

        services_to_restart = {}
        if create_cert == "false":
            services_to_restart = {'ipoffice': None,
                                   'ipoffice-demo': None,
                                   'ipoffice-sees': None,
                                   'voicemail': None,
                                   'onexportal': None,
                                   'onexcollab': None,
                                   'csipo': None,
                                   'callanalytics': None,
                                   'webmanager': None,
                                   'weblm': None,
                                   'webrtcgw': None,
                                   'authmodule': None}
        else:
            if called_by_ignition:
                if start_selected_services:
                    for service_id in start_selected_services:
                        services_to_restart[service_id] = None
                elif start_stopped_services:
                    for service_id in services_manager.services:
                        if services_manager[service_id].generic_service is True:
                            services_to_restart[service_id] = None
                else:
                    services_to_restart = {'ipoffice': None,
                                           'ipoffice-demo': None,
                                           'ipoffice-sees': None}
            else:
                services_to_restart = {'ipoffice': None,
                                       'ipoffice-demo': None,
                                       'ipoffice-sees': None}

        for service_id in services_to_restart:
            if service_id in services_manager.services:
                services_to_restart[service_id] = services_manager[service_id]
            if services_to_restart[service_id]:
                services_to_restart[service_id].update_run_info()
                if services_to_restart[service_id].is_running():
                    services_to_restart[service_id].stop()
                    audit.log(i18n.custom_gettext('change network configuration: stopping %s service...') %
                              services_to_restart[service_id].display_name)
                    count = self.STOP_SERVICE_RETRY_COUNT
                    while services_to_restart[service_id].is_running():
                        time.sleep(2)
                        count -= 1
                        if count < 0:
                            services_to_restart[service_id].force_stop()
                            break
                        services_to_restart[service_id].update_run_info()
                else:
                    if not start_stopped_services:
                        # if service not running, leave it that way
                        services_to_restart[service_id] = None
                    else:
                        if start_selected_services:
                            if not service_id in start_selected_services:
                                services_to_restart[service_id] = None

        # ip is not set at this stage for dhcp
        # check if exists an ip address on the current network for inteface
        if interf.use_dhcp == False:
            network.ping_ip(interf.ipaddr,interf.name)
        else:
            network.DUPLICATE_IP = False

        if "." in interf.name:
            network.remove_from_vlans(interf.name)
        network.set_interface(interf)
        network.apply_hostname(hostname)
        if interf.auto_dns == 'no':
            network.set_dns(dns)

        network.restart()
        audit.log(i18n.custom_gettext('change network interface configuration: restarted network interface %s') % interf.name)

        # XXX this is a super ugly hack required for VMPro server
        #     when eth0 configuration is changed we need to update
        #     HOSTIPADDRESS entry in /etc/vmpro_settings.ini file
        if interf.name == 'eth0':
            if 'voicemail' in services_manager.services:
                rpm_info = rpm.query_package(services_manager['voicemail'].package_name)
                if rpm_info:
                    try:
                        eth0_ip = network.eth0_ip()
                        log.info(TAG, "setting VMPro host IP to <%s>" % eth0_ip)
                        network.set_vmpro_host_ip(eth0_ip)
                    except (IOError, OSError, configobj.ConfigObjError):
                        log.exception(TAG, 'unable to set host IP address for VMPro')

        if create_cert == "true":
            generate_certificate_manager.create_cert(subject=subject, alt_subject=alt_subject,
                                                     duration=duration, rsa=rsa, sha=sha, auto_generate=auto_generate,
                                                     self_signed=self_signed)
            distribute_certificate_manager._install_certificate()

        for service_id in services_to_restart:
            if services_to_restart[service_id]:
                if called_by_ignition is False:
                    if not ("ipoffice" in service_id and network.DUPLICATE_IP is True):
                        rpm_info = rpm.query_package(services_to_restart[service_id].package_name)
                        if rpm_info:
                            services_to_restart[service_id].start()
                            audit.log(i18n.custom_gettext('change network configuration: starting %s service...')
                                      % services_to_restart[service_id].display_name)
                else:
                    if services_to_restart[service_id].generic_service:
                        if not ("ipoffice" in service_id and network.DUPLICATE_IP is True):
                            rpm_info = rpm.query_package(services_to_restart[service_id].package_name)
                            if rpm_info:
                                if start_stopped_services:
                                    if start_stopped_services is True:
                                        shell.enable_service(services_to_restart[service_id].service_name)
                                        audit.log(i18n.custom_gettext('change autostart state for %s to --add') %
                                                  services_to_restart[service_id].service_name)
                                if create_cert == "false":
                                    services_to_restart[service_id].start()
                                    audit.log(i18n.custom_gettext(
                                        'change network configuration: starting %s service...')
                                              % services_to_restart[service_id].display_name)

        if create_cert == "true":
            if called_by_ignition is True:
                for service_id in services_to_restart:
                    if services_to_restart[service_id]:
                        if "ipoffice" in service_id:
                            rpm_info = rpm.query_package(services_to_restart[service_id].package_name)
                            if rpm_info:
                                if not network.DUPLICATE_IP is True:
                                    unset_mapping_script = configuration.SHARED_CONFIGURATION['webapp']['unset_mapping_script']
                                    shell.sudo_call("chmod go+x %s" % unset_mapping_script)
                                    shell.sudo_call(unset_mapping_script)
                                    shell.sudo_call("chmod go-x %s" % unset_mapping_script)
                                    services_to_restart[service_id].start()
                                    audit.log(i18n.custom_gettext(
                                        'change network configuration: starting %s service...')
                                              % services_to_restart[service_id].display_name)
            distribute_certificate_manager.check_distrib_complete()
            if sysinfo.system_type()["cloud_system"] is True:
                if called_by_ignition is False:
                    shell.sudo_call("systemctl restart webcontrol")
            else:
                shell.sudo_call("systemctl restart webcontrol")

class CheckPortResource(object):
    """
    REST resource to verify a port status

    Methods:

    GET -- return port information
    """
    def GET(self):
        """
         Return a json string with port information.

        Data format:

        {   "is_well_known":   true|false,
            "service_name":    "server name"|""
            "is_opened":       true|false
        }
        """
        web.header("Content-Type","application/json")

        port = int(web.input(port=None).port)

        well_known_info = network.well_known_port(port)

        is_well_known_port = False
        service_name = ""

        if well_known_info is not None:
            is_well_known_port = True
            service_name = well_known_info[1]

        port_information = {'is_well_known': is_well_known_port,
                            'service_name': service_name,
                            'is_opened': network.is_port_in_use(port) }
        return json.dumps(port_information)


class DateResource(object):
    """
    REST resource to manage system date

    Methods:

    GET -- return system date
    PUT -- set system date
    """

    @synchronized(date_lock)
    def GET(self):
        """
        Return system date in locale format
        """
        return utils.format_timestamp(systime.date())


    @synchronized(date_lock)
    def PUT(self):
        """
        Set system date.

        Input:

        timestamp -- date to set as date string
        format    -- the format of the date (optional , defaults as the locale's format)

        Error code:

        400 -- invalid timestamp
        """
        try:
            timestamp = utils.string_to_timestamp(web.input(timestamp=None).timestamp,
                                                  web.input(format=None).format)
            systime.set_date(timestamp)
            audit.log(i18n.custom_gettext('set date to %s') % timestamp)
        except (ValueError, TypeError):
            log.exception(TAG, 'invalid date value %s' % timestamp)
            web.ctx.status = '400'


class StandardDateResource(object):
    """
    REST resource to manage system date

    Methods:

    GET -- return system date in standard format: yyyy-mm-dd
    """

    @synchronized(date_lock)
    def GET(self):
        """
        Return system date as POSIX timestamp
        """
        return utils.format_timestamp(systime.date(), utils.DATE_FORMAT)


class TimezoneResource(object):
    """
    REST resource to manage system date

    Methods:

    GET -- return system timezone
    PUT -- set system timezone
    """

    @synchronized(date_lock)
    def GET(self):
        """
        Return a json string with system timezone info.

        Data format:

        { "code":       "country code",
          "name":       "timezone name",
          "comments":   "comments" 
        }
        """
        web.header("Content-Type","application/json")        
        return json.dumps(timezone_manager.get_timezone(), default=utils.encode_json)

    @synchronized(date_lock)
    def PUT(self):
        """
        Set system date.

        Input:

        name -- timezone name (as used in /etc/sysconfig/clock)

        Error code:

        400 -- invalid timezone
        """
        name = web.input(name=None).name
        name = str(name)
        if name in timezone_manager.TIMEZONES:
            timezone_manager.set_timezone(name)
            audit.log(i18n.custom_gettext('set timezone to %s') % name)
        else:
            log.error(TAG, 'invalid timezone %s' % name)
            web.ctx.status = '400'


class UTCResource(object):
    """
    REST resource to manage UTC setting for internal clock

    Methods:

    GET -- return system timezone
    PUT -- set system timezone
    """

    @synchronized(date_lock)
    def GET(self):
        """
        Return True if internal clock is set to UTC.
        """
        web.header("Content-Type","application/json")
        return json.dumps(timezone_manager.get_utc(), default=utils.encode_json)

    @synchronized(date_lock)
    def PUT(self):
        """
        Set internal clock to UTC.

        Input:

        utc -- True/False
        """
        utc = web.input(utc=None).utc
        utc = str(utc)
        timezone_manager.set_utc(utc.lower() == 'true')
        audit.log(i18n.custom_gettext('set UTC to %s') % utc)


class TimezonesResource(object):
    """
    REST resource to read all available timezones

    Methods:

    GET -- return available timezones
    """

    def GET(self):
        """
        Return a json string with all timezones.

        Data format:

        [ { "code":       "country code",
            "name":       "timezone name",
            "comments":   "comments"  },
          ...
        ]
        """
        web.header("Content-Type","application/json")
        return json.dumps(list(timezone_manager.TIMEZONES.values()), default=utils.encode_json)


class NTPResource(object):
    """
    REST resource to manage NTP settings.

    Methods:

    GET -- return NTP settings
    PUT -- modify NTP settings
    """

    @synchronized(ntp_lock)
    def GET(self):
        """
        Return a json object with NTP settings.

        Data format:

        {
            "use_ntp":          true|false,
            "servers":          ["ntp_server1_url", "ntp_server2_url"]
            "use_local_source": true|false,
            "sync":             true|false
        }
        """
        web.header("Content-Type","application/json")        
        return json.dumps(ntp_manager.get_ntp(), default=utils.encode_json)

    @synchronized(ntp_lock)
    def PUT(self):
        """
        Modify NTP settings.

        Input:

        use_ntp             -- true|false - whether the system time should use NTP
        servers             -- list of NTP servers, separated by ','
        use_local_source    -- true|false - whether the system time should use local time source
        sync                -- true|false - whether the system clock should be synchronized to
                               the time server before starting the service
        
        Error code:

        400 -- invalid input (a parameter is missing)
        """
        use_ntp = web.input(use_ntp=None).use_ntp
        servers = web.input(servers=None).servers
        use_local_source = web.input(use_local_source=None).use_local_source
        ntp_sync = web.input(sync=None).sync
        
        # if use_ntp and servers and use_local_source and ntp_sync:
        ntp_config = {'use_ntp':            (use_ntp == 'true'),
                      'use_local_source':   (use_local_source == 'true'),
                      'sync':               (ntp_sync == 'true'),
                      'servers': []}
        if servers:
            ntp_config['servers'] = str(servers).split(',')
        ntp_manager.set_ntp(ntp_config)


class LANResource(object):
    """
    REST resource to manage IP Office LAN settings.

    Methods:

    GET -- return LAN settings
    PUT -- modify LAN settings
    """

    RETRY_COUNT = 40

    @synchronized(ipoffice_lan_config_lock)
    def GET(self):
        """
        Return a json object with LAN settings.
        """
        try:
            lan_info = {
                "interfaces": network.get_all_interfaces(),
                "lans": network.get_ipoffice_lan_info(),
                "tc_lan1": network.get_traffic_control_info(1),
                "tc_lan2": network.get_traffic_control_info(2)
            }
            web.header("Content-Type","application/json")
            return json.dumps(lan_info, default=utils.encode_json)
        except network.ConfigFileError as e:
            log.exception(TAG, 'unable to find IP Office config file')
            web.ctx.status = '400'
            return "%s" % e

    @synchronized(ipoffice_lan_config_lock)
    def PUT(self):
        """
        Apply LAN settings

        Input:
        - lan_id          -- IPOFFICE_LAN variable to modify
        - eth_id          -- Interface name to be used
        - lan_check_id    -- the id of the other IPOFFICE_LAN variable
        - services_ids    -- services to restart after the LAN mapping is changed
        - traffic_control -- enable/disable traffic control for LAN interface
        """
        lan_id = web.input(lan_id=None).lan_id
        eth_id = web.input(eth_id=None).eth_id
        lan_check_id = web.input(lan_check_id=None).lan_check_id
        services_ids = web.input(services_ids=None).services_ids
        traffic_control = web.input(traffic_control=None).traffic_control
        create_cert = str(web.input(create_cert=None).create_cert)
        self_signed = str(web.input(self_signed=None).self_signed)

        if lan_id and eth_id and lan_check_id and traffic_control:
            try:
                network.set_ipoffice_lan(str(lan_id), str(eth_id), str(lan_check_id), str(traffic_control))
                audit.log(i18n.custom_gettext("Changed IP Office LAN%s") % lan_id)
                if services_ids:
                    services_manager.restart_services(services_ids.split(","))
                    audit.log(i18n.custom_gettext("Restarted the following services: %s") % services_ids)
                if create_cert == "true":
                    generate_certificate_manager.create_cert(self_signed=self_signed)
            except network.ConfigFileError as e:
                web.ctx.status = '400'
                return "%s" %e
        else:
            web.ctx.status = '400'


class SubinterfaceConfigurationResource(object):
    """
    REST resource that creates or deletes subinterfaces

    Methods:

    PUT  -- create a new sub-interface
    POST -- delete an existing sub-interface
    """
    @synchronized(subinterface_config_lock)
    def PUT(self):
        """
        Creates a new sub-interface

        Input:

        name        -- interface name
        ipaddr      -- interface IP address
        netmask     -- interface netmask
        gateway     -- interface gateway
        use_dhcp    -- True|False
        auto_dns    -- yes|no (if automatically obtain DNS information from provider)
        """
        interf = network.Interface()
        interf.name = str(web.input(name=None).name)
        interf.ipaddr = str(web.input(ipaddr='').ipaddr)
        interf.netmask = str(web.input(netmask='').netmask)
        interf.gateway = str(web.input(gateway='').gateway)

        interf.use_dhcp = bool(web.input(use_dhcp=False).use_dhcp)
        interf.auto_dns = str(web.input(auto_dns='yes').auto_dns)
        if interf.name:
            network.ping_ip(interf.ipaddr)
            network.create_interface(interf)
            shell.sudo_call("service network restart")
        else:
            log.error(TAG, "No subinterface name provided")

    @synchronized(subinterface_config_lock)
    @configuration.csrf_protected
    def POST(self):
        """
        Deletes an existing sub-interface

        Input:

        name -- name of the sub-interface to be deleted. Example: eth0.1
        """
        name = str(web.input(name=None).name)
        if id:
            try:
                network.delete_interface(name)
                shell.sudo_call("service network restart")
            except IOError:
                log.error(TAG, "ifcfg-%s file not found" % name)
        else:
            log.error(TAG, "No subinterface name provided")


class BannerResource(object):
    """
    REST resource to display banner message.

    Methods:

    GET -- return the banner message
    """

    def GET(self):
        """
        Return a json object with banner message.
        """
        banner_content = shell.banner()
        banner_config = {'show_banner': False}
        banner = ""
        show_banner_file = configuration.SHARED_CONFIGURATION['webapp']['show_banner_file']
        if os.path.isfile(show_banner_file):
            banner_config["show_banner"] = True
        for row in banner_content.split("\n"):
            if not (row.startswith("Kernel") or row.startswith("kernel")):
                banner += row + "\n"
        web.header("Content-Type","application/json")
        return json.dumps({"banner" : banner, "show_banner" : banner_config["show_banner"]})

    def PUT(self):
        """
        Changes the banner message with the given one
        """
        banner_text = "%s" % (web.input(banner_text="").banner_text)
        show_banner = web.input(show_banner=None).show_banner
        banner_config = {'show_banner':            (show_banner == 'true')}
        banner_file = configuration.SHARED_CONFIGURATION['webapp']['banner_file']
        show_banner_file = configuration.SHARED_CONFIGURATION['webapp']['show_banner_file']
        shell.sudo_call("touch %s" % banner_file)
        if banner_config['show_banner']:
            shell.sudo_call("touch %s" % show_banner_file)
        else:
            shell.sudo_call("rm -f %s" % show_banner_file)
        try:
            with shell.file_open( banner_file, "w" ) as outfile:
                outfile.write(banner_text)
        except IOError:
            log.SHARED_LOGGER.exception("unable to write banner file")

            
class HardwareInfo(object):
    """
    REST resource to return hardware information.

    Methods:

    GET -- return the hardware information
    """

    def GET(self):
        """
        Return a json object with hardware information.

        Data format:
        {
            "cpu_model": "Dual",
            "cpu_mhz": "2400",
            "cpu_cores": "2",
            "disk_total": "500000000",
            "disk_raid_levels": "[[raid0], [raid1], ... [raid5]]",
            "disk_array_types": [[drive1, raidType1], [drive2, raidType2], ... [drive5, raidType5]],
            "mem_total": "12000000",
            "virtualised": "true"
        }
        """
        actual_hw_profile = sysinfo.hardware_info()
        web.header("Content-Type","application/json")
        return json.dumps({
            'cpu_model': actual_hw_profile.cpu.model,
            'cpu_mhz': actual_hw_profile.cpu.mhz,
            'cpu_cores' : actual_hw_profile.cpu.cores,
            'disk_total':utils.format_bytes(actual_hw_profile.disk.total * 1024),
            'disk_raid_levels': actual_hw_profile.disk.supportedLevels,
            'disk_array_types' : actual_hw_profile.disk.arraysTypes,
            'mem_total': utils.format_bytes(actual_hw_profile.mem.total * 1024),
            'virtualised': os.path.isfile(configuration.SHARED_CONFIGURATION['virtualization']['virtualization_file'])
            })

class VirtualSystemResource(object):
    """
    REST resource to determine if the system is virtual.

    Methods:

    GET -- return true if the system is virtual , false otherwise
    """

    def GET(self):
        """
        Return true if the system is virtual , false otherwise
        """
        web.header("Content-Type", "application/json")
        return json.dumps(sysinfo.system_type())

class AutoconfigStatusResource(object):
    """
    REST resource used to determine if autoconfig process is running.

    Methods:

    GET -- return true if autoconfig is running, false otherwise.
    """

    def GET(self):
        """
        Return true if autoconfig is running, false otherwise.
        """
        autoconfig_running = False
        if os.path.isfile(configuration.SHARED_CONFIGURATION['virtualization']['virtualization_file']) and \
                os.path.isfile(configuration.SHARED_CONFIGURATION['autoconfig']['cloud_flag']):
            if not os.path.isfile(configuration.SHARED_CONFIGURATION['autoconfig']['complete_flag']):
                autoconfig_running = True
        web.header("Content-Type", "application/json")
        return json.dumps({"autoconfig_running": autoconfig_running})


class GoldEditionResource(object):
    """
    REST resource to determine if Server Edition is gold

    GET -- return false if the release type is Server Edition or Application Server non-gold version, true otherwise
    """

    def GET(self):
        web.header("Content-Type", "application/json")
        return json.dumps(sysinfo.gold_edition())


class FlagResource(object):
    """
    REST resource for flag operations

    GET    -- check if flag exists
    PUT    -- create flag
    POST   -- delete flag
    """

    def GET(self):
        flag_id = str(web.input(flag_id=None).flag_id)
        exists = False
        if flag_id is not None and flag_id is not "None":
            if os.path.exists(configuration.SHARED_CONFIGURATION["flags"][flag_id]):
                exists = True
        web.header("Content-Type", "application/json")
        return json.dumps(exists)

    def PUT(self):
        flag_id = str(web.input(flag_id=None).flag_id)
        if flag_id is not None and flag_id is not "None":
            shell.sudo_call("touch %s" % configuration.SHARED_CONFIGURATION["flags"][flag_id])

    @configuration.csrf_protected
    def POST(self):
        flag_id = str(web.input(flag_id=None).flag_id)
        if flag_id is not None and flag_id is not "None":
            shell.sudo_call("rm %s" % configuration.SHARED_CONFIGURATION["flags"][flag_id])



class NewHardware(object):
    """
    REST resource to manage an additional hard disk

    Methods:

    GET    -- retrieve additional hard disk information
    PUT    -- mount the additional hard disk
    DELETE -- unmount the additional hard disk
    """

    NOT_FORMATTED_ERRORS = ["you must specify the filesystem type", "unknown filesystem type '(null)'"]
    RESTRICTED_FOLDERS = ["/",
                          "/proc/",
                          "/sys/",
                          "/etc/",
                          "/usr/",
                          "/usr3/",
                          "/boot/",
                          "/bin/",
                          "/sbin/",
                          "/lib/",
                          "/lib64/",
                          "/dev/"]
    MOUNT_POINT_ROOT = "/additional-hdd#"
    LOCALIZATION_PLACEHOLDERS = ["bytes",
                                 "TB",
                                 "GB",
                                 "MB",
                                 "KB",
                                 "Virtual disk",
                                 "disk"]

    def get_default_mount_point(self):
        count = 1
        while os.path.isdir(self.MOUNT_POINT_ROOT + str(count)):
            count += 1
        return self.MOUNT_POINT_ROOT + str(count)

    def set_label(self, path):
        label_script = configuration.SHARED_CONFIGURATION['hdd']['label_script']
        if path[-1].isdigit():
            label = 'additional-hdd' + path[-1]
        else:
            label = 'additional-hdd'
        set_label = "%s label %s %s" % (label_script, path, label)
        is_set = "%s check %s" % (label_script, path)
        try:
            check = subprocess.Popen((set_label), stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True, universal_newlines=True)
            output, error = check.communicate()
            if "Couldn't find valid filesystem superblock." in output:
                return None
            check = subprocess.Popen((is_set), stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True, universal_newlines=True)
            output, error = check.communicate()
            if error and check.returncode:
                return None
            else:
                if label in output:
                    return label
                else:
                    return None
        except subprocess.CalledProcessError as e:
            return None

    def save_mount_point(self, label, mount_path):
        """
        Writes a mount point into the /etc/fstab file
        """
        mount_point_file = configuration.SHARED_CONFIGURATION['hdd']['mount_point_file']
        with shell.file_open(mount_point_file) as infile:
            lines = infile.readlines()
            if "additional-hdd" in label:
                lines.append("\nLABEL=%s %s                       ext4    defaults,nofail        0 0" % (label,mount_path))
            else:
                lines.append("\n%s %s                       ext4    defaults,nofail        0 0" % (label,mount_path))
        with shell.file_open(mount_point_file, 'w') as outfile:
            outfile.writelines(lines)

    def remove_mount_point(self, path, label):
        """
        Removes a mount point from /etc/fstab
        """
        mount_point_file = configuration.SHARED_CONFIGURATION['hdd']['mount_point_file']
        partitions = self.get_partitions(path)
        paths = []

        if len(partitions) is 0:
            if len(self.get_mount_point(path)) is not 0:
                paths.append(path)
        else:
            for partition in partitions:
                if len(self.get_mount_point(partition['partition'])) is not 0:
                    paths.append(partition['partition'])

        with shell.file_open(mount_point_file) as infile:
            lines = infile.readlines()
            append_lines = list(lines)
            for line in lines:
                if line.startswith("#"):
                    continue
                if line.startswith('\n') or line.startswith(' '):
                    append_lines.remove(line)
                    continue
                for p in paths:
                    if p in line or label in line:
                        append_lines.remove(line)
                        continue
        with shell.file_open(mount_point_file, 'w') as outfile:
            outfile.writelines(append_lines)

    def get_mount_point(self, path):
        """
        Return the mount point of a specified drive or partition
        """
        mounts_file = configuration.SHARED_CONFIGURATION['hdd']['mounts_file']
        with shell.file_open(mounts_file) as infile:
            lines = infile.readlines()
            for line in lines:
                if line.startswith(path):
                    return line.split(" ")[1]
        return ""

    def format_path(self, mount_path):
        """
        Adds "/" characters to the beginning and end of a path , if they don't already exist
        """
        if mount_path:
            if not mount_path.startswith("/"):
                mount_path = "/" + mount_path
            if not mount_path.endswith("/"):
                mount_path += "/"
        return mount_path

    def get_remaining_space (self, path, partition1_size="", partition2_size="", partition3_size=""):
        """
        Returns effective space available on a drive
        """
        check_space_script = configuration.SHARED_CONFIGURATION['hdd']['hdd_check_space_script']
        space_available = shell.sudo_execute("%s %s %s %s %s"
            % (check_space_script, path, partition1_size, partition2_size, partition3_size), parse=shell.RAW).decode()
        return "%s" % space_available

    def get_partitions(self, path):
        """
        Returns an array containing the partitions of a specified drive
        """
        partitions_array = []
        partitions = shell.sudo_execute("fdisk -l %s" % path, parse=shell.LINES)
        for line in partitions:
            if len(line) > 0:
              if line.startswith("/dev/"):
                partition = line.split()[0]
                index = partition[-1:]
                partitions_array.append({"partition": partition, "index": index})
        if len(partitions_array) == 0: #unpartitioned disk
            for line in partitions:
              if len(line) > 0:
                  if line.startswith("Disk /"):
                    partition = line.split("Disk ")[1].split()[0].split(":")[0]
                    index = ''
                    partitions_array.append({"partition": partition, "index": index})
        return partitions_array

    def get_root_mount_point(self, path):
        """
        Returns the folder containing the mount points of a drive's partitions
        """
        partitions = self.get_partitions(path)
        if len(partitions) is 0:
            root_mount_point = self.get_mount_point(path)
        else:
            partition_mount_point = self.get_mount_point(partitions[0]['partition'])
            if partitions[0]['partition'][-1].isdigit():
                root_mount_point = os.path.dirname(partition_mount_point)
            else:
                root_mount_point=partition_mount_point
        if root_mount_point is not "" and not root_mount_point.endswith('/'):
            return root_mount_point + "/"
        return root_mount_point
    def get_partitions_mount_point(self, path):

        """
        Returns the list of folders containing the mount points of a drive's partitions
        """
        partitions = self.get_partitions(path)
        if len(partitions) is 0:
            partition_mount_point = self.get_mount_point(path)
        else:
            partition_mount_point = ""
            for partition in partitions:
                if self.get_mount_point(partition['partition']) is not "":
                    partition_mount_point += self.get_mount_point(partition['partition']) + '/\n'
        if partition_mount_point is not "":
            return partition_mount_point
        else:
            partition_mount_point  = "-"
        return partition_mount_point
    def format(self, path):
        """
        Formats a drive
        """
        format_cmd = "mkfs.ext4 -F %s" % path
        shell.sudo_call(format_cmd)

    def mount(self, activate, path, mount_path):
        """
        Mounts a drive
        """
        mount_script = configuration.SHARED_CONFIGURATION['hdd']['hdd_mount_script']
        label = self.set_label(path)
        if label == None:
            label = path
        mount_cmd = "%s mount %s %s" % (mount_script, label, mount_path)
        umount_cmd = "%s umount %s %s" % (mount_script, path, mount_path)
        format_error = False

        if activate:
            self.remove_mount_point(path, label)
            shell.sudo_call(umount_cmd)
            if activate == 'True':
                if not os.path.isdir(mount_path):
                    shell.sudo_call("mkdir -p %s" % mount_path)
                mount = shell.execute(mount_cmd, parse=shell.LINES)
                if mount:
                    if len(mount) > 0:
                        for line in mount:
                            for error in self.NOT_FORMATTED_ERRORS:
                                if error in line:
                                    format_error = True
                                    self.format(path)
                                    break
                            if format_error is True:
                                break
                        shell.sudo_call(mount_cmd)
                self.save_mount_point(label, mount_path)
                shell.sudo_call("chmod a+rw -R %s" % mount_path)

    def parse_value(self, value):
        """
        Parses a value received from Linux, for localization purposes.
        """
        for string in self.LOCALIZATION_PLACEHOLDERS:
            if string in value:
                value = value.replace(string, i18n.custom_gettext(string))
        return value

    @synchronized(new_hw_lock)
    def GET(self):
        """
        Return JSON object with new hardware information.

        Error codes:

        400 -- Error occurred while reading VNC settings (VNC configuration file couldn't be read).
        """
        new_disks_cmd = "%s get_options" % configuration.SHARED_CONFIGURATION['hdd']['hdd_details_script']
        new_disks = shell.sudo_execute(new_disks_cmd, parse=shell.LINES)
        hdd_info = []
        if new_disks:
            for disk in new_disks:
                if disk is not "":
                    disk_info = {"name": disk,
                                 "remaining_space": self.get_remaining_space(disk),
                                 "default_mount_point": self.get_default_mount_point(),
                                 "root_mount_point": self.get_root_mount_point(disk),
                                 "partitions_mount_point": self.get_partitions_mount_point(disk),
                                 "restricted_folders": self.RESTRICTED_FOLDERS,
                                 "virtualized": os.path.isfile(configuration.SHARED_CONFIGURATION['virtualization']['virtualization_file'])}
                    cmd = "%s %s" % (configuration.SHARED_CONFIGURATION['hdd']['hdd_details_script'], disk)
                    description = shell.sudo_execute(cmd, parse=shell.LINES)
                    for line in description:
                        if line is not "":
                            key = line.split(":")[0].strip().replace(" ", "_").lower()
                            value = self.parse_value(line.split(":")[1].strip())
                            disk_info[key] = value
                    hdd_info.append(disk_info)

        web.header("Content-Type","application/json")
        return json.dumps(hdd_info, default=utils.encode_json)

    @synchronized(new_hw_lock)
    def PUT(self):
        """
        Mounts the new hardware
        """
        activate = str(web.input(activate=None).activate)
        path = str(web.input(path=None).path)
        mount_path = self.format_path(str(web.input(mount_path=None).mount_path))

        partitions = self.get_partitions(path)
        if len(partitions) is 0:
            self.mount(activate, path, mount_path)
        else:
            for partition in partitions:
                if partition["partition"][-1].isdigit():
                    self.mount(activate, partition["partition"], "%spartition%s/" % (mount_path, partition["index"]))
                else:
                    self.mount(activate, partition["partition"], mount_path)

    @configuration.csrf_protected
    @synchronized(new_hw_lock)
    def POST(self):
        """
        Creates and mounts partitions
        """
        activate = str(web.input(activate=None).activate)
        path = str(web.input(path=None).path)
        mount_path = self.format_path(str(web.input(mount_path=None).mount_path))
        format_drive = str(web.input(format_drive=None).format_drive)
        partition1_size = int(float(web.input(partition1_size=None).partition1_size) * 1024)
        partition2_size = int(float(web.input(partition2_size=None).partition2_size) * 1024)
        partition3_size = int(float(web.input(partition3_size=None).partition3_size) * 1024)
        check_space = str(web.input(check_space=None).check_space)

        if check_space and check_space == "True":
            return self.get_remaining_space(path, partition1_size, partition2_size, partition3_size)
        else:
            create_partitions_script = configuration.SHARED_CONFIGURATION['hdd']['hdd_partition_script']
            create_partitions_cmd = ""

            if format_drive and format_drive == "True":
                self.format(path)

            partitions = [partition1_size, partition2_size, partition3_size]
            index = 1
            for partition in partitions:
                if partition > 0:
                    partition_cmd = 'echo n; echo p; echo %s; echo " "; echo "+%sMB"; ' % (index, partition)
                    create_partitions_cmd += partition_cmd
                index += 1
            cmd = "%s '%s' %s" % (create_partitions_script, create_partitions_cmd, path)
            shell.sudo_call(cmd)
            for partition in self.get_partitions(path):
                if format_drive and format_drive == "True":
                    self.format(partition["partition"])
                self.mount(activate, partition["partition"], "%spartition%s/" % (mount_path, partition["index"]))


class Firewall(object):
    """
    REST resource to manage Firewall settings

    Methods:

    GET    -- get Firewall status
    PUT    -- start or stop the Firewall
    """

    DEFAULT_PORTS_FILE = "/opt/Avaya/scripts/config_files/default_allowed_ports"
    ALLOWED_PORTS_FILE = "/opt/Avaya/security/allow_ports"
    SET_FIREWALL_SCRIPT = "/opt/Avaya/security/set_firewall.sh "

    @synchronized(firewall_lock)
    def GET(self):
        """
        Return firewall status
        """

        ports = {'default_tcp': None,
                 'default_udp': None}
        utils.read_config(self.DEFAULT_PORTS_FILE, ports)
        with shell.file_open(self.ALLOWED_PORTS_FILE) as f:
            lines = f.readlines()
        for line in lines:
            if line.startswith('TCP'):
                ports['allowed_tcp'] = line.strip().split(":")[1]
            if line.startswith('UDP'):
                ports['allowed_udp'] = line.strip().split(":")[1]
            if line.startswith('Filtering'):
                ports['filtering'] = line.strip().split(":")[1]
            if line.startswith('Activate'):
                status = line.strip().split(":")[1]

        web.header("Content-Type","application/json")
        return json.dumps({"status": status, "ports": ports}, default=utils.encode_json)

    @synchronized(firewall_lock)
    def PUT(self):
        """
        Apply firewall settings
        """
        activate = str(web.input(activate=None).activate)
        enable_tcp = str(web.input(enable_tcp="").enable_tcp)
        enable_udp = str(web.input(enable_udp="").enable_udp)
        filtering = web.input(filtering=None).filtering

        with shell.file_open(self.ALLOWED_PORTS_FILE) as infile:
            lines = infile.readlines()
        for i, line in enumerate(lines):
            if line.startswith('TCP'):
                lines[i] = "TCP:%s\n" % enable_tcp
            elif line.startswith('UDP'):
                lines[i] = "UDP:%s\n" % enable_udp
            elif line.startswith('Filtering'):
                if filtering:
                    lines[i] = "Filtering:%s\n" % filtering
            elif line.startswith('Activate'):
                if activate == "true":
                    lines[i] = "Activate:on\n"
                else:
                    lines[i] = "Activate:off\n"

        with shell.file_open(self.ALLOWED_PORTS_FILE, "w") as outfile:
            outfile.writelines(lines)

        if activate == "true":
            shell.sudo_call("%s" % self.SET_FIREWALL_SCRIPT)
        elif activate == "false":
            shell.sudo_call("%s flush" % self.SET_FIREWALL_SCRIPT)


class HTTPServerResource(object):

    HTTP_BACKUP_SCRIPT = "/opt/Avaya/security/httpBackup.sh"

    """
    REST resource to manage HTTP Server settings

    Methods:

    GET -- return HTTP Server settings
    PUT -- change HTTP Server settings
    """

    @synchronized(http_server_lock)
    def GET(self):
        """
        Return HTTP Server status.
        """
        output = shell.sudo_execute("%s status" % self.HTTP_BACKUP_SCRIPT, parse=shell.RAW)
        filestore = False
        if b"allowed" in output:
            filestore = True
        status = {"filestore": filestore}
        web.header("Content-Type", "application/json")
        return json.dumps(status, default=utils.encode_json)


    @synchronized(http_server_lock)
    def PUT(self):
        """
        Enable/disable file store for backup/restore.
        """
        enable = str(web.input(enable=None).enable)
        if enable:
            if enable == "true":
                shell.sudo_call("%s enable" % self.HTTP_BACKUP_SCRIPT)
            else:
                shell.sudo_call("%s disable" % self.HTTP_BACKUP_SCRIPT)


class VNC(object):
    """
    REST resource to manage VNC settings

    Methods:

    PUT    -- start the VNC server
    DELETE -- stop the VNC server
    POST   -- change VNC server password
    """

    def GET(self):
        """
        Return JSON object with VNC settings. Format:
        {
            port: 5807,
            running: True|False
        }

        Error codes:

        400 -- Error occurred while reading VNC settings (VNC configuration file couldn't be read).
        """
        try:
            vnc_settings  = vnc.get_vnc_settings()
            web.header("Content-Type","application/json")
            return json.dumps(vnc_settings, default=utils.encode_json)
        except vnc.ConfigFileError as e:
            log.exception(TAG, 'unable to find VNC config file')
            web.ctx.status = '400'
            return "%s" % e

    def PUT(self):
        """
        Starts the VNC server
        """
        try:
            port_id = vnc.port_to_id(vnc.get_vnc_settings()['port'])
            shell.call("vncserver -geometry 800x600 :%s" % port_id)
        except vnc.ConfigFileError as e:
            log.exception(TAG, 'unable to find VNC config file')
            web.ctx.status = '400'
            return "%s" % e


    def DELETE(self):
        """
        Stops the VNC server
        """
        try:
            port_id = vnc.port_to_id(vnc.get_vnc_settings()['port'])
            shell.call("vncserver -kill :%s" % port_id)
        except vnc.ConfigFileError as e:
            log.exception(TAG, 'unable to find VNC config file')
            web.ctx.status = '400'
            return "%s" % e

    def POST(self):
        """
        Changes the VNC Server's password
        """
        password = str(web.input(password=None).password)
        port = str(web.input(port=None).port)
        if password:
            try:
                vncpasswd_script = configuration.SHARED_CONFIGURATION['password']['vnc_password_script']
                cmd = "su - Administrator /bin/bash -c \"%s %s\"" % (vncpasswd_script, password)
                shell.sudo_call(cmd)
            except (IOError, OSError):
                log.exception(TAG, "unable to set VNC Server password")
                web.ctx.status = '400'
        try:
            vnc.set_vnc_settings(port)
        except vnc.ConfigFileError as e:
            log.exception(TAG, 'unable to find VNC config file')
            web.ctx.status = '400'
            return "%s" % e


class CheckHostnameResource(object):

    def PUT(self):

        status = False
        hostname = str(web.input(hostname=None).hostname)
        if hostname:
            script = configuration.SHARED_CONFIGURATION['network']['check_hostname_script']
            output = shell.sudo_execute("%s %s" % (script, hostname), parse=shell.RAW).strip()
            if output.decode() == "true":
                status = True

        web.header("Content-Type","application/json")
        return json.dumps(status, default=utils.encode_json)


