# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
REST API for updates and Windows clients.
"""

import os
import json
import web
import wsgiref.util
import mimetypes
import time
import datetime
import threading
import glob
import stat

import audit
import services_rest_api

import core.updates.applications
import core.updates.system
import core.updates.windows_clients

from core.common import utils
from core.common import log
from core.common import i18n
from core.common import configuration
from core.common import version
from core.common.decorators import synchronized
from core.settings import repositories
from core.system import yum
from core.system import rpm
from core.system import shell


__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

# Logging tag
TAG = 'updates-rest-api'

os_updates_lock = threading.Lock()
app_all_lock    = threading.Lock()

# block size for windows client file read
BLOCK_SIZE = 16384

windows_clients_manager = core.updates.windows_clients.WindowsClientsManager()
app_manager             = core.updates.applications.AppManager(services_rest_api.services_manager)
os_update_manager       = core.updates.system.OSUpdateManager()
app_all_update_manager  = core.updates.applications.AllApplicationsUpdateManager()


class WindowsClientsResource(object):
    """
    REST resource to download Windows clients files.

    Method:

    GET     -- list or download Windows clients
    DELETE  -- delete an existing Window client
               works only for local Windows repositories
    """

    FILE_EXTENSION_FILTER   = ('.exe', '.msi', '.zip', '.dmg')
    EXCLUDE_PATTERNS        = ('*AdminCD*.exe', )

    def GET(self, id):
        """
        Return a json string with information about existing Windows
        clients or start a download for an existing client.

        Data format:

        { "is_local_repo": true|false,
          "files": [
                {"modif": "YYYY-MM-DD HH:MM:SS", // last modification date = upload date
                 "id": "file basename",
                 "size": "file size in human readable format" },... ],
        }

        Input:

        id -- the basename of Windows client file to be downloaded

        Error codes:

        404 -- unknown Windows client file
        400 -- error creating Windows client file list
        """
        if id:
            windows_client_file = windows_clients_manager.full_path(id)
            if os.path.isfile(windows_client_file):
                if windows_clients_manager.is_local_repo:
                    try:
                        mime_type = mimetypes.guess_type(id)[0]
                        web.header('Content-Type',  mime_type)
                        stat = os.stat(windows_client_file)
                        web.header('Content-Length', stat.st_size)
                        web.header('Last-Modified',
                            web.http.lastmodified(datetime.datetime.fromtimestamp(stat.st_mtime)))
                        if web.ctx.protocol.lower() == 'https':
                            # add headers to fix issue with IE download over SSL failing when "no-cache" header set
                            web.header('Pragma', 'private')
                            web.header('Cache-Control', 'private,must-revalidate')
                        return wsgiref.util.FileWrapper(open(windows_client_file, 'rb'), BLOCK_SIZE)
                    except (OSError, IOError):
                        log.exception(TAG, 'unable to serve Windows client file <%s>' % windows_client_file)
                        web.ctx.status = '404'
            else:
                log.error(TAG, 'not a valid Windows client package <%s>' % id)
                web.ctx.status = '404'
        else:
            try:
                windows_clients_files = windows_clients_manager.list_repository(
                    extensions_filter=self.FILE_EXTENSION_FILTER,
                    exclude_patterns=self.EXCLUDE_PATTERNS)
                web.header("Content-Type","application/json")
                return json.dumps(windows_clients_files, default=utils.encode_json)
            except core.updates.windows_clients.Error as  e:
                log.exception(TAG, 'unable to list the contents of Windows clients repository')
                web.ctx.status = '400'
                return e.message 

    def DELETE(self, id):
        """
        Delete an existing Windows client file.

        Input:

        id -- the basename of Windows client to be deleted

        Error codes:

        404 -- unknown Windows client file
        """
        try:
            windows_clients_manager.delete(id)
            audit.log(i18n.custom_gettext('deleted Windows client file <%s>') % id)
        except OSError:
            log.exception(TAG, 'invalid Windows client id <%s>' % id)
            web.ctx.status = '404'


class LinuxDownloadsResource(object):
    """
    REST resource to download Linux files

    GET -- Return a list of files from a location or download a specified file
    """

    def GET(self, rpm_file):

        url = configuration.SHARED_CONFIGURATION['repositories']['local']['rpms']
        build_version = version.APP_VERSION['VERSION']
        release_version = version.APP_VERSION['RELEASE'].split(".")[0]
        repo_url = url.replace("version_placeholder", "%s-%s" % (build_version, release_version))

        if rpm_file:
            linux_file = os.path.join(repo_url, rpm_file)
            if os.path.isfile(linux_file):
                try:
                    mime_type = mimetypes.guess_type(rpm_file)[0]
                    web.header('Content-Type',  mime_type)
                    properties = os.stat(linux_file)
                    web.header('Content-Length', properties.st_size)
                    web.header('Last-Modified', web.http.lastmodified(datetime.datetime.fromtimestamp
                                                                      (properties.st_mtime)))
                    if web.ctx.protocol.lower() == 'https':
                        # add headers to fix issue with IE download over SSL failing when "no-cache" header set
                        web.header('Pragma', 'private')
                        web.header('Cache-Control', 'private,must-revalidate')
                    return wsgiref.util.FileWrapper(open(linux_file, 'rb'), BLOCK_SIZE)
                except (OSError, IOError):
                    log.exception(TAG, 'unable to serve Linux file <%s>' % linux_file)
                    web.ctx.status = '404'
            else:
                log.error(TAG, 'not a valid Linux package <%s>' % rpm_file)
                web.ctx.status = '404'
        else:
            try:
                linux_files = []
                for filepath in glob.glob(os.path.join(repo_url, '*.rpm')):
                    stats = os.stat(filepath)
                    filename = os.path.basename(filepath)
                    file_info = {'id': filename,
                                 'size': utils.format_bytes(stats[stat.ST_SIZE]),
                                 'modif': utils.format_timestamp(stats[stat.ST_MTIME])}
                    linux_files.append(file_info)
                web.header("Content-Type","application/json")
                return json.dumps(linux_files, default=utils.encode_json)
            except core.updates.windows_clients.Error as  e:
                log.exception(TAG, 'unable to list the contents of Linux downloads repository')
                web.ctx.status = '400'
                return e.message


class LinuxSystemDownloadsResource(object):
    """
    REST resource to download Linux files

    GET -- Return a list of files from a location or download a specified file
    """

    def GET(self, rpm_file):

        url = configuration.SHARED_CONFIGURATION['repositories']['local']['system_rpms']
        build_version = version.APP_VERSION['VERSION']
        release_version = version.APP_VERSION['RELEASE'].split(".")[0]
        repo_url = url.replace("version_placeholder", "%s-%s" % (build_version, release_version))

        if rpm_file:
            linux_file = os.path.join(repo_url, rpm_file)
            if os.path.isfile(linux_file):
                try:
                    mime_type = mimetypes.guess_type(rpm_file)[0]
                    web.header('Content-Type',  mime_type)
                    properties = os.stat(linux_file)
                    web.header('Content-Length', properties.st_size)
                    web.header('Last-Modified', web.http.lastmodified(datetime.datetime.fromtimestamp
                                                                      (properties.st_mtime)))
                    if web.ctx.protocol.lower() == 'https':
                        # add headers to fix issue with IE download over SSL failing when "no-cache" header set
                        web.header('Pragma', 'private')
                        web.header('Cache-Control', 'private,must-revalidate')
                    return wsgiref.util.FileWrapper(open(linux_file, 'rb'), BLOCK_SIZE)
                except (OSError, IOError):
                    log.exception(TAG, 'unable to serve Linux file <%s>' % linux_file)
                    web.ctx.status = '404'
            else:
                log.error(TAG, 'not a valid Linux package <%s>' % rpm_file)
                web.ctx.status = '404'
        else:
            try:
                linux_files = []
                for filepath in glob.glob(os.path.join(repo_url, '*.rpm')):
                    stats = os.stat(filepath)
                    filename = os.path.basename(filepath)
                    file_info = {'id': filename,
                                 'size': utils.format_bytes(stats[stat.ST_SIZE]),
                                 'modif': utils.format_timestamp(stats[stat.ST_MTIME])}
                    linux_files.append(file_info)
                web.header("Content-Type","application/json")
                return json.dumps(linux_files, default=utils.encode_json)
            except core.updates.windows_clients.Error as e:
                log.exception(TAG, 'unable to list the contents of Linux downloads repository')
                web.ctx.status = '400'
                return e.message


class AppUpdatesResource(object):
    """
    REST resource to list current update status for all applications.

    Methods:

    GET     -- return a json string with update status for all applications
    PUT     -- update or downgrade specified application
    POST    -- install or uninstall application
    """

    def GET(self):
        """
        Return a json string with applications update status.

        Data format:

        [ { id:                     "application ID"
            "status":                enum - 0 = up to date, 1 = outdated,
            "display_name":         "application name",
            "description":          "application description",
            "current_version":      "current version",
            "latest_version":       "latest version available in repository",
            "available_versions":   ["version_1", "version_2",...] },
          ...
        ]


        Error codes:

        400 -- unable to get updates status 
        """
        basic = web.input(basic=None).basic
        if str(basic).title() == 'True':
            return json.dumps(app_manager.get_version_status(), default=utils.encode_json)
        else:
            try:
                status = app_manager.get_status()
                web.header("Content-Type","application/json")
                return json.dumps(status, default=utils.encode_json)
            except core.updates.applications.AccessRepoError as e:
                log.exception(TAG, 'unable to get update status for applications')
                web.ctx.status = '400'
                return "%s" % e

    @synchronized(services_rest_api.services_lock)
    @configuration.csrf_protected
    def PUT(self):
        """
        Update or downgrade application to specified version.

        Input:

        id      -- application ID
        version -- version to update/downgrade to

        Error codes:

        400 -- id or version not set
        """
        id = web.input(id=None).id
        version = web.input(version=None).version

        if id and version:
            try:
                app_manager.upgrade(str(id), str(version))
                audit.log(i18n.custom_gettext('install %s %s') %
                          (app_manager.display_name_for_id(id), version))
            except core.updates.applications.AccessRepoError as e:
                log.exception(TAG, 'unable to install version <%s> for application <%s>' % (version, id))
                web.ctx.status = '400'
                return "%s" % e
            except rpm.RpmError as e:
                log.exception(TAG, 'unable to install version <%s> for application <%s>' % (version, id))
                web.ctx.status = '400'
                return str(e)
        else:
            web.ctx.status = '400'

    @synchronized(services_rest_api.services_lock)
    @configuration.csrf_protected
    def POST(self):
        """
        Install/uninstall application.

        Input:

        id      -- application ID
        action  -- install / uninstall
        version -- version to install

        Error codes:

        400 -- id or version not set or invalid action
        """
        id = web.input(id=None).id
        action = web.input(action=None).action
        version = web.input(version=None).version

        if id and version:
            if action in ('install', 'uninstall'):
                id = str(id)
                version = str(version)
                try:
                    if action == 'install':
                        app_manager.install(id, version)
                        audit.log(i18n.custom_gettext('install %s version %s') %
                                  (app_manager.display_name_for_id(id), version))
                    else:
                        app_manager.uninstall(id)
                        audit.log(i18n.custom_gettext('uninstall %s') %
                                  app_manager.display_name_for_id(id))
                except core.updates.applications.AccessRepoError as e:
                    log.exception(TAG, 'unable to %s application <%s>' % (action, id))
                    web.ctx.status = '400'
                    return "%s" % e
                except rpm.RpmError as e:
                    log.exception(TAG, 'unable to %s application <%s>' % (action, id))
                    web.ctx.status = '400'
                    return str(e)
            else:
                web.ctx.status = '400'
        else:
            web.ctx.status = '400'


class AllAppUpdateResource(object):

    @synchronized(app_all_lock)
    @configuration.csrf_protected
    def PUT(self):
        """
        Update applications to specified version.

        Input:

        packages      -- application IDs
        versions      -- versions to update/downgrade to

        Error codes:

        400 -- id or version not set
        """
        packages = web.input(packages=None).packages
        versions = web.input(versions=None).versions

        if packages and versions:
            update_packages = []
            packages_array = packages.split(",")
            versions_array = versions.split(",")
            if len(packages_array) == len(versions_array):
                for i in range(len(packages_array)):
                    update_packages.append({"id": str(packages_array[i]),
                                            "version": str(versions_array[i])})
                app_all_update_manager.apply_updates(update_packages)
            else:
                web.ctx.status = '400'
        else:
            web.ctx.status = '400'


class AllAppUpdateStatusResource(object):
    """
    REST resource to retrieve information about the status of an Application update process.

    Methods:

    GET -- return the status of Application update
    """
    def GET(self):
        """
        Return json string with Application update status.

        Data format:

        {
            'in_progress':      true if an Application update is in progress
        }
        """
        update_status = app_all_update_manager.get_update_status()
        web.header("Content-Type","application/json")
        return json.dumps(update_status)


class ServiceDependenciesResource(object):
    """
    REST resource used to handle package dependencies of the Avaya services.
    """

    STOPPED_SERVICES = []

    @synchronized(services_rest_api.services_lock)
    def GET(self):
        """
        Returns an array of affected services for a specified package array.

        Data format:

        [
            {'id': service_id, 'display_name': service_display_name}, ...
        ]
        """
        packages            = web.input(packages='').packages.split(',')
        affected_services   = []

        for service in services_rest_api.services_manager:
            for package in packages:
                if package in service.dependencies:
                    affected_services.append({'id': service.id,
                                              'display_name': service.display_name})
                    break

        web.header("Content-Type","application/json")
        return json.dumps(affected_services, default=utils.encode_json)

    @synchronized(services_rest_api.services_lock)
    @configuration.csrf_protected
    def PUT(self):
        """
        Stop services affected by dependencies
        """
        services = web.input(services=None).services
        if services:
            for id in services.split(","):
                service = services_rest_api.services_manager[id]
                service.update_run_info()
                if service.is_running():
                    service.stop()
                    time.sleep(10)
                    service.update_run_info()
                    if service.is_running():
                        service.force_stop()
                    self.STOPPED_SERVICES.append(service)
        else:
            web.ctx.status = '400'

    @synchronized(services_rest_api.services_lock)
    @configuration.csrf_protected
    def POST(self):
        """
        Start services which were previously stopped
        """
        for service in self.STOPPED_SERVICES:
            service.update_run_info()
            if not service.is_running():
                service.start()
        self.STOPPED_SERVICES = []


class OSUpdatesResource(object):
    """
    REST resource to manage OS updates.

    Methods:

    GET -- return available updates
    PUT -- apply selected updates
    """
    @synchronized(os_updates_lock)
    def GET(self):
        """
        Return json string with available updates.

        Data format:

        [ { "name":         "package name",
            "new_version":  "latest version available" },
          ...
        ]

        Error codes:

        400 -- unable to get OS updates status
        """
        try:
            status = os_update_manager.get_available_updates()
            kernel_reboot = os_update_manager.check_kernel()
            web.header("Content-Type","application/json")
            return json.dumps({"status": status, "kernel_reboot": kernel_reboot}, default=utils.encode_json)
        except yum.YumError as e:
            log.exception(TAG, 'unable to get update status for OS')
            web.ctx.status = '400'
            return "%s" % e

    @synchronized(os_updates_lock)
    @configuration.csrf_protected
    def PUT(self):
        """
        Apply selected updates.

        Input:

        packages = a comma separated list of packages to be updated 
                   if parameter is missing or it contains no packages
                   then the entire system will be updated
        """
        packages = web.input(packages=None).packages
        if packages:
            packages_list = map(str, packages.split(','))
        else:
            packages_list = [package_info[0] for package_info in yum.list_updates()]
        try:
            audit.log(i18n.custom_gettext('apply OS updates'))
            os_update_manager.apply_updates(packages_list)
        except core.updates.system.Error as e:
            log.exception(TAG, 'OS update error')
            web.ctx.status = '400'
            return e.message


class OSUpdateStatusResource(object):
    """
    REST resource to retrieve information about the status of an OS update process.

    Methods:

    GET -- return the status of OS update
    """
    def GET(self):
        """
        Return json string with OS update status.

        Data format:
        
        {
            'in_progress':      true if an OS update is in progress,
            'current':          the name of the package which is currently updated,
            'total:':           total packages to be updated,
            'completed':        how many packages were updated,
            'require_reboot':   true is an update requires a reboot (kernel updates)
        }
        """
        update_status = os_update_manager.get_update_status()
        web.header("Content-Type","application/json")
        return json.dumps(update_status)


class ClearCacheResource(object):
    """
    REST resource to clean the local cache.

    Methods:

    PUT -- cleans the local cache
    """
    @configuration.csrf_protected
    def PUT(self):
        files_to_delete = repositories.list_unused_cached_files()
        log.info(TAG, "clearing local cache - files to remove: %s" %
                      "\n".join(map(str, files_to_delete)) )
        shell.rm(files_to_delete)


class UpdatesAvailableResource(object):
    """
    REST resource used to list available updates for applications and OS.

    Methods:

    GET -- list available updates for applications and OS.
    """
    def GET(self):
        """
        Return true if there are updates available.

        Error codes:

        400 -- unable to get updates status
        """
        try:
            updates_available = False
            apps_status = app_manager.get_status()
            outdated_apps = [app for app in apps_status if app.is_outdated()]
            if outdated_apps:
                updates_available = True
            else:
                update_status = os_update_manager.get_update_status()
                if update_status['in_progress']:
                    updates_available = True
                else:
                    outdated_packages = os_update_manager.get_available_updates()
                    if outdated_packages:
                        updates_available = True
            web.header("Content-Type","application/json")
            return json.dumps(updates_available)
        except (core.updates.applications.AccessRepoError, yum.YumError) as e:
            log.exception(TAG, 'unable to get available updates')
            web.ctx.status = '400'
            return "%s" % e
