# Copyright 2011 Avaya Inc. All Rights Reserved.

"""
Provides uninstall for ABE components. Used by ABE Ignition procedure.
"""

import unittest
import os
import glob

from core.common import configuration
from core.common import version
from core.common import log
from core.system import rpm
from core.system import shell

from core.ext import ipoffice

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2011, Avaya Inc."

# Logging tag
TAG = 'se-apps-manager'

# IDs of ABE applications
APP_IPOFFICE            = 'ipoffice'
APP_VOICEMAIL           = 'voicemail'
APP_ONEXPORTAL          = 'onexportal'
ABE_APPS                = (APP_IPOFFICE, APP_VOICEMAIL, APP_ONEXPORTAL)
ALL_APPS                = configuration.SHARED_CONFIGURATION['applications']

LOCAL_APPS_REPOSITORY       = configuration.SHARED_CONFIGURATION['repositories']['local']['apps']
RPM_FILES                   = {APP_IPOFFICE: ['ipoffice*.rpm', 'ms*.rpm'],
                               APP_VOICEMAIL: ['vmpro*.rpm', 'TTSEnglish*.rpm'],
                               APP_ONEXPORTAL: ['oneXportal*.rpm', 'scs-webconferencing*.rpm', 'webRTCGateway*.rpm', 'ipoffice-webclient*.rpm', 'IPO-Reporting-Analytics*.rpm']}

ONEX_DB_FILE_PATTERN = '/opt/Avaya/onexdb/oneXportal-*.rpm'
DMIDECODE_FILE = '/usr/sbin/dmidecode'


def _package_name_for_id(app_id):
    """
    Returns RPM package name for specified application id.

    Args:
        app_id  --  application id (or service id) as defined
                        in <service>.ini configuration file
    Returns:
        RPM package name
    Throws:
        ValueError if app_id is not one of ABE applications.
    """
    if app_id in ABE_APPS:
        if app_id in ALL_APPS:
            return ALL_APPS[app_id]['package_name'][version.RELEASE_TYPE]
    raise ValueError("invalid application id <%s>" % app_id)


def _service_name_for_id(app_id):
    """
    Returns /etc/init.d service name for specified application id.

    Returns:
        /etc/init.d/<service name>
    Throws:
        ValueError if app_id is not one of ABE applications.
    """
    if app_id in ABE_APPS:
        if app_id in ALL_APPS:
            return ALL_APPS[app_id]['service_name']
    raise ValueError("invalid application id <%s>" % app_id)


def uninstall(app_id):
    """
    Uninstall specified application.
    Also this function will remove application RPM files
    from local applications repository.

    Args:
        app_id  --  application id (or service id) as defined
                        in <service>.ini configuration file
    Throws:
        ValueError if app_id is not one of ABE applications.
    """
    if app_id == APP_VOICEMAIL:
        # for Voicemail we need to uninstall TTS languages pack and CSIPO
        uninstall_languages_pack()
    if app_id == APP_IPOFFICE:
        # for IP Office we need to uninstall ms package
        uninstall_package("ms")
    if app_id == APP_ONEXPORTAL:
        # for one-X Portal we need to uninstall scs-webconferencing package
        uninstall_package("scs-webconferencing")
        uninstall_package("webRTCGateway")
        uninstall_package("ipoffice-webclient")
        uninstall_package("IPO-Reporting-Analytics")
    package_name = _package_name_for_id(app_id)
    log.info(TAG, "un-installing application <%s>" % package_name)
    rpm.uninstall(package_name)
    for rpm_pattern in RPM_FILES[app_id]:
        pathname = os.path.join(LOCAL_APPS_REPOSITORY, rpm_pattern)
        rpms_to_delete = glob.glob(pathname)
        if rpms_to_delete:
            log.info(TAG, "deleting RPM packages:\n%s" %
                          '\n'.join(rpms_to_delete))
            shell.rm(rpms_to_delete)
        else:
            log.warn(TAG, "no files to delete matching <%s>" % pathname)


def uninstall_languages_pack():
    """
    Uninstall TTS languages pack.
    """
    log.info(TAG, "un-installing VMPro language pack")
    for app_id in ALL_APPS:
        if app_id.startswith('tts_'):
            package_name = ALL_APPS[app_id]['package_name'][version.RELEASE_TYPE]
            if rpm.query_package(package_name):
                log.info(TAG, "un-installing TTS language pack <%s>" % package_name)
                rpm.uninstall(package_name)
    # XXX actually we have two language packs: TTS files and WAV files
    vmpro_wavs_pattern  = os.path.join(LOCAL_APPS_REPOSITORY, 'vmpro-wavs*.rpm')
    vmpro_wavs_files    = glob.glob(vmpro_wavs_pattern)
    for vmpro_wav_file in vmpro_wavs_files:
        package_info = rpm.query_rpm(vmpro_wav_file)
        if package_info and 'NAME' in package_info:
            package_name = package_info['NAME']
            log.info(TAG, "un-installing WAV language pack <%s>" % package_name)
            rpm.uninstall(package_name)


def clean_cli():
    """
    Changes CLI xml file for Application Server release types
    """
    shell.sudo_call('perl -pi -e "s/ipoffice\(ipoffice\)/ManagementServices\(ipoffice\)/;" /etc/clish/types.xml')
    shell.sudo_call('perl -pi -e "s/mediaserver\(ms\)//;" /etc/clish/types.xml')
    shell.sudo_call('perl -pi -e "s/\*              Avaya IP Office             \*/\*            Application Server            \*/;" /etc/clish/startup.xml')


def uninstall_package(name=None, delete_rpm=False):
    """
    Uninstall a package and optionally delete the installation rpm.

    Args:
        name       -- the name of the package in question
        delete_rpm -- if argument is set to True delete the installation rpm
    """
    if name:
        if rpm.query_package(name):
            log.info(TAG, "uninstalling %s" % name)
            rpm.uninstall(name, "--nodeps")
        if delete_rpm:
            pathname = os.path.join(LOCAL_APPS_REPOSITORY, "%s*.rpm" % name)
            rpm_to_delete = glob.glob(pathname)
            log.info(TAG, "deleting %s RPM package" % name)
            shell.rm(rpm_to_delete)

def delete_ipoffice_demo():
    pathname = os.path.join(LOCAL_APPS_REPOSITORY, "ipoffice-demo*.rpm")
    rpm_to_delete = glob.glob(pathname)
    log.info(TAG, "deleting ipoffice-demo RPM package")
    shell.rm(rpm_to_delete)

def delete_demo_default_config():
    pathname = os.path.join(LOCAL_APPS_REPOSITORY, "demo-default-config*.rpm")
    rpm_to_delete = glob.glob(pathname)
    log.info(TAG, "deleting demo-default-config RPM package")
    shell.rm(rpm_to_delete)


def enable(app_id):
    """
    Configure Auto-Start and start service for specified application.

     Args:
        app_id  --  application id (or service id) as defined
                        in <service>.ini configuration file
    Throws:
        ValueError if app_id is not one of ABE applications.
    """
    service_name = _service_name_for_id(app_id)
    shell.enable_service(service_name)
    shell.sudo_call("systemctl start %s.service" % service_name)


def install_onex_database():
    """
    Install one-X prepackaged database.
    """
    log.info(TAG, "searching for one-X portal " \
             "prepackaged database files to install <%s>" % ONEX_DB_FILE_PATTERN)
    onex_db_rpms = glob.glob(ONEX_DB_FILE_PATTERN)
    log.info(TAG, "found <%d> one-X database files" % len(onex_db_rpms))
    for rpm_file in onex_db_rpms:
        log.info(TAG, "installing one-X database file <%s>" % rpm_file)
        try:
            rpm.install(rpm_file)
        except rpm.RpmError:
            log.exception(TAG, "unable to install <%s>" % rpm_file)


def uninstall_packages(mode=None):
    """
    Used by ignition and the post upgrade script to remove unwanted packages

    Args:
        mode -- server type ("Primary" , "Secondary" , "Expansion" or "Appl"
    """
    if mode == ipoffice.MODE_PRIMARY:
        pass
    elif mode == ipoffice.MODE_SECONDARY:
        uninstall_package("AuthModule", False)
        uninstall_package("IPO-Reporting-Analytics")
        uninstall_package("MediaManager")
        uninstall_package("CallAnalytics", True)
        delete_demo_default_config()
    elif mode == ipoffice.MODE_EXPANSION:
        uninstall(APP_VOICEMAIL)
        uninstall(APP_ONEXPORTAL)
        uninstall_package("AuthModule", False)
        uninstall_package("WebLM", False)
        uninstall_package("MediaManager")
        uninstall_package("CallAnalytics", True)
        delete_demo_default_config()
    elif mode == ipoffice.MODE_APPL:
        uninstall_package("ms", True)
        uninstall_package("ipphonebin", True)
        clean_cli()
        delete_demo_default_config()
        delete_ipoffice_demo()
        shell.rm(configuration.SHARED_CONFIGURATION['web_manager']['standard_edition_folder'])


def change_user_rights(mode=None):
    """
    Used by ignition in order to change user rights on certain files
    """
    if ipoffice.is_virtualized():
        shell.sudo_call("chmod 4755 %s" % DMIDECODE_FILE)


def install_mode_package(mode=None):
    """
    Used by ignition to install the type specific package

    Args:
        mode -- server type ("Primary" , "Secondary" , "Expansion" or "Appl"
    """
    if mode in ipoffice.MODES:

        rpm_name = ipoffice.MODE_SPECIFIC_PACKAGE[mode]
        build_version = version.APP_VERSION['VERSION']
        release_version = version.APP_VERSION['RELEASE'].split(".")[0]
        rpm_file = "%s-%s-%s" % (rpm_name, build_version, release_version)
        rpm_location = configuration.SHARED_CONFIGURATION['repositories']['local']['apps']
        files = glob.glob("%s%s*" % (rpm_location, rpm_file))

        if files and len(files) > 0:
            mode_rpm = files[0]
            log.info(TAG, "installing <%s>" % mode_rpm)
            try:
                rpm.install(mode_rpm)
            except rpm.RpmError:
                log.exception(TAG, "unable to install <%s>" % mode_rpm)
        else:
            log.info(TAG, "%s package not found" % rpm_name)

        for server_type in ipoffice.MODE_SPECIFIC_PACKAGE:
            if not server_type == mode:
                rpm_remove_file = "%s-%s-%s" % (ipoffice.MODE_SPECIFIC_PACKAGE[server_type], build_version, release_version)
                remove_files = glob.glob("%s%s*" % (rpm_location, rpm_remove_file))
                if remove_files and len(remove_files) > 0:
                    remove_rpm = remove_files[0]
                    log.info(TAG, "deleting <%s>" % remove_rpm)
                    try:
                        shell.sudo_call("rm %s" % remove_rpm)
                    except rpm.RpmError:
                        log.exception(TAG, "unable to delete <%s>" % remove_rpm)

    else:
        raise ValueError("invalid mode <%s>" % mode)


class Test(unittest.TestCase):

    def test_package_name_for_id(self):
        self.assertTrue(_package_name_for_id(APP_IPOFFICE))

    def test_package_name_for_id(self):
        self.assertRaises(ValueError, _package_name_for_id, "invalid")

    def test_uninstall_invalid(self):
        self.assertRaises(ValueError, uninstall, "invalid")


class TestOneXDatabaseFiles(unittest.TestCase):

    def setUp(self):
        global ONEX_DB_FILE_PATTERN
        self.orig_onex_db_file_pattern = ONEX_DB_FILE_PATTERN
        ONEX_DB_FILE_PATTERN = '/tmp/something-that-does-not-exists*.rpm'

    def tearDown(self):
        global ONEX_DB_FILE_PATTERN
        ONEX_DB_FILE_PATTERN = self.orig_onex_db_file_pattern

    def test_install_non_existing_onex_database_files(self):
        install_onex_database()


if __name__ == "__main__":
    unittest.main()
