# Copyright 2011 Avaya Inc. All Rights Reserved.

"""
Provides IP Office configuration management functions.
"""

import unittest
import os
import socket
import fcntl
import struct

from core.common import utils, configuration
from core.common import log
from core.system import shell

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2011, Avaya Inc."

# Logging tag
TAG = 'se-ipoffice-common'

# IP Office configuration
SERVICE_NAME        = "ipoffice"
CONFIGURATION_FILE  = "/etc/sysconfig/ipoffice"

# Server Edition mode to be written in IP Office configuration file
MODE_ENV_VAR        = "IPOFFICE_BE_MODE"
MODE_PRIMARY        = "Primary"
MODE_SECONDARY      = "Secondary"
MODE_EXPANSION      = "Expansion"
MODE_APPL           = "Appl"
MODE_DEFAULT        = ""
MODES               = (MODE_PRIMARY, MODE_SECONDARY, MODE_EXPANSION, MODE_APPL, MODE_DEFAULT)
MODE_VALUE          = { MODE_PRIMARY: '"Primary"',
                        MODE_SECONDARY: '"Secondary"',
                        MODE_EXPANSION: '"Expansion"',
                        MODE_APPL: '"Application server on Linux PC"',
                        MODE_DEFAULT: ''}

# Companding settings
COMPANDING_LAW_ENV_VAR  = "IPOFFICE_BE_DEFAULT_LAW"
COMPANDING_A_LAW        = "ALaw"
COMPANDING_MU_LAW       = "MULaw"
COMPANDING_DEFAULT      = ""
COMPANDING              = (COMPANDING_A_LAW, COMPANDING_MU_LAW, COMPANDING_DEFAULT)

# Licensing types
LICENCE_USB_DONGLE  = "USB_Dongle"
LICENCE_INTERNAL    = "Internal"
LICENSES            = (LICENCE_USB_DONGLE, LICENCE_INTERNAL)
# TODO licensing configuration will be implemented later, for now this is a dummy placeholder
_LICENCE            = LICENCE_INTERNAL

# Physical description (SNMP sysDescr.0)
PHYSICAL_DESCRIPTION_ENV_VAR = "IPOFFICE_PHYSICAL_DESCR"
PHYSICAL_DESCRIPTION         = { MODE_PRIMARY: '"IP Office Server Edition Primary"',
                                 MODE_SECONDARY: '"IP Office Server Edition Secondary"',
                                 MODE_EXPANSION: '"IP Office Server Edition Expansion System (L)"',
                                 MODE_APPL: '"IP Office Application server on Linux PC"',
                                 MODE_DEFAULT: ''}

MODE_SPECIFIC_PACKAGE        = { MODE_PRIMARY: 'Primary',
                                 MODE_SECONDARY: 'Secondary',
                                 MODE_EXPANSION: 'Expansion',
                                 MODE_APPL: 'ApplicationServer',
                                 MODE_DEFAULT: ''}

# Server mode constants as returned by IP Office web service
PRIMARY_SERVER   = 'PrimaryServer'
SECONDARY_SERVER = 'SecondaryServer'
EXPANSION_SYSTEM = 'Expansion'

# IP Office types
IPOFFICE_LINUX   = 'IPO-Linux-PC'
IPOFFICE_IP500V2 = ''

# Virtualization mode
VIRTUALIZATION_MARKER   = '/opt/Avaya/.ova'

# Local UWS details
UWS_REPOSITORY_PATH     = '/var/www/html/avaya/uws'

#PLDS Marker:
HOST_ID_MARKER = 'PLDSHostID'

def uws_releases():
    """
    Return a list with all releases under UWS repository path.
    """
    return [entry for entry in os.listdir(UWS_REPOSITORY_PATH)
            if os.path.isdir(os.path.join(UWS_REPOSITORY_PATH, entry))]

def cleanup_uws_repository():
    """
    Remove all folders under UWS repository path.
    """
    dirs_to_delete = [os.path.join(UWS_REPOSITORY_PATH, entry)
                      for entry in uws_releases()]
    log.info(TAG, "removing from UWS repository: %s" % dirs_to_delete)
    shell.rm(dirs_to_delete)

def set_mode(mode):
    """
    Sets IP Office mode.

    Args:
        mode        -- IP Office mode to set, one of MODE_(PRIMARY|SECONDARY|EXPANSION|APPL)

    Throws:
        ValueError  -- if mode is not one of the expected values.
        IOError     -- if the IP Office configuration file is missing.
    """
    if mode in MODES:
        utils.update_config(CONFIGURATION_FILE, {MODE_ENV_VAR: MODE_VALUE[mode]})
    else:
        raise ValueError("invalid mode <%s>" % mode)


def get_mode():
    """
    Gets IP Office mode.

    Returns one of MODE_(PRIMARY|SECONDARY|EXPANSION|APPL).

    If the IP Office configuration file is missing MODE_DEFAULT will be returned
    """
    if os.path.isfile(CONFIGURATION_FILE):
        params = {MODE_ENV_VAR: None}
        utils.read_config(CONFIGURATION_FILE, params)
        mode = params[MODE_ENV_VAR]
        return mode.replace('"','')
    else:
        log.warn(TAG, "file <%s> not found" % CONFIGURATION_FILE)
        return MODE_DEFAULT


def get_companding_law():
    """
    Gets IP Office companding law.

    Returns one of COMPANDING_(A_LAW|MU_LAW).

    Throws:
        IOError     -- if the IP Office configuration file is missing.
    """
    params = {COMPANDING_LAW_ENV_VAR: None}
    utils.read_config(CONFIGURATION_FILE, params)
    return params[COMPANDING_LAW_ENV_VAR]


def set_companding_law(companding_law):
    """
    Sets IP Office companding law.

    Args:
        companding_law      -- companding law mode to set,  one of COMPANDING_(A_LAW|MU_LAW)
    Throws:
        ValueError          -- if companding_law is not one of the expected values.
        IOError             -- if the IP Office configuration file is missing.
    """
    if companding_law in COMPANDING:
        utils.update_config(CONFIGURATION_FILE, {COMPANDING_LAW_ENV_VAR: companding_law})
    else:
        raise ValueError("invalid companding law <%s>" % companding_law)


def set_phys_description(mode):
    """
    Set physical description (SNMP sysDescr.0) according to specified IP Office mode.

    Args:
        mode                -- IP Office mode (one of MODE_(PRIMARY|SECONDARY|EXPANSION|APPL)

    Throws:
        ValueError          -- if mode is not one of the expected values.
        IOError             -- if the IP Office configuration file is missing.
    """
    if mode in MODES:
            utils.update_config(CONFIGURATION_FILE,
                    {PHYSICAL_DESCRIPTION_ENV_VAR: PHYSICAL_DESCRIPTION[mode]})
    else:
        raise ValueError("invalid mode <%s>" % mode)


def get_phys_description():
    """
    Returns physical description (SNMP sysDescr.0).

    Throws:
        IOError -- if the IP Office configuration file is missing.
    """
    params = {PHYSICAL_DESCRIPTION_ENV_VAR: None}
    utils.read_config(CONFIGURATION_FILE, params)
    return params[PHYSICAL_DESCRIPTION_ENV_VAR]


def get_licence_type():
    """
    Gets IP Office licensing type.

    Returns one of LICENCE_USB_DONGLE or LICENCE_INTERNAL
    """
    # TODO this will read IP Office licensing scheme
    return _LICENCE


def set_licence_type(type):
    """
    Sets IP Office licensing type.

    Args:
        type    --  one of LICENCE_USB_DONGLE | LICENCE_INTERNAL
    Throws:
        ValueError if type is not one of the expected values
    """
    # TODO this will set IP Office licensing scheme
    if type in LICENSES:
        global _LICENCE
        _LICENCE = type
    else:
        raise ValueError("invalid licence type <%s>" % type)

	
def get_mac(ifname):
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        info = fcntl.ioctl(s.fileno(), 0x8927, struct.pack('256s', bytes(ifname[:15],'utf-8')))
        return ''.join(['%02X' % char for char in info[18:24]])

def host_id():
    """
    Returns Host ID for current installation.
    """
    exit_code, out, err = shell.sudo_execute2("%s get_PLDSHostID" % SERVICE_NAME)
    host_id_value = ""
    output = (out or err).decode().split("\n")
    for line in output:
        if line.startswith(HOST_ID_MARKER):
            host_id_value = line.split("=")[1]
            break

    exit_code, out, err = shell.sudo_execute2("service %s get_fingerprint" % SERVICE_NAME)
    sid = (out or err).strip().decode()

    weblm_host_id = ""
    weblm_host_id = get_mac("eth0")

    sysinfo = {
        "plds_host_id": host_id_value,
        "system_id": sid,
        "weblm_host_id": weblm_host_id
    }
    return sysinfo


def system_id():
    """
    Returns System ID for current installation.
    """
    exit_code, out, err = shell.sudo_execute2("/etc/init.d/%s get_fingerprint" %
                                              SERVICE_NAME)
    return out.decode() or err


def is_virtualized():
    """
    Convenience method to test whether IP Office runs on virtualized hardware.
    """
    return os.path.isfile(VIRTUALIZATION_MARKER)

def rename_release_file(old_release, new_release):
    """
    Rename the release file
    Given values must be one of 'isa', 'apc', 'abg', 'appl', 'abe'
    """
    if os.path.isfile(configuration.SHARED_CONFIGURATION['release_config'][old_release]):
        os.rename(configuration.SHARED_CONFIGURATION['release_config'][old_release],
            configuration.SHARED_CONFIGURATION['release_config'][new_release])
    else :
        log.error(TAG, "file %s can not be renamed, it dose not exist"
            % configuration.SHARED_CONFIGURATION['release_config'][old_release] )


class TestGeneral(unittest.TestCase):

    def setUp(self):
        import tempfile
        _, self.test_config_file = tempfile.mkstemp()
        with shell.file_open(self.test_config_file, "w") as f:
            f.write("# Test data for ipoffice module\n")
            f.write("export %s=%s\n" % (MODE_ENV_VAR, MODE_DEFAULT))
            f.write("export %s=%s\n" % (COMPANDING_LAW_ENV_VAR, COMPANDING_DEFAULT))
            f.write("export %s=%s\n" % (PHYSICAL_DESCRIPTION_ENV_VAR, ""))
            f.flush()
        global CONFIGURATION_FILE
        self.orig_config_file = CONFIGURATION_FILE
        CONFIGURATION_FILE = self.test_config_file

    def tearDown(self):
        shell.rm(self.test_config_file)
        global CONFIGURATION_FILE
        CONFIGURATION_FILE = self.orig_config_file

    # IP Office mode

    def test_get_mode(self):
        self.assertTrue(get_mode() is not  None)

    def test_get_mode_default(self):
        global CONFIGURATION_FILE
        CONFIGURATION_FILE="/tmp/something-that-does-not-exists"
        self.assertEquals(MODE_DEFAULT, get_mode())

    def test_set_mode_invalid_arg(self):
        self.assertRaises(ValueError, set_mode, "invalid")

    def test_set_mode_config_file_missing(self):
        global CONFIGURATION_FILE
        CONFIGURATION_FILE="/tmp/something-that-does-not-exists"
        self.assertRaises(IOError, set_mode, MODE_EXPANSION)

    def test_set_mode_non_default(self):
        new_mode = MODE_EXPANSION
        set_mode(new_mode)
        self.assertEqual(get_mode(), new_mode)

    def test_set_mode_default(self):
        self.orig_mode = get_mode()
        set_mode(MODE_DEFAULT)
        self.assertEqual(self.orig_mode, get_mode())

    # Companding settings

    def test_get_companding_law(self):
        self.assertTrue(get_companding_law() is not None)

    def test_get_companding_law_config_file_missing(self):
        global CONFIGURATION_FILE
        CONFIGURATION_FILE = "/tmp/something-that-does-not-exists"
        self.assertRaises(IOError, get_companding_law)

    def test_set_companding_law(self):
        new_companding_law = COMPANDING_A_LAW
        set_companding_law(new_companding_law)
        self.assertEqual(get_companding_law(), new_companding_law)

    def test_set_companding_law_invalid_arg(self):
        self.assertRaises(ValueError, set_companding_law, "invalid")

    def test_set_companding_law_config_file_missing(self):
        global CONFIGURATION_FILE
        CONFIGURATION_FILE = "/tmp/something-that-does-not-exists"
        self.assertRaises(IOError, get_companding_law)

    # Physical description

    def test_get_phys_description(self):
        self.assertTrue(get_phys_description() is not None)

    def test_get_phys_description_file_missing(self):
        global CONFIGURATION_FILE
        CONFIGURATION_FILE = "/tmp/something-that-does-not-exists"
        self.assertRaises(IOError, get_phys_description)

    def test_set_phys_description_invalid_arg(self):
        self.assertRaises(ValueError, set_phys_description, "invalid")

    def test_set_phys_description(self):
        mode = MODE_PRIMARY
        set_phys_description(mode)
        self.assertEqual(PHYSICAL_DESCRIPTION[mode], get_phys_description())

    def test_set_phys_description_config_file_missing(self):
        global CONFIGURATION_FILE
        CONFIGURATION_FILE = "/tmp/something-that-does-not-exists"
        self.assertRaises(IOError, get_phys_description)

    # Licence type

    def test_get_licence_type(self):
        self.assertTrue(get_licence_type() is not None)

    def test_set_licence_type(self):
        new_licence_type = LICENCE_USB_DONGLE
        set_licence_type(new_licence_type)
        self.assertEqual(get_licence_type(), new_licence_type)

    def test_set_licence_type_invalid_type(self):
        self.assertRaises(ValueError, set_licence_type, "invalid")

    # System ID

    def test_get_system_id(self):
        self.assertTrue(system_id())


class TestVirtualized(unittest.TestCase):

    def setUp(self):
        import tempfile
        global VIRTUALIZATION_MARKER
        self.orig_virt_marker = VIRTUALIZATION_MARKER
        _, VIRTUALIZATION_MARKER = tempfile.mkstemp()

    def tearDown(self):
        global VIRTUALIZATION_MARKER
        shell.rm(VIRTUALIZATION_MARKER)
        VIRTUALIZATION_MARKER = self.orig_virt_marker

    def test_marker_file_exist(self):
        self.assertTrue(is_virtualized())

    def test_marker_file_is_missing(self):
        shell.rm(VIRTUALIZATION_MARKER)
        self.assertFalse(is_virtualized())

class TestUWSReleases(unittest.TestCase):

    def setUp(self):
        self.tmp_dir = '/tmp/uws_test'
        for release_dir in ('8.1.40-23',  '8.1.40-24'):
            shell.sudo_call('mkdir -p %s/%s' % (self.tmp_dir, release_dir))
        global UWS_REPOSITORY_PATH
        self.orig_uws_repo_path = UWS_REPOSITORY_PATH
        UWS_REPOSITORY_PATH = self.tmp_dir

        with open("%s/8.1.40-24/ABE_release" % self.tmp_dir, 'w') as f:
            f.write("""
ABE_VERSION=8.1.40\(24_el6\);    export ABE_VERSION
IPOFFICE_CORE_VERSION=8.1.9285; export IPOFFICE_CORE_VERSION
IPO_REVISION=14335;               export IPO_REVISION
JADE_VERSION=8.1.90.2_el6;     export JADE_VERSION
VMPRO_VERSION=8.1.0710.0;                 export VMPRO_VERSION
ONEX_VERSION=8.1.72.18;                   export ONEX_VERSION
AdminCD_VERSION=8.1\(28\); export AdminCD_VERSION
BUILD_DATE=2012-05-14_11:52;                        export BUILD_DATE
            """)

    def tearDown(self):
        if os.path.isdir(self.tmp_dir):
            shell.rm(self.tmp_dir)
        global UWS_REPOSITORY_PATH
        UWS_REPOSITORY_PATH = self.orig_uws_repo_path

    def test_list_uws_releases(self):
        releases = uws_releases()
        self.assertEqual(2, len(releases))

    def test_cleanup_uws_repository(self):
        cleanup_uws_repository()
        self.assertEqual([], uws_releases())

if __name__ == "__main__":
    unittest.main()
