# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Provides authentication mechanism for serveredition
"""

import audit
import os
import web

from xml.etree.ElementTree import XML, ElementTree, Element, SubElement

from core.auth import pam
from core.common import log, configuration, i18n
from core.ext import ipoffice_ws
from core.logs import login

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

TAG = 'se-security'

REFERRAL_AUTH_MARKER = '/opt/Avaya/.referral_auth'

class ServerEditionAuthenticator(object):
    USERNAME_KEY = 'username'
    PASSWORD_KEY = 'password'
    REFERRER_KEY = 'referrer'
    REQUEST_SID_KEY = 'reqsid'
    WCP_ADMIN_RIGHTS = 256
    WCP_SECURITY_RIGHTS = 512

    def __init__(self):
        self.referred = os.path.exists(REFERRAL_AUTH_MARKER)
        self.ws_port = 8443
        self.ws_protocol = 'https'

    def is_referred(self):
        return self.referred

    def set_referred(self, referred_enabled):
        self.referred = referred_enabled

    def _pam_authenticate(self, user, passwd):
        return True if user in configuration.SHARED_CONFIGURATION['webapp'].as_list('users') \
            and pam.authenticate(user, passwd) else False

    def _referred_authenticate(self, user, passwd):
        client = ipoffice_ws.WSClient(host='127.0.0.1', port=self.ws_port, protocol=self.ws_protocol, username=user, password=passwd)
        try:
            xml_str = client.get_authentication()
            doc = XML(xml_str)
            nodes = doc.findall(".//service")
            if nodes:
                for node in nodes:
                    if node.attrib['name'] == 'External':
                        rights_node = node.findall('./AccessRights')
                        rights = int(rights_node[0].text)
                        if (rights & self.WCP_ADMIN_RIGHTS) == self.WCP_ADMIN_RIGHTS:
                            web.ctx.session.admin_rights = True
                        if (rights & self.WCP_SECURITY_RIGHTS) == self.WCP_SECURITY_RIGHTS:
                            web.ctx.session.security_rights = True
                        return web.ctx.session.admin_rights | web.ctx.session.security_rights
            return False
        except ipoffice_ws.WSError as e:
            if e.err_code:
                return False
            else:
                #IP Office service not responding => authenticate using PAM
                auth_ok = self._pam_authenticate(user, passwd)
                if auth_ok:
                    if user in configuration.SHARED_CONFIGURATION['webapp'].as_list('users'):
                        web.ctx.session.admin_rights = True
                        if user == "root":
                            web.ctx.session.security_rights = True
                return auth_ok

    def authenticate(self, client_data):
        try:
            user = client_data[self.USERNAME_KEY]
            if configuration.add_session(user) == False:
                web.ctx.session.user = user
                audit.log(i18n.custom_gettext('Maximum number of sessions for user was reached' ))
                login.mark_fail()
                web.ctx.status = '503'
                web.ctx.session.kill()
                return
            else:
                passwd = client_data[self.PASSWORD_KEY]
                if self.is_referred():
                    auth_ok = self._referred_authenticate(user, passwd)
                else:
                    auth_ok = self._pam_authenticate(user, passwd)
        except KeyError:
            auth_ok = False

        if auth_ok:
            web.ctx.session.user = user
            web.ctx.session.password = passwd
            if self.REFERRER_KEY in client_data:
                web.ctx.session.referrer = client_data[self.REFERRER_KEY]

            audit.log(i18n.custom_gettext('logged in'))
            if not self.is_referred():
                web.ctx.session.admin_rights = True
                if client_data['username'] == "root":
                    web.ctx.session.security_rights = True
            login.mark_success()
            web.ctx.status = '200'
            if self.REQUEST_SID_KEY in client_data:
                return web.ctx.session.get('session_id')
        else:
            login.mark_fail()
            web.ctx.status = '401'
            web.ctx.session.kill()

    def get_referrer(self):
        return web.ctx.session.get(self.REFERRER_KEY)
