# Copyright 2011 Avaya Inc. All Rights Reserved.

"""
Sub-application used to implement **Ignition** procedure.
"""
import json

import web
import os
import datetime

from core.common import log
from core.common import vendor_i18n
from core.common import version
from core.common import configuration
from core.system import shell

from core.ext import security

# Logging tag
TAG = 'ignition-webapp'

IGNITION_MARKER = '/opt/Avaya/.ignition'
NOICU_MARKER = '/opt/Avaya/.noICU'


__author__      = "Avaya Inc."
__copyright__   = "Copyright 2011, Avaya Inc."

urls = ('/run_once',    'IgnitionRunOnceResource',
        '/auth_mode',   'AuthenticationModeResource',
        '/',            'IgnitionResource')

APP = web.application(urls, locals())
URL = '/setup'

UNSECURED = {'GET': ('/run_once'),
             'PUT': (),
             'POST': (),
             'DELETE': ()}

# Authentication support

def auth_processor(handle):
    """
    If the user is not authenticated the
    client will be redirected to login page.
    """
    if web.ctx.path in UNSECURED[web.ctx.method.upper()]:
        return handle()
    else:
        if not web.ctx.session.get('user'):
            raise web.seeother('/login', absolute=True)
        else:
            return handle()

APP.add_processor(auth_processor)

# Custom app processor for Web Control main UI

def client_webapp_processor(handle):
    if web.ctx.session.get('user') and not os.path.exists(IGNITION_MARKER):
        log.info(TAG, 'starting Ignition')
        raise web.seeother('/setup/')
    return handle()

# Template render with i18n support

render = web.template.render('templates/', globals={'_': vendor_i18n.custom_gettext, 'build': version.BUILD_UID, 'csrf_token':configuration.csrf_token})


class IgnitionResource(object):
    """
    REST resource used to initiate/complete ignition procedure.

    Methods:

    GET     -- start ignition process
    PUT     -- save user input and mark ignition procedure as completed
    """
    def GET(self):
        if web.ctx.session.get('user') and os.path.exists(IGNITION_MARKER):
            raise web.seeother('/main', absolute=True)
        return render.ignition(configuration.get_authenticator().get_referrer())
    
    def PUT(self):
        log.info(TAG, "creating Ignition marker file <%s>" % IGNITION_MARKER)
        shell.sudo_call("touch %s" % IGNITION_MARKER)
        shell.sudo_call("chown Administrator %s" % IGNITION_MARKER)
        shell.sudo_call("touch %s" % NOICU_MARKER)
        shell.sudo_call("chown Administrator %s" % NOICU_MARKER)
        with shell.file_open(IGNITION_MARKER, 'w') as outfile:
            outfile.write("Initial setup started at %s" % datetime.datetime.now())


class IgnitionRunOnceResource(object):
    """
    REST resource used to check if the ignition was executed or not.

    Methods:

    GET     -- check if ignition was executed
    """
    def GET(self):
        """
        Returns a JSON string with following format:

            {'run_once': true|false}
        """
        web.header("Content-Type", "application/json")
        return json.dumps({'run_once': os.path.exists(IGNITION_MARKER)})

class AuthenticationModeResource(object):
    """
    REST resource used to set authentication mode (referred) at the end of Ignition.

    Methods:

    GET     -- indicate is referred authentication mode is set
    PUT     -- create marker file to indicate that referred authentication should be used
    """
    def GET(self):
        web.header("Content-Type", "application/json")
        return json.dumps({'referred': os.path.exists(security.REFERRAL_AUTH_MARKER)})

    def PUT(self):
        referred = web.input(referred='true').referred.lower() == 'true'
        if referred:
            log.info(TAG, "enabling referred authentication")
            shell.sudo_call("touch %s" % security.REFERRAL_AUTH_MARKER)
            shell.sudo_call("chown Administrator %s" % security.REFERRAL_AUTH_MARKER)
        else:
            log.info(TAG, "disabling referred authentication")
            shell.sudo_call("rm %s" % security.REFERRAL_AUTH_MARKER)
        auth = configuration.get_authenticator()
        if hasattr(auth, 'set_referred'):
            auth.set_referred(referred)
