# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Sub application for web UI.

This application is a client to REST API sub application.
"""
import glob
import json
import mimetypes
import sys
import datetime
import wsgiref
import web
import os

from core.common import configuration
from core.common import log
from core.common import i18n
from core.common import version
from core.updates import windows_clients

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

TAG = 'client-webapp'

urls = ('/',                     'Main',
        '/user',                 'UserApplications',
        '/login',                'Login',
        '/help',                 'Help',
        '/help/login',           'HelpLogin',
        '/help/main',            'HelpMain',
        '/help/home',            'HelpHome',
        '/help/logs',            'HelpLogs',
        '/help/updates',         'HelpUpdates',
        '/help/settings',        'HelpSettings',
        '/help/windows_clients', 'HelpWindowsClients',
        '/help/vnc',             'HelpVnc',
        '/help/troubleshooting', 'HelpTroubleshooting',
        '/main',                 'Main',
        '/home',                 'Home',
        '/logs',                 'Logs',
        '/updates',              'Updates',
        '/settings',             'Settings',
        '/messages',             'Messages',
        '/windows_clients',      'WindowsClients',
        '/linux_downloads',      'LinuxDownloads',
        '/vnc',                  'Vnc')

APP = web.application(urls, locals())
URL = ''
LOAD_ORDER = sys.maxsize     # must be loaded last because is mapped to /


render = web.template.render('templates/', globals={'_': i18n.custom_gettext, 'build': version.BUILD_UID, 'csrf_token':configuration.csrf_token})

# authentication support

UNSECURED = { 'GET': ('/login', '/user'),
              'PUT': (),
              'POST': (),
              'DELETE': ()}

def auth_processor(handle):
    """
    If the user is not authenticated and the context requires
    authentication then the client will be redirected to login page or return 401 HTTP code
    if the request is made through an ajax call.
    """
    if web.ctx.path in UNSECURED[web.ctx.method.upper()]:
        return handle()
    else:
        if not web.ctx.session.get('user'):
            if web.ctx.env.get('HTTP_X_REQUESTED_WITH') == 'XMLHttpRequest':
                web.ctx.status = '401'
            else:
                raise web.seeother('/login')
        else:
            return handle()

APP.add_processor(auth_processor)

# load custom 3rd party web application processors

apps = []

for filename in glob.glob('*_webapp.py'):
    try:
        mod_name = os.path.basename(filename).replace('.py', '')
        mod =  __import__(mod_name, fromlist=[''])
        if hasattr(mod, 'client_webapp_processor'):
            apps.append(mod)
    except:
        log.exception(TAG, 'unable to load web application from %s' % filename)

# sort applications according to their LOAD_ORDER
apps.sort(key=lambda m: m.LOAD_ORDER if hasattr(m, "LOAD_ORDER") else 0)

for app in apps:
    log.info(TAG, 'registering application processor from %s' % app.__name__)
    APP.add_processor(app.client_webapp_processor)


class Login:
    """Login handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        web.header('X-FRAME-OPTIONS', 'DENY')
        ref = web.input(referrer=None).referrer
        return render.login(ref)


class Main:
    """Main page handler"""
    def GET(self):
       
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')
        return render.main(configuration.get_authenticator().get_referrer())
        

class Home:
    """Home tab handler"""
    def GET(self):
       
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')
        return render.home()
        

class Logs:
    """Logs tab handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.logs()
        

class Updates:
    """Updates tab handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.updates()
        

class Settings:
    """Settings tab handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
            #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.settings()
        

class WindowsClients:
    """Windows Clients tab handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
            #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.windows_clients()


class LinuxDownloads:
    """Linux Downloads Clients tab handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.linux_downloads()


class Vnc:
    """Remote Desktop tab handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.vnc()


class Messages:
    """Messages handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.messages()


class Help:
    """Help handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.help()


class HelpLogin:
    """Login help file handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.help_login()


class HelpMain:
    """Main page help file handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.help_main()


class HelpHome:
    """Home tab help file handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.help_home()


class HelpLogs:
    """Logs tab help file handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.help_logs()


class HelpUpdates:
    """Updates tab help file handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.help_updates()


class HelpSettings:
    """Settings tab help file handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.help_settings()


class HelpWindowsClients:
    """Windows Clients tab help file handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.help_windows_clients()


class HelpVnc:
    """VNC tab help file handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.help_vnc()


class HelpTroubleshooting:
    """Troubleshooting help file handler"""
    def GET(self):
        lang = web.cookies().get('avaya_webcontrol_lang')
        if lang:
            web.ctx.session.lang = lang
        #web.header('X-FRAME-OPTIONS', 'DENY')

        return render.help_troubleshooting()



class UserApplications(object):
    """
    REST resource used to list the available client application files.

    Methods:

    GET --  check if current hardware profile is suitable
            according to specified IP Office mode
    """

    FILE_EXTENSION_FILTER = ('.exe', '.msi', '.zip', '.dmg')

    # Block size for windows client file upload
    BLOCK_SIZE = 16384

    windows_clients_manager = windows_clients.WindowsClientsManager()

    def GET(self):
        """
        Returns a list with the available client application files.

        Input:

        id -- the basename of Windows client file to be downloaded

        Error codes:

        404 -- unknown Windows client file
        """
        params = web.input()

        if "files" in params:
            files = self.windows_clients_manager.list_local_user_repository(
                                                    self.FILE_EXTENSION_FILTER)
            web.header("Content-Type","application/json")
            return json.dumps(files)
        elif "file" in params:
            # serve file
            files = self.windows_clients_manager.list_local_user_repository(
                                                    self.FILE_EXTENSION_FILTER)
            file_name = params['file']
            windows_client_file = ''.join(
                                        [self.windows_clients_manager.local_user_repo,
                                        file_name.replace('/','')])
                       
            log.info(TAG,"PATH %s %s %s" % (windows_client_file, self.windows_clients_manager.local_user_repo,files))
            found = False
            for f in files:
                log.info(TAG," File : %s" % f['id'])
                if f['id'] == file_name:
                    found = True
                    break
            if not found:
                log.exception(TAG,'Not known path %s' % file_name)
                web.ctx.status = '404'
            else:            
                try:
                    mime_type = mimetypes.guess_type(file_name)[0]
                    web.header('Content-Type',  mime_type)
                    web.header('Content-Disposition', 'attachment; filename=' + file_name)
                    stat = os.stat(windows_client_file)
                    web.header('Content-Length', stat.st_size)
                    web.header('Last-Modified',
                    web.http.lastmodified(datetime.datetime.fromtimestamp(stat.st_mtime)))
                    if web.ctx.protocol.lower() == 'https':
                        # add headers to fix issue with IE download over SSL failing when "no-cache" header set
                        web.header('Pragma', 'private')
                        web.header('Cache-Control', 'private,must-revalidate')
                    return wsgiref.util.FileWrapper(open(windows_client_file, 'rb'), self.BLOCK_SIZE)
                except (OSError, IOError):
                    log.exception(TAG, 'unable to serve Windows client file %s' % windows_client_file)
                    web.ctx.status = '404'
        else:
            files = self.windows_clients_manager.list_local_user_repository(
                                                self.FILE_EXTENSION_FILTER)
            if not files:
                raise web.seeother('/login')
            return render.user_applications()
