# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Provides password management for user accounts.
Built as a thin wrapper over usermod utility. 
"""
import threading
import os
import unittest
import crypt
from . import pam
import string
import random

from core.system import shell
from core.common import pexpect
from core.common import i18n
from core.common import log
from core.common.decorators import synchronized

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

_CHANGE_PASSWORD_LOCK = threading.Lock()

#file containing the Administrator password
PASSWORD_FILE = '/opt/Avaya/.password'

class AuthError(Exception):
    """
    Raised for authentication errors.
    """
    def __str__(self):
        return repr(self.args[0])


def change_pass(username, old_pass, new_pass):
    """
    Change user password using usermod utility.

    Return the exit code for 'usermod -p crypted_pass username' command.

    Raises Error if the old password is invalid.
    """    
    if not pam.authenticate(username, old_pass):
        raise AuthError(i18n.custom_gettext("invalid login"))
    else:                       
        crypted_pass = crypt_pass(new_pass)
        return shell.sudo_call('usermod -p %s %s' % (crypted_pass, username))


def change_root_pass(old_pass, new_pass, user):
    """
    Change password using passwd command.
    """
    command = "echo -e \'%s\n%s\' | sudo passwd %s" % (new_pass, new_pass, user)
    return shell.bash_execute(command, parse=shell.RAW)


def check_root_pass(old_pass, new_pass, user):
    """
    Use pexpect module to determine if the new password is correct.
    """
    child=pexpect.spawn("sudo passwd %s" % user)
    child.expect("Enter new password: ")
    child.sendline(new_pass)
    output = child.expect(['Weak password:', 'Re-type new password:'])
    error_text = ""
    if output == 0:
        child.expect("Try again.")
        error_text += i18n.custom_gettext("Weak password: ") + str(child.before).strip(' \t\n\r')
        child.send(chr(3))
        child.sendline("")
    else:
        child.send(chr(3))
        child.sendline("")

    return error_text


def crypt_pass(password):
    """
    Encrypt given password using standard crypt utility and a random 4 letter salt.
    """
    salt = "".join(random.sample(string.letters, 4))
    return crypt.crypt(password, "$1$" + salt)


@synchronized(_CHANGE_PASSWORD_LOCK)
def write_password(password):
    """
    Writes WCP user password in a file.
    """
    shell.sudo_call("touch %s" % PASSWORD_FILE)
    with shell.file_open(PASSWORD_FILE, 'w') as outfile:
        outfile.write(password)


@synchronized(_CHANGE_PASSWORD_LOCK)
def read_password():
    """
    Reads WCP user password.
    """
    password = ""
    if os.path.isfile(PASSWORD_FILE):
        try:
            with shell.file_open(PASSWORD_FILE) as infile:
                password = infile.readline().strip()
        except (IOError, OSError):
            log.SHARED_LOGGER.exception("unable to read the password file <%s>" %
                                        PASSWORD_FILE)
    else:
        log.SHARED_LOGGER.error("password file <%s> does not exists" %
                                PASSWORD_FILE)
    return password


class Test(unittest.TestCase):

    def test_crypt_pass(self):
        self.assertTrue(crypt_pass("password"))

    def test_change_pass(self):
        username = "testwci"
        password = "testwci"
        pass_changed = False
        try:
            crypted_pass = crypt_pass(password)
            # create a temporary user account
            shell.sudo_call("useradd -p %s %s"  % (crypted_pass, username))
            # change the password
            new_password = password + "abc"
            change_pass(username, password, new_password)
            # verify if the password was changed
            pass_changed = pam.authenticate(username, new_password)
        finally:
            # remove temporary user account
            shell.sudo_call("userdel %s" % username)
        if not pass_changed:
            self.fail("password not changed")

if __name__ == "__main__":
    unittest.main()
