# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Global logging utility.

Debug messages are logged only if the WEBCONTROL_DEBUG environment variable is set.
"""

import os
import logging
import time

from logging.handlers import RotatingFileHandler
from logging.handlers import SysLogHandler

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

def parse_offset(offset):
    if offset < 0:
        return "+%s" % time.strftime("%H:%M", time.gmtime(abs(offset)))
    return "-%s" % time.strftime("%H:%M", time.gmtime(offset))

utc_offset = "-NoUTC"
if time.daylight is 0:
    utc_offset = parse_offset(time.timezone)
else:
    utc_offset = parse_offset(time.altzone)

global SHARED_LOGGER

# logging configuration
SHARED_LOGGER_NAME                = 'webcontrol'
SHARED_LOGGER_FILE_NAME           = "%s/api.log" % os.getenv('WEBCONTROL_LOG_DIR', '.')
SHARED_LOGGER_FILE_SIZE           = 100 * 1024 * 1024     # maximum size in bytes for log file (100M)
SHARED_LOGGER_DATE_FORMAT         = "%Y-%m-%d %H:%M:%S"
SHARED_LOGGER_ENTRY_FORMAT        = "%(asctime)s - %(name)s - %(levelname)s - %(message)s"
SYSLOG_LOGGER_PRODUCT_TYPE        = '.ipowebcontrol'
SYSLOG_LOGGER_FACILITY            = 'user'
SYSLOG_LOGGER_ADDRESS             = '/dev/log'
SYSLOG_DATE_FORMAT                = "%Y"
SYSLOG_LOGGER_ENTRY_FORMAT        = "ipowebcontrol[%(process)d]: " + utc_offset + " %(asctime)s 000 1 %(name)s | 1 %(name)s AUD %(message)s"

# WEBCONTROL_DEBUG env variable is used to enable debug mode.
# If this variable is set to 1 (or any value Python interprets as True)
# then all log.debug(...) method calls will log a debug message in log files.
# By default log.debug(...) does nothing since WEBCONTROL_DEBUG is not set.
DEBUG_ENABLED = os.getenv('WEBCONTROL_DEBUG')
_log_level = logging.DEBUG if DEBUG_ENABLED else logging.INFO

# configure global shared logger
# this logger is intended for general use across WCP components
SHARED_LOGGER = logging.getLogger(SHARED_LOGGER_NAME)
SHARED_LOGGER.setLevel(_log_level)

_file_handler = RotatingFileHandler(SHARED_LOGGER_FILE_NAME, maxBytes=SHARED_LOGGER_FILE_SIZE)
_file_handler.setFormatter(logging.Formatter(fmt=SHARED_LOGGER_ENTRY_FORMAT, datefmt=SHARED_LOGGER_DATE_FORMAT))
_file_handler.setLevel(_log_level)

SHARED_LOGGER.addHandler(_file_handler)

# configure syslog logging
# this logger is intended to be used for audit logging
SYSLOG_LOGGER = logging.getLogger(SYSLOG_LOGGER_PRODUCT_TYPE)
SYSLOG_LOGGER.setLevel(logging.INFO)

class BOMLessFormatter(logging.Formatter):
    """
    Workaround for removing BOM insertion from log messages.
    See http://bugs.python.org/issue14452
    """
    def format(self, record):
        return logging.Formatter.format(self, record) #.encode('utf-8')

try:
    _syslog_handler = SysLogHandler(facility=SYSLOG_LOGGER_FACILITY, address=SYSLOG_LOGGER_ADDRESS)
    _syslog_handler.setFormatter(BOMLessFormatter(fmt=SYSLOG_LOGGER_ENTRY_FORMAT, datefmt=SYSLOG_DATE_FORMAT))
    _syslog_handler.setLevel(logging.INFO)

    SYSLOG_LOGGER.addHandler(_syslog_handler)
except:
    SHARED_LOGGER.error("Could not write into syslog. Please make sure rsyslog service is running, then restart webcontrol.")


# convenience methods for logging, it is recommended NOT to use SHARED_LOGGER
# and SYSLOG_LOGGER directly, use instead one of the methods below for logging

def debug(tag, message):
    SHARED_LOGGER.debug("%s: %s" % (tag, message))

def info(tag, message):
    SHARED_LOGGER.info("%s: %s" % (tag, message))

def warn(tag, message):
    SHARED_LOGGER.warn("%s: %s" % (tag, message))

def error(tag, message):
    SHARED_LOGGER.error("%s: %s" % (tag, message))

def exception(tag, message):
    SHARED_LOGGER.exception("%s: %s" % (tag, message))

def syslog(message):
    SYSLOG_LOGGER.info(message.encode())

