# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Provides access to Avaya Inside applications log.
"""
import unittest

from core.system import shell
from core.common import configuration

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

class LogEntry(object):
    """
    Application log entry
    """
    def __init__(self, timestamp=None, application=None, level=None, message=None):
        self.timestamp = timestamp
        self.application = application
        self.level = level
        self.message = message

class AppLog(object):
    """
    Read applications log
    """

    # a set of ignored patterns in the output of show_logs_script
    IGNORED_PATTERNS = ('#',
                        'Output of the last command',
                        'Press q to quit')

    # a set of patterns used to identify application for which the logging is shown 
    SERVICE_NAME_PATTERNS = ('Watchdog alarms for',
                            'Last output for',
                            'Logs for application')

    def __init__(self, services_manager, config=configuration.SHARED_CONFIGURATION):
        """
        services_manager -- installed applications
        config           -- app configuration data
        """
        self.services_manager = services_manager
        self.logging_script = config['logging']['show_logs_script']
        self.tail_count = config['logging'].as_int('tail_count')

    def log_history(self, id):
        """
        Return log entries for specified application
        
        id -- application ID or 'all' for all applications.
        """
        entries = []
        if self.tail_count > 0:
            cmd = '%s %s %s' % (self.logging_script, id, self.tail_count)
        else:
            cmd = '%s %s' % (self.logging_script, id)
        output = shell.sudo_execute(cmd, parse=shell.LINES)
        if output:
            name = '-'
            for line in output:
                line = line.strip()
                if line and not self._should_skip(line):
                    service_name = self._parse_service_name(line)
                    if service_name:
                        for service in self.services_manager:
                            if service_name in service.service_name:
                                name = service.display_name
                                break
                    else:
                        entries.append(LogEntry(application=name, message=line))
        return entries

    def _should_skip(self, line):
        """
        Return True if the line starts with one of the IGNORED_PATTERNS.
        """
        for pat in self.IGNORED_PATTERNS:
            if line.startswith(pat):
                return True

    def _parse_service_name(self, line):
        """
        Return binary name if the line starts with one of the SERVICE_NAME_PATTERNS.
        """
        for pat in self.SERVICE_NAME_PATTERNS:
            if line.startswith(pat):
                return line.split()[-1].strip()

class Test(unittest.TestCase):

    def test_log_history(self):
        import core.services.manager
        app_log = AppLog(core.services.manager.ServicesManager())
        self.assertTrue(app_log.log_history('all'))

if __name__ == '__main__':
    unittest.main()
