# Copyright 2011 Avaya Inc. All Rights Reserved.

"""
Login logger

The login log file is stored under path defined by WEBCONTROL_LOG_DIR environment variable.
If WEBCONTROL_LOG_DIR is not set then current working directory is used.
"""

import os
import hashlib
import threading
import unittest
import configobj

from core.common import configuration
from core.common.decorators import synchronized
from core.common import log
from core.common import utils
from core.system import shell
from core.system import systime

__author__ = "Avaya Inc."
__copyright__ = "Copyright 2011, Avaya Inc."

_LOGIN_LOCK = threading.Lock()

# marker file used to signal the default password change
PASSWORD_CHANGED_MARKER = '/opt/Avaya/.password-changed'
# file containing encrypted default password
DEFAULT_PASSWORD_FILE = '/opt/Avaya/.default-password'

DEFAULT_PASSWORD_MD5 = ''

LOGIN_LOG = configuration.WEBCONTROL_LOG_DIR + "/" \
                + configuration.SHARED_CONFIGURATION['webapp']['login_file']

# Initialize login audit log
if os.path.isfile(LOGIN_LOG):
    LOGIN_DATA = configobj.ConfigObj(LOGIN_LOG)
    if not 'last_successful_login' in LOGIN_DATA:
        LOGIN_DATA['last_successful_login'] = ''
    if not 'current_successful_login' in LOGIN_DATA:
        LOGIN_DATA['current_successful_login'] = ''
    if not 'failed_login_attempts' in LOGIN_DATA:
        LOGIN_DATA['failed_login_attempts'] = 0
    if not 'default_password_changed' in LOGIN_DATA:
        LOGIN_DATA['default_password_changed'] = os.path.isfile(PASSWORD_CHANGED_MARKER)
else:
    LOGIN_DATA = configobj.ConfigObj(LOGIN_LOG, create_empty=True)
    LOGIN_DATA['last_successful_login'] = ''
    LOGIN_DATA['current_successful_login'] = ''
    LOGIN_DATA['failed_login_attempts'] = 0
    LOGIN_DATA['default_password_changed'] = os.path.isfile(PASSWORD_CHANGED_MARKER)
    LOGIN_DATA.write()

# Read default password details
if os.path.isfile(DEFAULT_PASSWORD_FILE):
    try:
        with shell.file_open(DEFAULT_PASSWORD_FILE) as infile:
            DEFAULT_PASSWORD_MD5 = infile.readline().strip()
    except (IOError, OSError) as e:
        log.SHARED_LOGGER.exception("unable to read default password file")
else:
    log.SHARED_LOGGER.error("default password file <%s> does not exists" %
                            DEFAULT_PASSWORD_FILE)


@synchronized(_LOGIN_LOCK)
def info():
    return LOGIN_DATA.dict()


@synchronized(_LOGIN_LOCK)
def mark_success():
    current_successful_login =  LOGIN_DATA.get('current_successful_login', '')
    if not current_successful_login:
        LOGIN_DATA['last_successful_login'] = utils.format_timestamp(systime.date())
    else:
        LOGIN_DATA['last_successful_login'] = current_successful_login
    LOGIN_DATA['current_successful_login'] = utils.format_timestamp(systime.date())
    LOGIN_DATA.write()


@synchronized(_LOGIN_LOCK)
def mark_fail():
    LOGIN_DATA['failed_login_attempts'] = int(LOGIN_DATA['failed_login_attempts']) + 1
    LOGIN_DATA.write()


def _encrypt(data):
    h1=hashlib.md5()
    h1.update(data)
    md5_step1=h1.hexdigest()
    h2=hashlib.md5()
    h2.update(md5_step1)
    return h2.hexdigest()


@synchronized(_LOGIN_LOCK)
def mark_default_password_changed(new_password):
    new_password_hash = _encrypt(new_password)
    if DEFAULT_PASSWORD_MD5 == new_password_hash:
        shell.rm(PASSWORD_CHANGED_MARKER)
    else:
        shell.sudo_call("touch %s" % PASSWORD_CHANGED_MARKER)
    LOGIN_DATA['default_password_changed'] = os.path.isfile(PASSWORD_CHANGED_MARKER)
    LOGIN_DATA.write()


class Test(unittest.TestCase):

    def test_info(self):
        self.assertTrue(info is not None)

    def test_mark_success_change_timestamp(self):
        mark_success()
        login_data = info()
        self.assertEquals(login_data['failed_login_attempts'], 0)

    def test_mark_success_reset_failed_login_counter(self):
        mark_success()
        login_data = info()
        self.assertNotEquals(login_data['last_successful_login'], '')

    def test_mark_fail(self):
        login_data = info()
        count_before_mark = int(login_data['failed_login_attempts'])
        mark_fail()
        login_data = info()
        count_after_mark = int(login_data['failed_login_attempts'])
        self.assertEquals(count_before_mark + 1, count_after_mark)

    def test_mark_default_password_changed(self):
        mark_default_password_changed()
        login_data = info()
        shell.rm(PASSWORD_CHANGED_MARKER)
        self.assertTrue(login_data['default_password_changed'])

if __name__ == '__main__':
    unittest.main()
