# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Services management
"""

import unittest
import re
import time
import os

from core.system import shell
from core.system import rpm
from core.system import sysinfo
from core.common import configuration
from core.common import version
from core.common import utils
from core.common import i18n
from core.common.version import RELEASE_TYPE

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

class RunInfo(object):
    """
    Service running information.
    """
    def __init__(self, state=None, cpu=None, mem=None, uptime=None):
        """
        Attributes:

        state       -- service state
        cpu         -- CPU usage
        mem         -- mem usage
        uptime      -- uptime
        """
        self.state = state
        self.cpu = cpu
        self.mem = mem
        self.uptime = uptime

class Service(object):
    """
    Managed service

    Attributes:

    id                          -- service id
    binary_names                -- a list of binary names for this service
    display_name                -- display name
    service_name                -- init.d service name
    package_name                -- package name for this service
    version                     -- current version
    run_at_startup              -- if true the service will be started at system boot
    backup_supported            -- True if this service supports backup action
    restore_suppored            -- True if this service supports restore action
    control_supported           -- True if this service supports start/stop/force_stop and restart actions
    chkconfig_supported         -- True if this service supports auto-start setting
    optional_service            -- True if this service is an optional one , False if it's a core one
    change_version_supported    -- True if this service supports upgrade/downgrade/install/unisntall actions
    date_time_dependent         -- True if this service needs restart when OS date time settings are changed
    mapping_dependent           -- True if this service needs restart when LAN mapping is changed for IP Office
    restart_required            -- True if this service requires restart after an upgrade or downgrade action
    run_info                    -- process state and performance data
    """

    # constants for service state

    STARTING = 0
    RUNNING = 1
    STOPPING = 2
    STOPPED = 3
    UNKNOWN = 4

    _STR2STATE = {'starting': STARTING,
                 'running': RUNNING,
                 'stopping': STOPPING,
                 'stopped': STOPPED,
                 'died': STOPPED,
                 'dead': STOPPED,
                 'unknown': UNKNOWN}

    # regular expression used to parse service status information
    # from the output of <service NAME status> command
    _SERVICE_STATUS_RE = re.compile('.+?:(.+),.+?:(.+),.+?:(.+)')

    # Number of times webcontrol checks if a service has been stopped
    RETRY_COUNT = 15

    # Time interval between the service status checks
    DELAY = 2

    def __init__(self):
        self.id = None
        self.binary_names = None
        self.display_name = None
        self.service_name = None
        self.package_name = None
        self.version = None
        self.run_at_startup = False
        self.backup_supported = False
        self.restore_supported = False
        self.control_supported = False
        self.chkconfig_supported = False
        self.optional_service = False
        self.gold_edition_service = False
        self.logging_supported = False
        self.change_version_supported = False
        self.date_time_dependent = False
        self.mapping_dependent = False
        self.restart_required = False
        self.run_info = None
        self.generic_service = None
        self.dependencies = []

    def update_run_info(self):
        """
        Refresh service running state, resource usage and performance data.

        Parse service run info from 'service <name> status' command.

        Expected output of the command is:

        <name> is starting|running|stopping|stopped|died|dead|unknown
        CPU Usage: x.x %, Memory Usage: xxxxx KB, Uptime: d-hh:mm:ss

        If the command output doesn't match the expected ouput then
        ps command is used to extract status information.
        """
        self.run_info = RunInfo(state=self.UNKNOWN, cpu=0.0, mem=0)
        stats = []

        lines = shell.sudo_execute("/etc/init.d/%s status" % self.service_name, parse=shell.LINES)
        if lines:
        # try parse service status and performance data from service <name> status command
            for s in self._STR2STATE:
                if s in lines[0]:
                    self.run_info.state = self._STR2STATE[s]
                    if self.run_info.state in (self.RUNNING, self.STOPPING, self.STARTING):
                        for line in lines[1:]:
                            match = self._SERVICE_STATUS_RE.search(line)
                            if match:
                                cpu = utils.parse_float(match.group(1))
                                mem = utils.parse_int(match.group(2))
                                uptime = match.group(3).strip()
                                if "-" in uptime:
                                    try:
                                        days, hours = uptime.split("-")
                                        if days.startswith("0"):
                                            days = days[1]
                                        if days == "1" or days == "01":
                                            uptime = i18n.custom_gettext("%s day, %s") % (days, hours)
                                        else:
                                            uptime = i18n.custom_gettext("%s days, %s") % (days, hours)
                                    except:
                                        pass
                                stats.append((cpu, mem, uptime))
                    break

        # if the service is stopped or control is not supported , don't bother in trying to extract performance data
        if self.run_info.state != self.STOPPED and self.control_supported:

            if not stats:
                # try extract performance data from ps output
                for name in self.binary_names:
                    process_status = shell.ps(name)
                    if process_status:
                        stats.append(process_status[0:3])
    
            if self.run_info.state == self.UNKNOWN:
                if stats:
                    self.run_info.state = self.RUNNING
                else:
                    self.run_info.state = self.STOPPED

            if stats:
                # consolidate performance data from each of the service's executable
                # total_CPU = sum(executable_CPU)
                # total_Mem = sum(executable_Mem)
                # uptime    = min(executable_Uptime)
                for process_status in stats:
                    self.run_info.cpu += process_status[0]
                    self.run_info.mem += process_status[1]
                    if not self.run_info.uptime:
                        self.run_info.uptime = process_status[2]
                    else:
                        if utils.parse_uptime(self.run_info.uptime) > utils.parse_uptime(process_status[2]):
                            self.run_info.uptime = process_status[2]
                # to avoid ugly formatted float values for cpu usage in UI
                self.run_info.cpu = round(self.run_info.cpu, 1)

        run_levels = shell.chkconfig(self.service_name)
        if run_levels:
            self.run_at_startup = True
        else:
            self.run_at_startup = False

    def update_service_version(self):
        """
        Updates service version info
        """
        rpm_info = rpm.query_package(self.package_name)
        if rpm_info:
            self.version = "%s.%s" % (rpm_info['VERSION'], rpm_info['RELEASE'])


    def start(self):
        """
        Start service.
        Return the exit code for 'service name start' command.
        The command is executed only if the service.control_supported is True.
        """
        if self.control_supported:
            return shell.sudo_call("systemctl start %s" % self.service_name)

    def stop(self):
        """
        Stop service.
        Return the exit code for 'service name stop' command.
        The command is executed only if the service.control_supported is True.
        """
        if self.control_supported:
            return shell.sudo_call("systemctl stop %s" % self.service_name)

    def force_stop(self):
        """
        Force stop service.
        Return the exit code for 'service name forcestop' command.
        The command is executed only if the service.control_supported is True.
        """
        if self.control_supported:
            return shell.sudo_call("systemctl stop %s" % self.service_name)

    def controlled_stop(self):
        """
        Stops service and checks if it stopped. Calls force_stop command after several retries.
        The command is executed only if the service.control_supported is True.
        """
        if self.control_supported:
            self.update_run_info()
            if self.is_running():
                self.stop()
                count = self.RETRY_COUNT
                while self.is_running():
                    time.sleep(self.DELAY)
                    count -= count
                    if count < 0:
                        self.force_stop()
                        break
                    self.update_run_info()

    def restart(self):
        """
        Restart service.
        The command is executed only if the service.control_supported is True.
        """
        if self.control_supported:
            self.update_run_info()
            if self.is_running():
                self.stop()
                count = self.RETRY_COUNT
                while self.is_running():
                    time.sleep(self.DELAY)
                    count -= count
                    if count < 0:
                        self.force_stop()
                        break
                    self.update_run_info()
                return self.start()

    def is_running(self):
        """
        Return True if the service state is in one of RUNNING, STARTING, STOPPING
        """
        return self.run_info.state in (self.RUNNING, self.STARTING, self.STOPPING)

class ServicesManager(object):
    """
    Services manager

    Usage:

    manager = ServicesManager()

    # to iterate over managed services
    for service in manager:
        print service

    # to access an individual service
    service = manager['ipoffice']
    """

    def __init__(self, config=configuration.SHARED_CONFIGURATION):
        """
        Initialize services database.

        config -- configuration data
        """
        self.config = config
        self.load_services()

    def load_services(self):
        """
        Initialize services database from configuration.
        """
        self.services = {}
        for id in self.config['applications']:
            if id == "webrtcgw" and os.path.exists("/opt/Avaya/.russian_licenses"):
                pass
            else:
                section = self.config['applications'][id]
                try:
                    package_name = section['package_name'][version.RELEASE_TYPE]
                    rpm_info = rpm.query_package(package_name)
                    if rpm_info:
                    # service is installed
                        service = Service()
                        service.id = id
                        service.package_name = package_name
                        service.binary_names = section.as_list('binary_names')
                        service.display_name = section['display_name']
                        service.service_name = section['service_name']
                        service.backup_supported = section.as_bool('backup_supported')
                        service.restore_supported = section.as_bool('restore_supported')
                        service.control_supported = section.as_bool('control_supported')
                        service.chkconfig_supported = section.as_bool('chkconfig_supported')
                        service.logging_supported = section.as_bool('logging_supported')
                        service.change_version_supported = section.as_bool('change_version_supported')
                        service.date_time_dependent = section.as_bool('date_time_dependent')
                        service.mapping_dependent = section.as_bool('mapping_dependent')
                        service.optional_service = section.as_bool('optional_service')
                        service.gold_edition_service = section.as_bool('gold_edition_service')
                        service.restart_required = section.as_bool('restart_required')
                        if "dependencies" in section:
                            service.dependencies = section.as_list('dependencies')
                        if "generic_service" in section:
                            service.generic_service = section.as_bool("generic_service")
                        self.services[id] = service
                        release_props = "%s_props" % RELEASE_TYPE
                        if release_props in section:
                            if "backup_supported" in section[release_props]:
                                service.backup_supported = section[release_props].as_bool("backup_supported")
                            if "restore_supported" in section[release_props]:
                                service.restore_supported = section[release_props].as_bool("restore_supported")
                            if "display_name" in section[release_props]:
                                service.display_name = section[release_props]["display_name"]
                except KeyError:
                # service is not managed in this release
                    pass

    def exclude_gold_edition_service(self, service):
        """
        Check if Server Edition version is gold edition, exclude gold-only services if not
        """
        if sysinfo.gold_edition() is False:
            return service.gold_edition_service
        return False


    def start_all(self):
        """
        Start all services
        """
        for service in self:
            if self.exclude_gold_edition_service(service):
                continue
            service.start()

    def stop_all(self):
        """
        Stop all services
        """
        for service in self:
            if self.exclude_gold_edition_service(service):
                continue
            service.controlled_stop()

    def restart_all(self):
        """
        Restart all services
        """
        for service in self:
            if self.exclude_gold_edition_service(service):
                continue
            service.restart()

    def restart_services(self, services_array):
        """
        Restart the selected services

        Input:
        - services_array -- array of services ids to restart
        """
        for id in services_array:
            service = self[id]
            if service:
                if self.exclude_gold_edition_service(service):
                    continue
                service.restart()

    def update_run_info(self):
        """
        Update running info for all services
        """
        for service in self:
            service.update_run_info()

    def update_services_version(self):
        """
        Update version info fro all services
        """
        for service in self:
            service.update_service_version()

    def dependent_services(self, property):
        """
        Return the running services that depend on a certain attribute
        """
        affected_services = []
        for service in self:
            if rpm.query_package(service.package_name) is None:
                continue
            if len(property) > 0:
                if getattr(service, property):
                    self.append_running_services(affected_services, service)
            else:
                self.append_running_services(affected_services, service)
        return affected_services

    def append_running_services(self, service_array, service):
        """
        Appends an object containing a service id and display name to an array.

        Inputs:
        - service_array -- the container array
        - service       -- an instance of the Service class
        """
        service.update_run_info()
        if service.is_running():
            service_array.append({'id': service.id,
                                  'display_name': service.display_name})

    def __iter__(self):
        return self.services.values().__iter__()

    def __getitem__(self, key):
        return self.services[key]

    def __len__(self):
        return len(self.services)

class Test(unittest.TestCase):

    def setUp(self):
        self.manager = ServicesManager()

    def test_init_manager(self):
        self.assertTrue(self.manager.services)

    def test_load_service(self):
        # {package_name: service_id}
        test_data = {'vmpro': 'voicemail', 
                     'ipoffice': 'ipoffice',
                     'oneXportal': 'onexportal'}
        for package_name in test_data:
            if rpm.query_package(package_name):
                self.assertTrue(test_data[package_name] in self.manager.services)
            else:
                self.assertFalse(test_data[package_name] in self.manager.services)

    def test_start_stop_service(self):
        # [service_id0,...]
        test_data = ['watchdog', 'voicemail', 'ipoffice']
        for service_id in test_data:
            if service_id in self.manager.services:
                service = self.manager[service_id]
                if service.control_supported:
                    service.update_run_info()
                    if service.is_running():
                        service.stop()
                        expected_state = Service.STOPPED
                    else:
                        service.start()
                        expected_state = Service.RUNNING

                    import time
                    time.sleep(10)

                    service.update_run_info()
                    if expected_state == Service.STOPPED and service.is_running():
                        service.force_stop()

                    time.sleep(10)

                    service.update_run_info()
                    self.assertEqual(service.run_info.state, expected_state)

    def test_restart_service(self):
        test_data = ['watchdog', 'voicemail', 'ipoffice']
        for service_id in test_data:
            if service_id in self.manager.services:
                service = self.manager[service_id]
                service.update_run_info()
                if service.is_running():
                    expected_state = Service.RUNNING
                else:
                    expected_state = Service.STOPPED
                service.restart()

                service.update_run_info()
                self.assertEqual(service.run_info.state, expected_state)

    def test_service_update_run_info(self):
        # [service_id0,...]
        test_data = ['watchdog', 'voicemail', 'ipoffice']
        for service_id in test_data:
            if service_id in self.manager.services:
                service = self.manager[service_id]
                service.update_run_info()
                if service.is_running():
                    self.assertTrue(service.run_info.cpu)
                    self.assertTrue(service.run_info.mem)
                    self.assertTrue(service.run_info.uptime)

if __name__ == "__main__":
    unittest.main()
