# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Utility module for handling Avaya Inside applications settings 
"""

import unittest
import re

from core.system import shell
from core.common import log

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

def get_logs_age(config):
    """
    Returns the log files age (days) setting.
    The value is global for all Avaya Inside applications so the search stops 
    with the first log config file containing the parameter.
    
    Input:
    
    config -- a dictionary on the following form:
              keys   = the path to log config files 
              values = a list containing specific sections within those files 
                       from where the 'rotate' parameter should be read or 'None' 
                       if this can be read from any section
    """
    logs_age = None
    rotate_param = '(^|\s)rotate(\s)'
    for log_config in config.keys():
        try:
            with shell.file_open(log_config) as f:
                log_lines = f.readlines()
            log_section = None
            for line in log_lines:
                line = line.strip()
                if line:
                    if line.endswith('{'):
                        log_section = line.partition('{')[0].rstrip()
                        continue
                    if not config[log_config] or log_section in config[log_config]:
                        if re.search(rotate_param, line):
                            parts = line.split(' ')
                            logs_age = int(parts[-1].strip())
                            return logs_age
        except (IOError, OSError) as e:
            log.SHARED_LOGGER.warn('unable to parse  log rotate parameter from: %s, error: %s' %
                                   (log_config, str(e)))

    return logs_age

def set_logs_age(config, age):
    """
    Set the log files age (days) for rotating logs.
    
    Input:
    
    config -- a dictionary on the following form:
              keys   = the path to log config files 
              values = a list containing specific sections within those files 
                       where the 'rotate' parameter should be changed or 'None' 
                       if the change should apply to all sections 
    """
    rotate_param = '(^|\s)rotate(\s)'
    for log_config in config.keys():
        try:
            with shell.file_open(log_config) as f:
                log_lines = f.readlines()
            log_section = None
            for i, line in enumerate(log_lines):
                nline = line.strip()
                if nline:
                    if nline.endswith('{'):
                        log_section = nline.partition('{')[0].rstrip()
                        continue
                    if not config[log_config] or any([section in log_section for section in config[log_config]]):
                        if re.search(rotate_param, nline):
                            parts = nline.split(' ')
                            old_age = parts[-1].strip()
                            log_lines[i] = line.replace(old_age, str(age))

            with shell.file_open(log_config, 'w') as f:
                f.writelines(log_lines)
        except (IOError, OSError) as e:
            log.SHARED_LOGGER.warn('unable to save log rotate parameter to: %s, error: %s' %
                                   (log_config, str(e)))

def read_snmpd_settings(snmpd_file, snmpd_params):
    """
    Read SNMP settings from file into dictionary
    """
    with shell.file_open(snmpd_file) as f:
        lines = f.readlines()
    params = snmpd_params.copy()
    for i, line in enumerate(lines):
        line = line.strip()
        if line.startswith('#'):
            continue
        for param_name in params:
            if line.lower().startswith(param_name.lower()):
                parts = line.split(' ')
                snmpd_params[param_name] = ' '.join(parts[1:])
                del params[param_name]
                break
        if not params:
            break

def update_snmpd_config(snmpd_file, snmpd_params):
    """
    Save SNMP settings to file.
    """
    with shell.file_open(snmpd_file) as f:
        lines = f.readlines()

    params = snmpd_params.copy()
    for i, line in enumerate(lines):
        if line.lstrip().startswith('#'):
            continue
        for param_name in params:
            tmp_line = line.strip()
            if tmp_line.lower().startswith(param_name.lower()):
                parts = tmp_line.split(' ')
                parts[1:] = [str(params[param_name])]
                lines[i] = ' '.join(parts) + '\n'
                del params[param_name]
                break
        if not params:
            break

    if params:
        for p in params:
            lines.append('\n%s %s\n' % (p, params[p]))
    with shell.file_open(snmpd_file, 'w') as f:
        f.writelines(lines)


class Test(unittest.TestCase):

    def setUp(self):
        self.config = {'/etc/logrotate.d/ipoffice': None,
                       '/etc/logrotate.d/vmpro_logrotate': None}

    def test_get_logs_age(self):
        age = get_logs_age(self.config)
        self.assertTrue(age)

    def test_read_snmp_settings(self):
        wdog_snmp_settings = {'sysdescr': None, 'syslocation': None}
        
        from core.common import configuration
        wdog_snmp_file = configuration.SHARED_CONFIGURATION['snmp']['snmpd_config']

        read_snmpd_settings(wdog_snmp_file, wdog_snmp_settings)
        for k in wdog_snmp_settings:
            self.assertTrue(wdog_snmp_settings[k])

if __name__ == "__main__":
    unittest.main()    
