# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Handle backup and restore for installed applications.
"""
import unittest
import os
import threading

from core.system import shell
from core.common import configuration
from core.common import i18n
from core.common import log

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

class Error(Exception):
    pass

class ApplyRestore(threading.Thread):
    """
    Working thread for restoring an application
    """
    def __init__ (self):
        self.command = ""
        self.lock = threading.Lock()
        threading.Thread.__init__ (self)

    def run(self):
        """
        Running restore operation and update status info.
        """
        shell.sudo_call(self.command)
        with self.lock:
            self.command = ""

    def set_command(self, script, path):
        """
        Set the restore command.
        """
        self.command = '%s /Restore:"%s"' % (script, path)

    def get_restore_status(self):
        """
        Retrieve the status of a restore process.
        """
        with self.lock:
            restore_status = {
                'in_progress': self.isAlive()
            }
        return restore_status

class ApplyRestoreManager(object):
    """
    Manage restore operations.
    """
    def __init__ (self):
        self.restore_thread = None

    def apply_restore(self, script, path):
        """
        Apply restore for an application. The update process can take some time, therefore
        it runs in a separate thread.

        Use get_restore_status() to find details about the running restore.

        Input:

        script -- the script to be used to apply the restore
        path   -- the path to the restore folder
        """
        if self.restore_thread and self.restore_thread.isAlive():
            raise Error(i18n.custom_gettext("A restore operation is still in progress."))
        else:
            if script is None:
                script = ""
            if path is None:
                path = ""
            self.restore_thread = ApplyRestore()
            self.restore_thread.set_command(script, path)
            self.restore_thread.setDaemon(True)
            self.restore_thread.start()

    def get_restore_status(self):
        """
        Retrieve the status of the currently running restore.
        """
        if self.restore_thread:
            return self.restore_thread.get_restore_status()
        else:
            restore_status = {
                'in_progress': False
            }
            return restore_status

restore_manager = ApplyRestoreManager()

class BackupError(Exception):
    """
    Backup and restore errors.
    """
    def __str__(self):
        return repr(self.args[0])

class BackupManager(object):
    """
    Manage backup activities
    """

    def __init__(self, config=configuration.SHARED_CONFIGURATION):
        """
        config -- app configuration data
        """
        self.scripts = {'backup': config['backup']['backup_script'],
                        'backup_webrtcgw': config['backup']['backup_webrtcgw'],
                        'backup_location_webrtcgw': config['backup']['backup_location_webrtcgw'],
                        'restore': config['backup']['restore_script'],
                        'restore_vmpro': config['backup']['restore_script_vmpro'],
                        'restore_webrtcgw': config['backup']['restore_script_webrtcgw']}

    def backup(self, service):
        """
        Create a new backup:
        
        service - name of the service to create the backup for
        """
        if service == "webrtcgw":
            msg = shell.sudo_execute('%s' % self.scripts['backup_webrtcgw'], parse=shell.RAW)
        else:
            msg = shell.sudo_execute('%s %s' % (self.scripts['backup'], str(service)), parse=shell.RAW)
        return msg.strip()

    def list(self, service):
        """
        Lists available backups:

        service - name of the service to create the backup for
        """
        msg = shell.sudo_execute('%s list %s' % (self.scripts['backup'], str(service)), parse=shell.RAW)
        return msg.strip()

    def restore(self, service):
        """
        Restore service configuration from backup:
        
        service - name of the service
        """
        msg = shell.sudo_execute('%s %s' % (self.scripts['restore'], str(service)), parse=shell.RAW)
        return msg.strip()

    def webrtcgw_backups_list(self):
        """
        List backups for webrtcgw.
        """
        backups = []
        output = shell.sudo_execute("ls %s" % self.scripts['backup_location_webrtcgw'], parse=shell.LINES)
        if output:
            for f in output:
                if os.path.isfile(os.path.join(self.scripts['backup_location_webrtcgw'], f)) and ".tar.gz" in f:
                    backups.append(f)
        return backups
    
    def vmpro_backups_list(self):
        """
        List backups:
        
        service - name of the service to list the backups for
        """
        backups = shell.sudo_execute('%s /List:All' % self.scripts['restore_vmpro'], parse=shell.LINES)
        backups_info = []
        for backup in backups:
            backup = backup.strip()
            if backup and (backup != '<Backups>') and (backup != '</Backups>'):
                info_parts = backup.split('|')
                if len(info_parts) < 6:
                    continue
                raw_contents = int(info_parts[3])
                contents_mask = {'Voicemails': 0x0400,
                                 'User Settings and Greetings': 0x1000,
                                 'Campaigns': 0x0010,
                                 'Call Flow Modules and Conditions': 0x0001,
                                 'Module Recording': 0x0004,
                                 'System Settings': 0x0040}
                contents = []
                for key in contents_mask:
                    if contents_mask[key] & raw_contents == contents_mask[key]:
                        contents.append(key)

                if len(info_parts) == 6:
                    info = {'name': os.path.basename(info_parts[0]),
                            'location': info_parts[0],
                            'type': info_parts[1],
                            'timestamp': info_parts[2],
                            'contents': contents,
                            'size': info_parts[4],
                            'os': info_parts[5]}
                    backups_info.append(info)
        return backups_info
    
    def webrtcgw_restore(self):
        """
        Restore webrtcgw configuration from backup:
        """
        output = shell.sudo_execute('%s' % self.scripts['restore_webrtcgw'], parse=shell.LINES)
        msg = None
        if output:
            for line in output:
                if line.startswith("ERROR"):
                    msg = line.split(":")[1]
                    break
        return msg

    def vmpro_restore(self, backup_path=None):
        if not backup_path:
            backups = sorted(self.vmpro_backups_list(), key=lambda k: k['timestamp'], reverse=True)
            if backups:
                backup_path = backups[0]['location']
            else:
                # XXX is this really required ???
                raise BackupError(i18n.custom_gettext("No backups from which to perform a restore."))

        restore_manager.apply_restore(self.scripts['restore_vmpro'], backup_path)

class Test(unittest.TestCase):

    def setUp(self):
        self.backup_manager = BackupManager()

    def test_backup(self):
        exit_msg = self.backup_manager.backup('ipoffice')
        self.assertEqual(exit_msg, 'Backup complete')

    def test_restore(self):
        exit_msg = self.backup_manager.restore('ipoffice')
        self.assertEqual(exit_msg, 'Backup restored')

    def test_restore_vmpro(self):
        try:
            exit_msg = self.backup_manager.vmpro_backups_list()
            exit_code = self.backup_manager.vmpro_restore()
            self.assertEqual(exit_code, 0)
        except BackupError as e:
            self.fail("restore error: %s, output: %s" % (str(e), exit_msg))

if __name__ == '__main__':
    unittest.main()
