# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Manage updates for Avaya Inside applications.
"""

import glob
import unittest
from urllib.request import urlopen
from urllib.error import *
import os
#import BaseHTTPServer
from http.server import BaseHTTPRequestHandler,HTTPServer
import threading
import services_rest_api
import web

from web.bs4 import BeautifulSoup

from core.system import rpm
from core.system import shell
from core.settings import repositories
from core.common import configuration
from core.common import log
from core.common import version
from core.common import i18n

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

class AccessRepoError(Exception):
    """
    For listing repository errors.
    """
    def __str__(self):
        return self.args[0]

class AppStatus(object):
    """
    Package status for a managed application.
    """

    # app update status

    UP_TO_DATE = 0
    OUTDATED = 1
    NOT_INSTALLED = 2
    UNKNOWN = 3

    def __init__(self, id, display_name, description,
                 current_version=None,
                 available_versions=None,
                 change_version_supported=True):
        """
        Args:

        id                  -- app ID (service ID)
        display_name        -- app display name to be used in UI
        current_version     -- app current version
        available_versions  -- a list with available versions
        change_version_supported -- True if this application supports upgrade/downgrade
        """
        self.id = id
        self.display_name = display_name
        self.description = description
        self.current_version = current_version
        self.change_version_supported = change_version_supported
        self.update(available_versions)

    def update(self, available_versions):
        """
        Update app package status with available versions. 
        """
        self.available_versions = available_versions
        from functools import cmp_to_key
        if self.available_versions:
            self.available_versions.sort(key=cmp_to_key(rpm.compare_versions))

        if self.current_version:
            self.status = self.UP_TO_DATE
            self.latest_version = self.current_version
            if self.available_versions:
                if rpm.compare_versions(self.current_version, self.available_versions[-1]) < 0:
                    self.latest_version = self.available_versions[-1]
                    self.status = self.OUTDATED
        else:
            self.status = self.NOT_INSTALLED
            if self.available_versions:
                self.latest_version = self.available_versions[-1]

    def is_outdated(self):
        return self.status == self.OUTDATED

class AppManager(object):
    """
    Upgrade/install and uninstall applications.
    """

    APPS_REPOSITORY_NAME = 'apps'
    REQUIRED_RPM_TAGS = ['NAME', 'VERSION', 'RELEASE']

    def __init__(self, services_manager, config=configuration.SHARED_CONFIGURATION):
        """
        Args:

        services_manager -- installed applications
        config           -- app configuration
        """
        self.services_manager = services_manager
        self.config = config
        self.url_open_timeout = config['webapp'].as_int('url_open_timeout')

    def get_version_status(self):
        status = []
        for app_id in self.config['applications']:
            if self.is_managed(app_id):
                package_name       = self.package_name_for_id(app_id)
                rpm_info           = rpm.query_package(package_name)

                if rpm_info:
                    current_version = '%s.%s' % (rpm_info['VERSION'], rpm_info['RELEASE'])
                else:
                    current_version = None

                app_config = self.config['applications'][app_id]
                display_name = app_config['display_name']
                description = self.return_description(package_name, True, web.ctx.session.get('lang'))
                release_props = "%s_props" % version.RELEASE_TYPE
                if release_props in app_config:
                    if "display_name" in app_config[release_props]:
                        display_name = app_config[release_props]["display_name"]


                app_status = AppStatus(app_id,
                                       display_name=display_name,
                                       description=description,
                                       current_version=current_version,
                                       available_versions=[],
                                       change_version_supported=False)
                app_status.status = AppStatus.UNKNOWN
                app_status.latest_version = None
                if current_version and app_config.as_bool('change_version_supported'):
                    status.append(app_status)
        return status

    def get_status(self):
        """
        Return update status for all applications.         
        """
        repo_packages = self.list_repo_packages()
        status = []

        for app_id in self.config['applications']:
        # managed services
            if self.is_managed(app_id):
                app_config = self.config['applications'][app_id]
                package_name       = self.package_name_for_id(app_id)
                rpm_info           = rpm.query_package(package_name)
                available_versions_number = 0

                if package_name in repo_packages and app_config.as_bool('change_version_supported'):
                    available_versions = [v for v in repo_packages[package_name]]
                    del repo_packages[package_name]
                    available_versions_number = len(available_versions)
                else:
                    available_versions = []

                if rpm_info:
                    current_version = '%s.%s' % (rpm_info['VERSION'], rpm_info['RELEASE'])
                    available_versions = [v for v in available_versions
                                          if rpm.compare_versions(current_version, v) != 0]
                else:
                    current_version = None

                display_name = app_config['display_name']
                description = self.return_description(package_name, True, web.ctx.session.get('lang'))
                release_props = "%s_props" % version.RELEASE_TYPE
                if release_props in app_config:
                    if "display_name" in app_config[release_props]:
                        display_name = app_config[release_props]["display_name"]


                app_status = AppStatus(app_id,
                                       display_name=display_name,
                                       description=description,
                                       current_version=current_version,
                                       available_versions=available_versions,
                                       change_version_supported=app_config.as_bool('change_version_supported'))
                if available_versions_number > 0 or app_status.status != AppStatus.NOT_INSTALLED:
                    if app_config.as_bool('change_version_supported'):
                        if (app_status.id == "webRTCGateway" or app_status.id == "webrtcgw")and os.path.exists("/opt/Avaya/.russian_licenses"):
                            pass
                        else:
                            status.append(app_status)

        for package_name in repo_packages:
        # non-managed services
            app_id = self.id_for_package_name(package_name)
            if self.is_managed(app_id):
                rpm_info           = rpm.query_package(package_name)
                available_versions = [v for v in repo_packages.get(package_name, [])]

                if rpm_info:
                    current_version = '%s.%s' % (rpm_info['VERSION'], rpm_info['RELEASE'])
                    # exclude current installed version
                    available_versions = [v for v in available_versions
                                          if rpm.compare_versions(current_version, v) != 0]
                else:
                    current_version = None

                display_name = app_id
                if "vmpro-wavs" in app_id:
                    display_name = app_id.replace("vmpro", "Voicemail")
                description = self.return_description(package_name, False, web.ctx.session.get('lang'))


                app_status = AppStatus(app_id,
                                       display_name=display_name,
                                       description=description,
                                       current_version=current_version,
                                       available_versions=available_versions)
                status.append(app_status)
            else:
                log.SHARED_LOGGER.warn('ignoring unmanaged application %s' % app_id)

        return status

    def upgrade(self, id, new_version):
        """
        Upgrade or downgrade installed application.

        Args:

        id          -- application ID or package name
        new_version -- version to upgrade/downgrade to

        If

        1. there's no package with new_version in repository        OR
        2. there's an Avaya application not part of current release OR
        3. the package or application is not installed

        this method will do nothing.

        A special package is the Web Control itself, in this case
        the web application will be restarted after update.
        """
        if self.is_managed(id):
            package_name = self.package_name_for_id(id)
            rpm_info = rpm.query_package(package_name)
            if rpm_info:
                current_version = '%s.%s' % (rpm_info['VERSION'], rpm_info['RELEASE'])
                repo_packages = self.list_repo_packages()
                if package_name in repo_packages:
                    if new_version in repo_packages[package_name]:
                        version_diff = rpm.compare_versions(current_version, new_version)
                        rpm_url = repo_packages[package_name][new_version]

                        # required if some service requires restart after package upgrade/downgrade
                        service_name = None

                        # verify if the package corresponds to one of the managed services
                        # if yes collect service name to restart it after upgrade/downgrade
                        # XXX the rpm upgrade section it may have its own restart action after applying upgrade;
                        #     we cannot check for that so that scenario is another story...
                        app_id = self.id_for_package_name(package_name)
                        if self.is_avaya_app(app_id):
                            service_name = self.config['applications'][app_id]['service_name']

                        # HACK  for ms (Media Server) and cli-commands: we need to add --nodeps extra option
                        #       to allow upgrade/downgrade
                        if id == 'ms' or id == 'cli-commands':
                            extra_opts = '--nodeps'
                        elif id == "demo-default-config":
                            extra_opts = '--force'
                        else:
                            extra_opts = None

                        if rpm.valid_rpm(rpm_url):
                            if version_diff < 0:
                                rpm.upgrade(rpm_url, service_name=service_name, extra_opts=extra_opts)
                                if self.is_avaya_app(id):
                                    self.services_manager.load_services()
                            elif version_diff > 0:
                                rpm.downgrade(rpm_url, service_name=service_name, extra_opts=extra_opts)
                                if self.is_avaya_app(id):
                                    self.services_manager.load_services()
                            else:
                                log.SHARED_LOGGER.warn("%s will NOT be upgraded, current version is %s, new version is %s" %
                                                       (id, current_version, new_version))
                        else:
                            raise AccessRepoError(i18n.custom_gettext('The %s rpm is corrupted. Please upload it again.')
                            % os.path.basename(rpm_url))
                    else:
                        log.SHARED_LOGGER.warn('unknown version %s for %s for upgrade' % (new_version, id))
                else:
                    log.SHARED_LOGGER.warn('no upgrade packages for %s' % id)
            else:
                log.SHARED_LOGGER.warn('trying to upgrade not installed application %s' % id)
        else:
            log.SHARED_LOGGER.warn('trying to upgrade unamanaged application %s to version %s' % (id, new_version))

    def install(self, id, new_version):
        """
        Install new application.

        Args:

        id          -- application ID
        new_version -- which version to install

        If there's no package with specified version in repository this method will do nothing.
        """
        if self.is_managed(id):
            package_name = self.package_name_for_id(id)
            repo_packages = self.list_repo_packages()
            if package_name in repo_packages:
                if new_version in repo_packages[package_name]:
                    rpm_url = repo_packages[package_name][new_version]
                    if rpm.valid_rpm(rpm_url):
                        if id == 'demo-default-config':
                            extra_opts = '--force'
                        else:
                            extra_opts = None
                        output = rpm.install(rpm_url, extra_opts)
                        if self.is_avaya_app(id):
                            self.services_manager.load_services()
                        if output:
                            raise rpm.RpmError(output)
                    else:
                        raise AccessRepoError(i18n.custom_gettext('The %s rpm is corrupted. Please upload it again.')
                            % os.path.basename(rpm_url))
                else:
                    log.SHARED_LOGGER.warn('unknown version %s for %s for installation' % (new_version, id))
            else:
                log.SHARED_LOGGER.warn('no install packages for %s' % id)
        else:
            log.SHARED_LOGGER.warn('trying to install unmanaged application %s' % id)

    def uninstall(self, id):
        """
        Uninstall application.

        Args:

        id -- application ID or package name
        """
        if self.is_managed(id):
            package_name = self.package_name_for_id(id)
            if package_name == "asgsshd" or package_name == "asgtools":
                output = rpm.uninstall(package_name, "--nodeps")
            else:
                output = rpm.uninstall(package_name)
            if self.is_avaya_app(id):
                self.services_manager.load_services()
            if output:
                raise rpm.RpmError(output)
        else:
            log.SHARED_LOGGER.warn('trying to unistall unmanaged application %s' % id)

    def list_repo_packages(self):
        """
        List applications repository and return a dictionary:

        { package_name_1: {version1: rpm_url1, version2: rpm_url2},... }
        """
        available_packages = {}
        app_repo = repositories.get_config(configuration.SHARED_CONFIGURATION['repositories']['config_file'],
                                           self.APPS_REPOSITORY_NAME)[self.APPS_REPOSITORY_NAME]
        rpm_files = []
        if app_repo['local']:
            for f in  glob.glob('%s/*.rpm' % app_repo['url'].replace('file://', '')):
                rpm_files.append(f)
        else:
            try:
                page = urlopen(app_repo['url'], timeout=self.url_open_timeout)
                if page.code == 200:
                    soup = BeautifulSoup(page.read())
                    for link in soup('a'):
                        rpm_files.append(os.path.join(app_repo['url'], str(link['href'])))
                else:
                    raise AccessRepoError(i18n.custom_gettext('Could not connect to applications repository. (HTTP error: %s)')
                                          % page.code)
            except HTTPError as e:
                log.SHARED_LOGGER.exception('unable to connect to applications repository')
                raise AccessRepoError(i18n.custom_gettext('Could not connect to applications repository. (HTTP Error: %s - %s)')
                                      % (e.code, BaseHTTPServer.BaseHTTPRequestHandler.responses[e.code]))
            except URLError as e:
                log.SHARED_LOGGER.exception('unable to connect to applications repository')
                raise AccessRepoError(i18n.custom_gettext('Could not connect to applications repository. (URL Error: %s)') % e.reason)
        for rpm_file in rpm_files:
            rpm_info = rpm.query_rpm(rpm_file)
            if rpm_info:
                missing_tags = []
                for tag in self.REQUIRED_RPM_TAGS:
                    if not tag in rpm_info:
                        missing_tags.append(tag)
                if not missing_tags:
                    available_packages.setdefault(rpm_info['NAME'],
                                              {})['%s.%s' % (rpm_info['VERSION'], rpm_info['RELEASE'])] = rpm_file
                else:
                    log.SHARED_LOGGER.error('malformed RPM file: %s, missing tags: %s'
                                            % (rpm_file, missing_tags))
            else:
                log.SHARED_LOGGER.warn('error analyzing rpm file: %s' % rpm_file)
        return available_packages

    def package_name_for_id(self, id):
        """
        Return package name for specified application ID according to current release type.

        For non Avaya applications this method returns id.

        Args:

        id -- application ID or package name
        """
        if id in self.config["applications"]:
            return self.config['applications'][id]['package_name'][version.RELEASE_TYPE]
        else:
            return id

    def change_version_supported_for_id(self, id):
        """
        Return package name for specified application ID according to current release type.

        For non Avaya applications this method returns id.

        Args:

        id -- application ID or package name
        """
        if id in self.config["applications"]:
            return self.config['applications'][id]['change_version_supported']
        else:
            return id

    def id_for_package_name(self, package_name):
        """
        Return application ID for specified package name.

        For non Avaya application this method returns package_name

        Args:

        package_name -- package_name
        """
        for id in self.config['applications']:
            if self.config['applications'][id].as_bool("restart_required") == True:
                if package_name in [name for name in self.config['applications'][id]['package_name'].values()]:
                    return id
        return package_name

    def display_name_for_id(self, id):
        """
        Return application display name for specified ID.

        For non Avaya applications this method returns id.

        Args:

        id  -- application ID or package name
        """
        if id in self.config["applications"]:
            return self.config['applications'][id]['display_name']
        else:
            return id

    def is_managed(self, id):
        """
        Return False for Avaya applications not managed in current release.
        For example IP Office is not managed in Application Server release.

        id -- application ID
        """
        if id in self.config['applications']:
            return version.RELEASE_TYPE in self.config['applications'][id]['package_name']
        return True

    def is_avaya_app(self, id):
        """
        Return True if specified ID corresponds to an Avaya application.

        id -- application ID
        """
        return id in self.config['applications']

    def return_description(self, name=None, managed=False, lang="en_US"):
        """
        Get the description corresponding to a package.

        name    -- package name to look up.
        managed -- True for applications which have a corresponding .ini file, False otherwise.
        lang    -- Session language.
        """

        description = ""
        default_description = ""
        description_found = False
        descriptions_file = configuration.SHARED_CONFIGURATION['repositories']['local']['descr_file_basename']\
            .replace("en_US", lang)
        default_avaya_descr = configuration.SHARED_CONFIGURATION['repositories']['local']['descr_avaya_default']
        default_os_descr = configuration.SHARED_CONFIGURATION['repositories']['local']['descr_os_default']

        if descriptions_file and os.path.exists(descriptions_file):
            with shell.file_open(descriptions_file) as infile:
                for line in infile:
                    if managed is True:
                        if line.startswith(default_avaya_descr):
                            default_description = line.split("%s=" % default_avaya_descr)[1].strip()
                    else:
                        if line.startswith(default_os_descr):
                            default_description = line.split("%s=" % default_os_descr)[1].strip()
                    if line.startswith(name):
                        description = line.split("%s=" % name)[1].strip()
                        description_found = True
                        break
        if description_found is False:
            description = default_description

        return description


class Error(Exception):
    pass

class ApplicationsAllUpdater(threading.Thread):
    """
    Working thread for updating all applications
    """
    def __init__ (self):
        self.packages = []
        self.lock = threading.Lock()
        threading.Thread.__init__ (self)

    def run(self):
        """
        Running rpm update on outdated packages and updates status info.
        """
        for i, package in enumerate(self.packages):
            app_manager = AppManager(services_rest_api.services_manager, config=configuration.SHARED_CONFIGURATION)
            app_manager.upgrade(package["id"], package["version"])
        with self.lock:
            self.packages = []

    def set_packages(self, packages):
        """
        Set the packages to be updated.
        """
        self.packages = packages

    def get_update_status(self):
        """
        Retrieve the status of an update process.
        """
        with self.lock:
            update_status = {
                'in_progress': self.isAlive()
            }
        return update_status

class AllApplicationsUpdateManager(object):
    """
    Manage All Applications software updates and patches.
    """
    def __init__ (self):
        self.updater_thread = None

    def apply_updates(self, packages):
        """
        Apply updates for the selected packages. The update process can take some time, therefore
        it runs in a separate thread.

        Use get_update_status() to find details about the running update.

        Input:

        packages = the list of packages to be updated
        """
        if self.updater_thread and self.updater_thread.isAlive():
            raise Error(i18n.custom_gettext("An application system update is still in progress."))
        else:
            if packages is None:
                packages = []
            self.updater_thread = ApplicationsAllUpdater()
            self.updater_thread.set_packages(packages)
            self.updater_thread.setDaemon(True)
            self.updater_thread.start()

    def get_update_status(self):
        """
        Retrieve the status of the currently running update.
        """
        if self.updater_thread:
            return self.updater_thread.get_update_status()
        else:
            update_status = {
                'in_progress': False
            }
            return update_status

class Test(unittest.TestCase):

    def setUp(self):
        from core.services import manager

        self.app_manager = AppManager(manager.ServicesManager())

    def test_get_status(self):
        try:
            self.assertTrue(self.app_manager.get_status())
        except AccessRepoError:
        # no need to fail the test if the repo is not accessible
            pass

    def test_package_name_for_id(self):
        avaya_apps = ('watchdog', 'voicemail', 'onexportal')
        for id in avaya_apps:
            self.assertTrue(self.app_manager.package_name_for_id(id))
        if version.RELEASE_TYPE == 'isa' or version.RELEASE_TYPE == 'abe':
            self.assertTrue(self.app_manager.package_name_for_id('ipoffice'))
        self.assertEqual('ntp', self.app_manager.package_name_for_id('ntp'))

    def test_id_for_package_name(self):
        avaya_packages = ('watchdog', 'vmpro', 'oneXportal', 'ipoffice')
        for name in avaya_packages:
            self.assertTrue(self.app_manager.id_for_package_name(name))

    def test_display_name_for_id(self):
        avaya_apps = ('watchdog', 'voicemail', 'onexportal', 'ipoffice')
        for id in avaya_apps:
            self.assertTrue(self.app_manager.display_name_for_id(id))
        self.assertEqual('ntp', self.app_manager.display_name_for_id('ntp'))

    def test_list_repo_packages(self):
        try:
            repo_packages = self.app_manager.list_repo_packages()
            self.assertTrue(isinstance(repo_packages, dict))
        except AccessRepoError:
        # no need to fail the test if the repo is not accessible
            pass

    def test_is_managed(self):
        avaya_apps = ('watchdog', 'voicemail', 'onexportal')
        rel_type = False
        for id in avaya_apps:
            self.assertTrue(self.app_manager.is_managed(id))
        if version.RELEASE_TYPE == 'isa' or version.RELEASE_TYPE == 'abe':
            rel_type = True
        self.assertEqual(rel_type, self.app_manager.is_managed('ipoffice'))
        self.assertTrue(self.app_manager.is_managed('ntp'))

    def test_is_avaya_app(self):
        default_apps = ('watchdog', 'voicemail', 'onexportal', 'ipoffice')
        for id in default_apps:
            self.assertTrue(self.app_manager.is_avaya_app(id))
        self.assertFalse(self.app_manager.is_avaya_app('ntp'))

if __name__ == '__main__':
    unittest.main()
