# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Manage OS updates
"""

import unittest
import threading

from core.system import yum
from core.system import shell
from core.system import sysinfo
from core.common import i18n
from core.common import version

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

class Error(Exception):
    pass

class OSUpdater(threading.Thread):
    """
    Working thread for updating selected packages
    """
    def __init__ (self):
        self.packages = []
        self.current = 0
        self.completed = 0
        self.reboot_required = False
        self.lock = threading.Lock()
        threading.Thread.__init__ (self)
         
    def run(self):
        """
        Running yum update on selected packages and updates status info.
        """
        self.reboot_required = False
        reboot = False
        webcontrol = ""
        for i, package in enumerate(self.packages):
            with self.lock:
                self.completed = i
                self.current = package
                
            if 'webcontrol' in package:
                webcontrol = webcontrol + " " + package
            elif 'cli-commands' in package:
                webcontrol = webcontrol + " " + package
            else:
                shell.sudo_call('yum -y update %s' % package)
            if 'kernel' in package:
                reboot = True

        if webcontrol != "":
            shell.sudo_call('nohup yum -y update %s &' % webcontrol)

        with self.lock:
            self.reboot_required = reboot
            self.packages = []
            self.current = ''
            self.completed = 0
        sysinfo.mark_last_update()

    def set_packages(self, packages):
        """
        Set the packages to be updated.
        """
        self.packages = packages

    def get_update_status(self):
        """
        Retrieve the status of an update process. 
        """
        with self.lock:
            update_status = {
                'in_progress': self.isAlive(),
                'current': self.current,
                'total': len(self.packages),
                'completed': self.completed,
                'require_reboot': self.reboot_required
            }
        return update_status

class PackageStatus(object):
    """
    Update status for installed package.
    """
    def __init__(self, name, new_version):
        self.name = name
        self.new_version = new_version

class OSUpdateManager(object):
    """
    Manage OS software updates and patches.
    """
    def __init__ (self):
        self.updater_thread = None

    def get_available_updates(self):
        """
        Returns available OS updates.
        """
        outdated_packages = []
        reboot_required = False
        if yum.updates_available():
            for pkg_info in yum.list_updates():
                outdated_packages.append(PackageStatus(pkg_info[0], pkg_info[1]))
        return outdated_packages

    def check_kernel(self):
        """
        Checks if the currently installed kernel is the one that is running on the OS.
        """
        if version.RELEASE_TYPE == "apc":
            return False
        latest_kernel = str(shell.sudo_execute("rpm -q --last kernel")[0][0]).split("kernel-")[1]
        running_kernel = str(shell.sudo_execute("uname -a", parse=shell.RAW))
        if latest_kernel not in running_kernel:
            return True
        return False

    def apply_updates(self, packages):
        """
        Apply updates for the selected packages. The update process can take some time, therefore
        it runs in a separate thread.

        Use get_update_status() to find details about the running update.

        Input:

        packages = the list of packages to be updated
        """
        if self.updater_thread and self.updater_thread.isAlive():
            raise Error(i18n.custom_gettext("An operating system update is still in progress."))
        else:
            if packages is None:
                packages = []
            self.updater_thread = OSUpdater()
            self.updater_thread.set_packages(packages)
            self.updater_thread.setDaemon(True)
            self.updater_thread.start()

    def get_update_status(self):
        """
        Retrieve the status of the currently running update.
        """
        if self.updater_thread:
            return self.updater_thread.get_update_status()
        else:
            update_status = {
                'in_progress': False,
                'current': '',
                'total': 0,
                'completed': 0,
                'require_reboot': False
            }
            return update_status

class Test(unittest.TestCase):

    def setUp(self):
        self.os_update_manager = OSUpdateManager()

    def test_get_status(self):
        if yum.updates_available():
            self.assertTrue(self.os_update_manager.get_available_updates())

if __name__ == '__main__':
    unittest.main()
