# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Windows client files download
"""

import os
import fnmatch
import stat
import glob
from urllib.request import urlopen
import unittest

from web.bs4 import BeautifulSoup

from core.common import utils
from core.common import configuration
from core.common import i18n
from core.common import log
from core.settings import repositories

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."


class Error(Exception):
    """
    Raised when remote Windows clients repository cannot be accessed.
    """
    pass


class WindowsClientsManager(object):
    """
    Manage *winapps* and *user* client applications repositories.
    """

    WINDOWS_REPO_NAME   = 'winapps'

    def __init__(self, config=configuration.SHARED_CONFIGURATION):
        """
        config -- app configuration data
        """
        self.repos_config_file  = config['repositories']['config_file']
        self.url_open_timeout   = config['webapp'].as_int('url_open_timeout')
        self.local_user_repo    = config['repositories']['local']['user']
        self._read_win_repo_config()

    def _read_win_repo_config(self):
        """
        Read configuration for Windows clients repository
        """
        repo_config = repositories.get_config(self.repos_config_file, self.WINDOWS_REPO_NAME)[self.WINDOWS_REPO_NAME]
        self.is_local_repo  = repo_config['local']
        self.repo_url       = repo_config['url']
        if self.is_local_repo:
            self.repo_url = self.repo_url.replace('file://', '')

    def list_repository(self, extensions_filter=None, exclude_patterns=None):
        """
        Return a listing with available Windows clients files.

        Args:

        extension_filter: allowed extensions
        exclude_patterns: patterns for files to exclude

        This method returns a dictionary:

        windows_clients['is_local_repo'] = True | False
        windows_clients['files']         = [file_info...]

        Each file_info object is a dictionary:

        for local repository:

            {'id':          file basename,
             'size':        file size in human readable format,
             'last_modif':  last modification date,
             'icon':        icon file basename,
             'description': description of the file}

        for remote repository:

            {'id': file basename,
             'url': link to Windows client file on the remote machine }

        this method raises Error when the repository is remote and cannot be accessed.
        """

        # re-read Windows repository configuration since it may have been changed
        self._read_win_repo_config()

        # read windows clients files
        windows_clients = {'is_local_repo': self.is_local_repo, 'files': []}

        if self.is_local_repo:
            # for local repository just list files in the repository path
            for filepath in glob.glob(os.path.join(self.repo_url, '*.*')):
                stats = os.stat(filepath)
                filename = os.path.basename(filepath)
                if not filename.startswith("Softphone_Mac"):
                    file_info = {'id': filename,
                                 'size': utils.format_bytes(stats[stat.ST_SIZE]),
                                 'modif': utils.format_timestamp(stats[stat.ST_MTIME]),
                                 'icon': self._find_icon(self.repo_url, filename),
                                 'description': self._find_description(self.repo_url, filename)}
                    windows_clients['files'].append(file_info)
        else:
            # for remote repository parse directory listing and extract windows clients links
            try:
                page = urlopen(self.repo_url, timeout=self.url_open_timeout)
                if page.code == 200:
                    soup = BeautifulSoup(page.read())
                    for link in soup('a'):
                        file_info = {'id' : link.text, 'url':  self.full_path(link.text)}
                        windows_clients['files'].append(file_info)
                else:
                    raise Error(i18n.custom_gettext('Could not connect to Windows clients repository. (HTTP error: %s)')
                                % page.code)
            except urllib2.URLError as e:
                raise Error(i18n.custom_gettext('Could not connect to Windows clients repository. (I/O error: %s)') % e.reason)

        if extensions_filter:
            filtered_by_extension = [file_info for file_info in windows_clients['files']
                                     if any([file_info['id'].endswith(ext) for ext in extensions_filter])]
            windows_clients['files'] = filtered_by_extension

        if exclude_patterns:
            filtered_by_pattern = []
            for file_info in windows_clients['files']:
                excluded = False
                for pat in exclude_patterns:
                    if fnmatch.fnmatch(file_info['id'], pat):
                        excluded = True
                        break
                if not excluded:
                    filtered_by_pattern.append(file_info)
            windows_clients['files'] = filtered_by_pattern

        return windows_clients

    def list_local_user_repository(self, extensions_filter=None):
        """
        Returns a list of available application files in *local* user repository.

        Each item in list is a file info object with following structure:

            {'id':          file basename,
             'size':        file size in human readable format,
             'last_modif':  last modification date,
             'icon':        icon file basename,
             'description': description of the file}
        """
        file_list = []
        for file_path in glob.glob(os.path.join(self.local_user_repo, '*.*')):
            stats = os.stat(file_path)
            try:
                stats = os.stat(file_path)
            except os.error:
                continue;
            filename = os.path.basename(file_path)
            file_info = {'id': filename,
                         'size': utils.format_bytes(stats[stat.ST_SIZE]),
                         'modif': utils.format_timestamp(stats[stat.ST_MTIME]),
                         'icon': self._find_icon(self.local_user_repo, filename),
                         'description': self._find_description(self.local_user_repo, filename)}
            file_list.append(file_info)

        if extensions_filter:
            file_list = [f for f in file_list
                         if any([f['id'].endswith(ext) for ext in extensions_filter])]

        return file_list

    def full_path(self, id):
        """
        Return full path for specified Windows client.

        id -- Windows client file basename
        """
        return os.path.join(self.repo_url, id)

    def delete(self, id):
        """
        Delete a Windows client file from local repository.

        id -- Windows client file basename
        """
        if self.is_local_repo:
            windows_client_file = self.full_path(id)
            os.remove(windows_client_file)

    def _find_icon(self, repo, windows_client_file):
        """
        Find the icon file associated with specified windows client installation kit.
        Returns the basename of the icon file if a suitable icon was found for this windows
        client package or emty string otherwise.
        """
        windows_client_file         = os.path.basename(windows_client_file)
        windows_client_file_no_ext  = os.path.splitext(windows_client_file)[0]
        icon_files = glob.glob(os.path.join(repo, "*.png"))
        for icon_file in icon_files:
            icon_file           = os.path.basename(icon_file)
            icon_file_no_ext    = os.path.splitext(icon_file)[0]
            if windows_client_file_no_ext == icon_file_no_ext:
                return icon_file
        return ''

    def _find_description(self, repo, windows_client_file):
        """
        Find the description file associated with specified windows client installation kit.
        Returns the description from the file if a suitable one was found for this windows
        client package or emty string otherwise.
        """
        windows_client_file         = os.path.basename(windows_client_file)
        windows_client_file_no_ext  = os.path.splitext(windows_client_file)[0]
        description_files = glob.glob(os.path.join(repo, "*.txt"))
        for description_file in description_files:
            description_file           = os.path.basename(description_file)
            description_file_no_ext    = os.path.splitext(description_file)[0]
            if windows_client_file_no_ext == description_file_no_ext:
                description = {"description": ""}
                utils.read_config(os.path.join(repo, description_file), description)
                return description["description"]
        return ''


class Test(unittest.TestCase):

    def setUp(self):
        self.clients_manager = WindowsClientsManager()

    def test_list_repository(self):
        self.assertTrue(self.clients_manager.list_repository()['files'] is not None)

    def test_list_local_user_repository(self):
        self.assertTrue(self.clients_manager.list_local_user_repository() is not None)


class TestWindowsClientIcon(unittest.TestCase):

    def setUp(self):
        import tempfile
        from core.system import shell

        self.repo                   = tempfile.mkdtemp()
        self.test_file_basename     = 'test'
        self.windows_client_file    = "%s.exe" % self.test_file_basename
        self.icon_file              = "%s.png" % self.test_file_basename

        shell.touch("%s/%s" % (self.repo, self.windows_client_file))
        shell.touch("%s/%s" % (self.repo, self.icon_file))

        self.clients_manager = WindowsClientsManager()

    def tearDown(self):
        from core.system import shell
        shell.rm(self.repo)

    def test_find_icon(self):
        self.assertEqual(self.icon_file,
            self.clients_manager._find_icon(self.repo, self.windows_client_file))

    def test_find_default_icon(self):
        from core.system import shell
        shell.rm(os.path.join(self.repo, self.icon_file))
        self.assertEqual('',
            self.clients_manager._find_icon(self.repo, self.windows_client_file))

if __name__ == '__main__':
    unittest.main()
