/*
 * Copyright 2010 Avaya Inc. All Rights Reserved.
 */

//
// 'globals' namespace is used to share utility functions and
// information about managed services between all scripts
//

var globals = {};

globals.REST_URL = {
    "SID":              "/serveredition/hardware/sid",
    "HOST_ID":          "/serveredition/host_id",
    "LICENCE":          "/serveredition/ipoffice/licence",
    "IPOFFICE_MODE":    "/serveredition/ipoffice/mode",
    "ONEX_CONFIG":      "/serveredition/onex/config",
    "INSTALLER_TYPE":   "/serveredition/installer",
    "HW_PROFILES":      "/serveredition/hardware/profiles",
    "COMPANDING":       "/serveredition/ipoffice/companding",
    "VIRTUALIZATION":   "/serveredition/ipoffice/virtualization"
};

globals.plugins = {};

globals.variables = {};
globals.functions = {};
globals.validators = {};

// tells the resize login function which login window is visible (formFields or changePasswordFields)
globals.variables.loginFormId = "formFields";

// used to show or hide error messages
globals.variables.showErrorMessages = true;

// global ajax timeout values (milliseconds)
globals.variables.GLOBAL_AJAX_TIMEOUT = 60000;
globals.variables.UPDATE_AJAX_TIMEOUT = 3600000;
globals.variables.UPDATES_AVAILABLE_TIMEOUT = 180000;
globals.variables.UPDATES_OS_TIMEOUT = 180000;
globals.variables.SYSLOG_SETTINGS_TIMEOUT = 180000;
globals.variables.STOP_SERVICES_TIMEOUT = 120000;
globals.variables.STOP_ALL_SERVICES_TIMEOUT = 180000;
globals.variables.UPLOAD_AJAX_TIMEOUT = 3600000;
globals.variables.INSTALL_AJAX_TIMEOUT = 300000;
globals.variables.SUBINTERFACE_AJAX_TIMEOUT = 300000;
globals.variables.SAVE_LAN_TIMEOUT = 300000;
globals.variables.SAVE_NETWORK_TIMEOUT = 480000;
globals.variables.GET_UPDATES_TIMEOUT = 360000;
globals.variables.RESTART_SERVICES_TIMEOUT = 180000;
globals.variables.REBOOT_SHUTDOWN_TIMEOUT = 180000;
globals.variables.BACKUP_RESTORE_TIMEOUT = 120000;
globals.variables.CERTIFICATE_DEFAULTS_TIMEOUT = 120000;
globals.variables.CERTIFICATE_DISTRIBUTE_TIMEOUT = 420000;
globals.variables.CREATE_ARCHIVE_TIMEOUT = 600000;
globals.variables.NEW_HARDWARE_TIMEOUT = 420000;

// used to upgrade Web Control after all the other application upgrades are finished
globals.variables.upgradeWebcontrol = false;

// used to upgrade Web Control after all the other OS application upgrades are finished
globals.variables.upgradeWebcontrolOs = false;

// used to start the services which were stopped before the update all process
globals.variables.startStoppedServices = false;

// counts the number of application updates threads in progress
globals.variables.appUpdateThreads = 0;

// used to clear the home tab interval when a different tab is selected 
globals.variables.homeTabInterval = null;

// used to clear the updates tab services interval when a different tab is selected
globals.variables.appsInterval = null;

// used to clear the updates tab system interval when a different tab is selected
globals.variables.osInterval = null;

// used to keep the public ip address
globals.variables.ip = null;

// used to clear the settings tab restore interval when a different tab is selected
globals.variables.restoreInterval = null;

// shows if a tab's html content has been loaded
globals.variables.contentLoaded = true;

// current tab's name
globals.variables.currentTab = "";

// type of the release (appl, isa, abg, apc or abe)
globals.variables.releaseType = "";

// type of the release (Primary, Secondary or Expansion)
globals.variables.serverType = "";

// true to automatically generate and apply a new certificate when necessary
globals.variables.autoCertCreation = false;

// true if the system is virtual, false otherwise
globals.variables.isVirtual = false;

// true if the system is cloud, false otherwise
globals.variables.isCloud = false;

// true if the system is gold, false otherwise
globals.variables.goldEdition = true;

// Web Control's language code (default is en_US)
globals.variables.language = "en_US";

// build uid
globals.variables.buildUID = "";

// Maximum character sequence length allowed for a new password
globals.variables.maxSequenceLength = 4;

// Minimum password length allowed for a new password
globals.variables.minPasswordLength = 8;

// Minimum password length allowed for a new System password
globals.variables.minSystemPasswordLength = 1;

// Maximum password length allowed for a new password
globals.variables.maxPasswordLength = 31;

// Minimum password character code point sets
globals.variables.minCodePointSets = 2;

// Password sequence length
globals.variables.passwordSequenceLength = 3;

// true to show the license expire warning
globals.variables.showLicenseExpireWarning = true;

//true to show the certificate warning
globals.variables.showCertificateModal = true;

// true to periodically check the certificate status
globals.variables.performCertificateCheck = true;

// true if logged in user has security rights
globals.variables.hasSecurityRights = false;

// true if logged in user has administrator rights
globals.variables.hasAdminRights = false;

// true if EASG is installed
globals.variables.hasAsg = "false";
// url to the avaya online knowledge base page
globals.variables.onlineKnowledgebaseUrl = "http://marketingtools.avaya.com/knowledgebase/";

//path to increase partition in progress flag
globals.variables.increasePartitionFlag = "expand_in_progress_flag";

//upgrades TCP port
globals.variables.upgradesTcpPort = 8000;

// Constants for service running state
var ServiceState = {};
ServiceState.STARTING = 0;
ServiceState.RUNNING  = 1;
ServiceState.STOPPING = 2;
ServiceState.STOPPED  = 3;
ServiceState.UNKNOWN  = 4;

// Constants for service version status
var ServiceVersionStatus = {};
ServiceVersionStatus.UP_TO_DATE = 0;
ServiceVersionStatus.OUT_OF_DATE = 1;
ServiceVersionStatus.UNINSTALLED = 2;
ServiceVersionStatus.UNKNOWN = 3;

// Constants for certificate status
var CertificateStatus = {};
CertificateStatus.UNAVAILABLE = 0;
CertificateStatus.OUT_OF_DATE = 1;
CertificateStatus.UP_TO_DATE = 2;

// Constants for code point sets.
var CodePointSets = {};
CodePointSets.LOWERCASE = 0;
CodePointSets.UPPERCASE = 1;
CodePointSets.NUMERIC = 2;
CodePointSets.SPECIAL = 3;

globals.variables.historyChartData = null;
globals.variables.usageHistoryPlot = null;
globals.variables.memoryUsagePlotData = null;
globals.variables.memoryUsagePlot = null;
globals.variables.diskUsagePlotData = null;
globals.variables.diskUsagePlot = null;

globals.variables.chartYFormat = "%.1f%";

globals.variables.NOTIFY_NEW_SESSION = -1;
globals.variables.REDIRECT_REBOOT_SHUTDOWN = 10;
globals.variables.REDIRECT_USER_INACTIVITY = 11;
globals.variables.REDIRECT_USER_LOGOUT = 12;
globals.variables.REDIRECT_SESSION_EXPIRED = 13;
globals.variables.REDIRECT_WEBCONTROL_NOT_RESPONDING = 14;
globals.variables.REDIRECT_WEBCONTROL_RESTART = 15;
globals.variables.REDIRECT_SSL_CERTIFICATE_CHANGED = 16;
globals.variables.REDIRECT_WEBCONTROL_SETTINGS_CHANGED = 17;
globals.variables.REDIRECT_NETWORK_CHANGED = 18;
globals.variables.REDIRECT_UNAUTHENTICATED_REQUEST = 19;
globals.variables.REDIRECT_DATETIME_CHANGED = 20;

// add trim() method to String
if(typeof String.prototype.trim !== 'function') {
    String.prototype.trim = function () {
        return this.replace(/^\s*/, "").replace(/\s*$/, "");
    };
}

//
// User input validation functions
//

globals.validators.isAlphanumeric = function(name) {
    var re = /^[a-zA-Z_0-9]+$/;
    return re.test(name);
};

globals.validators.isAlpha = function(name) {
    var re = /^[a-zA-Z_0-9\_]+$/;
    return re.test(name);
};

globals.validators.isHost = function(name) {
    var re =/^(([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9\-]*[a-zA-Z0-9])\.)*([A-Za-z0-9]|[A-Za-z0-9][A-Za-z0-9\-]*[A-Za-z0-9])$/;
    return re.test(name);
};

globals.validators.isDNS = function(name) {
    return (globals.validators.isHost(name) || globals.validators.isValidIP(name));
};

globals.validators.isDNSArray = function(names) {
    var namesArray = names.split(",");
    for (var i = 0; i < namesArray.length; i++) {
        var dns = namesArray[i].trim();
        if (!globals.validators.isDNS(dns)) {
            return false;
        }
    }
    return true;
};

globals.validators.isNumeric = function(value) {
    var re = /^[0-9]+$/;
    return re.test(value);
};

globals.validators.isFloat = function(value) {
    if (globals.validators.isNumeric(value)) {
        return true;
    }
    if (value.split(".").length != 2) {
        return false;
    } else {
        var integerPart = value.split(".")[0];
        var decimalPart = value.split(".")[1];
        if (globals.validators.isNumeric(integerPart) && globals.validators.isNumeric(decimalPart)) {
            return true;
        }
    }
    return false;
};

globals.validators.isEmail = function(email) {
    var re = /^([A-Za-z0-9_\-\.])+\@([A-Za-z0-9_\-\.])+\.([A-Za-z]{2,3})$/;
    return re.test(email);
};

globals.validators.isMultipleEmail = function(email) {
    var addrs = email.split(",");
    for (var i = 0; i < addrs.length; i++) {
        if (!globals.validators.isEmail(addrs[i].trim())) {
        return false;
        }
    }
    return true;
};

globals.validators.isValidExtension = function(filename,extension) {
    var re = /(?:\.([^.]+))?$/;
    var ext = re.exec(filename)[1];
    if (ext == extension) {
        return true;
    }
    return false;
}

globals.validators.isValidPort = function(port) {
    if (port == "") {
        return false;
    } else if (!globals.validators.isNumeric(port)) {
        return false;
    } else if (!(port > 0 && port < 65536)) {
        return false;
    }
    return true;
};

globals.validators.isValidIP = function(ip) {
    if (ip == null) {
        return false;
    }
    var ipAddress = ip.trim();
    var isValid = false;

    if (ipAddress.indexOf(':') >= 0) {
        // format is IPv6
        isValid = globals.validators.isValidIpv6Address(ipAddress);
    } else {
        // assume IPv4 format
        isValid = globals.validators.isValidIPv4Address(ipAddress);
    }
    return isValid;
};

globals.validators.isValidIPv4Address = function(ipaddr) {
    var re = /^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/;
    if (re.test(ipaddr)) {
        var parts = ipaddr.split(".");
        var firstPart = parseInt(parseFloat(parts[0]));
        if (firstPart == 0        // 0 address
            || firstPart == 127      // loopback
            || firstPart >= 224      // D & E class
            ) {
            return false;
        }
        for (var i = 0; i < parts.length; i++) {
            if (/^0\d/.test(parts[i])
                || parseInt(parseFloat(parts[i])) > 255) {
                return false;
            }
        }
        return true;
    } else {
        return false;
    }
};

globals.validators.isValidIpv6Address = function(ipaddr) {
    var parts = ipaddr.split(":");
    if (parts.length < 2 || parts.length > 8) {
        return false;
    } else {
        var re = /^([0-9a-fA-F]{1,4}|0)/;
        for (var i = 0; i < parts.length; i++) {
            if (parts[i].length >= 1 && !re.test(parts[i])) {
                return false;
            }
        }
        return true;
    }
};

globals.validators.isValidNetMask = function (mask) {
    var re = /^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/;
    if (re.test(mask)) {
        var correct_range = {128:1,192:1,224:1,240:1,248:1,252:1,254:1,255:1,0:1};
        var m = mask.split('.');
        for (var i = 0; i <= 3; i ++) {
            if (!(m[i] in correct_range)) {
                return false
            }
        }
        if ((m[0] == 0)
            || (m[0] != 255 && m[1] != 0)
            || (m[1] != 255 && m[2] != 0)
            || (m[2] != 255 && m[3] != 0)) {
            return false;
        }
    } else {
        return false;
    }
	return true;
};

globals.validators.isValidIpv4Gateway = function(gateway, ipAddr, netmask) {
    if (!globals.validators.isValidIPv4Address(gateway)) {
        return false;
    }
    var gatewayArray = gateway.split(".");
    var ipArray = ipAddr.split(".");
    var netmaskArray = netmask.split(".");
    var gatewayBinary = ( parseInt(gatewayArray[0]) << 24 ) | ( parseInt(gatewayArray[1]) << 16 ) | ( parseInt(gatewayArray[2]) << 8 ) | parseInt(gatewayArray[3]);
    var ipBinary = ( parseInt(ipArray[0]) << 24 ) | ( parseInt(ipArray[1]) << 16 ) | ( parseInt(ipArray[2]) << 8 ) | parseInt(ipArray[3]);
    var netmaskBinary = ( parseInt(netmaskArray[0]) << 24 ) | ( parseInt(netmaskArray[1]) << 16 ) | ( parseInt(netmaskArray[2]) << 8 ) | parseInt(netmaskArray[3]);
    return ( ( ipBinary & netmaskBinary ) == ( gatewayBinary & netmaskBinary ) );
};

globals.validators.isHour = function(number) {
    if (number == "00") {
        return true;
    }
    if (number.length > 2) {
        return false;
    }
    if (!globals.validators.isNumeric(number)) {
        return false;
    } else {
        if (number < 0 || number >= 24) {
            return false;
        }
    }
    return true;
};

globals.validators.isMinute = function(number) {
    if (number == "00") {
        return true;
    }
    if (!globals.validators.isNumeric(number)) {
        return false;
    } else {
        if (number <= 0 || number >= 60) {
            return false;
        }
    }
    return true;
};

globals.validators.isDate = function(sDate, format) {
    var re = null;
    var dArr = null;
    var d = null;
    if (format === undefined) {
        format = "yyyy-mm-dd";
    }
    if (format == "dd-mm-yyyy") {
        re = /^\d{1,2}-\d{1,2}\-\d{4}$/;
        if (re.test(sDate)) {
            dArr = sDate.split("-");
            d = new Date(dArr[2], dArr[1]-1, dArr[0]);
            return d.getMonth() + 1 == dArr[1] && d.getDate() == dArr[0] && d.getFullYear() == dArr[2];
        }
    } else if (format == "dd.mm.yyyy") {
        re = /^\d{1,2}.\d{1,2}\.\d{4}$/;
        if (re.test(sDate)) {
            dArr = sDate.split(".");
            d = new Date(dArr[2], dArr[1]-1, dArr[0]);
            return d.getMonth() + 1 == dArr[1] && d.getDate() == dArr[0] && d.getFullYear() == dArr[2];
        }
    } else if (format == "yyyy.mm.dd") {
        re = /^\d{4}\.\d{1,2}\.\d{1,2}$/;
        if (re.test(sDate)) {
            dArr = sDate.split(".");
            d = new Date(dArr[0], dArr[1]-1, dArr[2]);
            return d.getMonth() + 1 == dArr[1] && d.getDate() == dArr[2] && d.getFullYear() == dArr[0];
        }
    } else if (format == "yyyy-mm-dd") {
        re = /^\d{4}\-\d{1,2}\-\d{1,2}$/;
        if (re.test(sDate)) {
            dArr = sDate.split("-");
            d = new Date(dArr[0], dArr[1]-1, dArr[2]);
            return d.getMonth() + 1 == dArr[1] && d.getDate() == dArr[2] && d.getFullYear() == dArr[0];
        }
    }
    return false;
};

globals.validators.isVlan = function (value) {

    if (globals.validators.isNumeric(value) && value > 0 && value < 4095) {
        return true;
    }
    return false;

};

globals.validators.isASCII = function (value) {
    return /^[\x00-\x7F]*$/.test(value);
};

/**
 * Returns the subnet of an ip/netmask pair
 *
 * @param ip            The IP address to consider
 * @param netmask       The subnet mask to consider
 */
globals.functions.getSubnet = function (ip, netmask) {

    var ipArray = ip.split(".");
    var ipBinary = ( parseInt(ipArray[0]) << 24 ) | ( parseInt(ipArray[1]) << 16 ) |
        ( parseInt(ipArray[2]) << 8 ) | parseInt(ipArray[3]);
    var netmaskArray = netmask.split(".");
    var netmaskBinary = ( parseInt(netmaskArray[0]) << 24 ) | ( parseInt(netmaskArray[1]) << 16 ) |
        ( parseInt(netmaskArray[2]) << 8 ) | parseInt(netmaskArray[3]);
    return (ipBinary & netmaskBinary);

};

//
// UI widgets utility functions
//

/**
 * Loads the content of an html file into the main page.
 *
 * @param tabName the name of the page to load
 */
globals.functions.loadContent = function (tabName) {
    $("#content").load("/" + tabName, function (response, status, xhr) {
        if ((status == "error") || (xhr.status == 0) || (response == "Session expired")) {
            globals.functions.ajaxError(xhr, status, response);
        } else {
            globals.variables.contentLoaded = true;
            globals.variables.currentTab = tabName;
        }
    });
};

/**
 * Adjusts the width of a header to the width of the container
 * Required only in IE
 */
globals.functions.adjustWidth = function () {
    if (navigator.appName.indexOf("Internet Explorer") > -1) {
        $(".fieldset").each( function () {
            var containerWidth = parseInt($("#" + this.id).width());
            $("#" + this.id + " .groupHeader").width(containerWidth);
        });
    }
};

/**
 * Formats pie chart legend in order to also display the MB of memory.
 *
 * @param name  the name of the legend item
 * @param data  the number of KB to transform in MB
 */
globals.functions.formatPieLegend = function (name, data) {
    var valueString = "<label>" + name + ' (' + Math.round((data/1024) * 100) /100
        + $("#js_megaBytes").text() + ')</label>';
    return [valueString.replace(".", $("#js_separator").text()),data];
};

/**
 * Draws a XY chart.
 *
 * @param target            the div where to draw the chart
 * @param gTitle            the title of the chart
 * @param gLabel            the text on the chart's legend
 * @param params            the parameters of the chart. Format: [[[x1, y1], [x2, y2], [x3, y3]]]
 * @param xlabel            the label of the X axis
 * @param ylabel            the label of the Y axis
 * @param computeYLimits    true to compute the limits of the Y axis , false to use default limits
 */
globals.functions.drawChart = function (target, gTitle, gLabel, params, xlabel, ylabel, computeYLimits) {
    if ( xlabel === undefined ) {
        xlabel = "";
    }
    if ( ylabel === undefined ) {
        ylabel = "";
    }
    if (computeYLimits === undefined) {
        computeYLimits = false;
    }
    //round values at 2 decimals
    for (i in params[0]) {
        params[0][i][1] = Math.round(params[0][i][1] * 100) / 100;
    }
    var x = {};
    var y = {};
    var paramElements = params[0];
    x.min = paramElements[0][0];
    x.max = paramElements[paramElements.length-1][0];
    x.ticks = [];
    y.min = paramElements[0][1];
    y.max = paramElements[paramElements.length-1][1];
    for (var i = 0; i < paramElements.length; i++) {
        x.ticks.push(paramElements[i][0]);
        if (paramElements[i][1] < y.min) {
            y.min = paramElements[i][1];
        }
        if (paramElements[i][1] > y.max) {
            y.max = paramElements[i][1];
        }
    }
    if (y.min - 1 > 0) {
        y.min--;
    } else {
        y.min = 0;
    }
    if (y.max + 1 < 100) {
        y.max++;
    } else {
        y.max = 100;
    }
    /*
    var optionsObj = {
    	title: gTitle,
    	legend:{show:(gLabel != null)},
        series:[{label:gLabel}, {pointLabels:{show:true}}],
        seriesColors: ["#71a0c0"],
        seriesDefaults: {
            fillToZero: true,
            showMarker:true,
            pointLabels: { show:true },
            fill: true,
            fillAndStroke: true,
            fillAlpha: 0.3,
            shadow: false
        },
        grid: {
            gridLineColor: '#e2e2e2',
            borderWidth: 0.1,
            borderColor: '#fff',
            background: '#FFFFFF',
            shadow: false
        },
        axes:{
            xaxis:{
                autoscale: true,
                renderer:$.jqplot.DateAxisRenderer,
                labelRenderer: $.jqplot.CanvasAxisLabelRenderer,
                labelOptions: {
                    enableFontSupport: true,
                    fontFamily: 'Verdana',
                    fontSize: '10px'
                },
                ticks: x.ticks,
                tickOptions: {
                    formatString:'%H:%M'
                },
                min: x.min,
                max: x.max
            },
            yaxis:{
                autoscale: true,
                labelRenderer: $.jqplot.CanvasAxisLabelRenderer,
                labelOptions: {
                    enableFontSupport: true,
                    fontFamily: 'Verdana',
                    fontSize: '10px'
                },
                tickOptions: {
                    formatString: globals.variables.chartYFormat
                }
            }
        }

    };
    */
    var optionsObj = {
        title: {
            text: gTitle,
            textAlign: "right"
        },
        /*legend:{show:(gLabel != null)},*/
        series:[{label:gLabel}, {pointLabels:{show:true}}],
        seriesColors: ["#A1D0F0"],
        seriesDefaults: {
            fillToZero: true,
            showMarker:true,
            pointLabels: { show:true },
            fill: true,
            fillAndStroke: true,
            fillAlpha: 0.3,
            shadow: false
        },
        grid: {
            gridLineColor: '#e2e3e3',
            borderWidth: 0.1,
            borderColor: '#fff',
            background: '#FFFFFF',
            shadow: false
        },
        axesDefaults: {
            autoscale: true,
            labelRenderer: $.jqplot.CanvasAxisLabelRenderer,
            labelOptions: {
                enableFontSupport: true,
                fontFamily: 'Arial',
                fontSize: '12px',
                textColor: '#5F798F',
                showLabel: true
            }
        },
        axes:{
            xaxis:{
                renderer:$.jqplot.DateAxisRenderer,
                ticks: x.ticks,
                tickOptions: {
                    formatString:'%H:%M'
                },
                min: x.min,
                max: x.max,
                pad: 0
            },
            yaxis:{
                tickOptions: {
                    formatString: globals.variables.chartYFormat
                }
            }
        }

    };
    if (computeYLimits) {
        optionsObj.axes.yaxis.min = y.min;
        optionsObj.axes.yaxis.max = y.max;
    }
    var graphElem = $("#" + target);
    graphElem.append("<div class=\"graphAxeLabelY\"><span>" + ylabel + " &#8593;</span></div>");
    graphElem.append("<div id=\"" + target + "-chart\" class=\"graphChart\"></div>");
    var graph = $.jqplot(target + "-chart",  params, optionsObj);
    graphElem.append("<div class=\"graphAxeLabelX\"><span>" + xlabel + " &#8594;</span></div>");

    return (graph);
};

/**
 * Draws a pie chart with 2 slices.
 *
 * @param target        the div where to draw the pie chart
 * @param graphTitle    the title of the pie chart
 * @param params        the parameters of the pie chart. Format: [[item1, 20], [item2, 80]]
 */
globals.functions.drawPieChart = function (target, graphTitle, params, label1, label2) {

    if ( label1 === undefined ) {
        label1 = "";
    }
    if ( label2 === undefined ) {
        label2 = "";
    }

    var line1 = [[globals.functions.formatPieLegend(label1, params.used),
        globals.functions.formatPieLegend(label2, params.free)]];
    /*
    var optionsObj = {
        title: graphTitle,
        seriesDefaults: {
            renderer:$.jqplot.PieRenderer,
            rendererOptions:{
                padding: 10
            }
        },
        seriesColors: [ "#6190b0", "#c1e0ff"],
        legend: {
            show: true
        },
        grid: {
            borderWidth: 0.1,
            borderColor: '#fff',
            background: '#FFFFFF',
            shadow: false
        },
        series:[{pointLabels:{show:false}}]

    };
    */
    var optionsObj = {
        title: graphTitle,
        seriesDefaults: {
            renderer:$.jqplot.PieRenderer,
            rendererOptions:{
                showDataLabels: true,
                shadow: false,
                padding: 10
            }
        },
        seriesColors: ["#488FCD", "#A1D0F0"],
        legend: {
            show: true,
            location: 's'
        },
        grid: {
            borderWidth: 0.1,
            borderColor: '#fff',
            background: '#FFFFFF',
            shadow: false
        },
        series:[{pointLabels:{show:false}}]

    };
    return ($.jqplot(target, line1, optionsObj));
};

/**
 * Convert unix timestamp to date string.
 *
 * Date format is: yyyy-mm-dd hh:mm:ss.
 *
 * @param timestamp     Unix timestamp to be converted
 */
globals.functions.timestampToString = function (timestamp, lang) {

    var date = new Date();
    date.setTime(timestamp * 1000);

    if (isNaN(date)) return "";

    var year, month, day, hours, minutes, seconds;

    year = date.getFullYear();
    month = date.getMonth() + 1;
    day = date.getDate();
    hours = date.getHours();
    minutes = date.getMinutes();
    seconds = date.getSeconds();

    // 0 padding
    if (month < 10) {
        month = "0" + month;
    }
    if (day < 10) {
        day = "0" + day;
    }
    if (hours < 10) {
        hours = "0" + hours;
    }
    if (minutes < 10) {
        minutes = "0" + minutes;
    }
    if (seconds < 10) {
        seconds = "0" + seconds;
    }
    if (!lang) {
        lang = "en_US";
    }
    var dateFormat = "";
    switch (lang) {
        case "en_US": //yyyy-mm-dd
            dateFormat = "" + year + "-" + month + "-" + day;
            break;
        case "de_DE": //dd.mm.yyyy
            dateFormat = "" + day + "." + month + "." + year;
            break;
        case "zh_CN": //yyyy.mm.dd
            dateFormat = "" + year + "." + month + "." + day;
            break;
        default: //dd-mm-yyyy
            dateFormat = "" + day + "-" + month + "-" + year;
            break;
    }
    return dateFormat + " " + hours + ":" + minutes + ":" + seconds;
};

/**
 * Transforms a date string into an object with the following properties:
 * date , time , year , month , day , hour , minutes , seconds.
 *
 * @param dateString the date to parse.
 *                   expected format: yyyy-mm-dd hh:mm:ss.
 */
globals.functions.splitDateString = function(dateString) {
    var returnObject = {};
    returnObject.date = dateString.split(" ")[0];
    returnObject.time = dateString.split(" ")[1];
    returnObject.year = returnObject.date.split("-")[0];
    returnObject.month = returnObject.date.split("-")[1];
    returnObject.day = returnObject.date.split("-")[2];
    returnObject.hour = returnObject.time.split(":")[0];
    returnObject.minutes = returnObject.time.split(":")[1];
    returnObject.seconds = returnObject.time.split(":")[2];
    return returnObject;
};

/**
 * Transforms a date string into an unix timestamp.
 *
 * @param dateString the date, under the yyyy-dd-mm hh:mm:ss format.
 */
globals.functions.stringToTimestamp = function (dateString) {
    var timestamp;
    var dateObject = globals.functions.splitDateString(dateString);
    var theDate = new Date(dateObject.year, dateObject.month-1, dateObject.day, dateObject.hour, dateObject.minutes);
    timestamp = theDate.getTime()/1000;
    return timestamp;
};

/**
 * Adjusts the width and the borders of a table and its
 * container div depending on the presence of the scroll.
 *
 * @param initialDivHeight  the initial height of the container div
 *                          (before the table had rows dynamically appended)
 * @param divId   the container div's HTML id
 * @param tableId the table's HTML id
 */
globals.functions.adjustTableBorder = function (initialDivHeight, divId, tableId) {
    var currentDivHeight = $("#" + divId)[0].scrollHeight;
    $("#" + tableId).addClass("fullWidth");
    if (currentDivHeight > initialDivHeight) { // scroll present
        $("#" + tableId).removeClass("grayBorder");
        $("#" + divId).addClass("grayBorder");
    } else {  // scroll absent
        $("#" + divId).removeClass("grayBorder");
        $("#" + tableId).addClass("grayBorder");
    }
};

/**
 * Disables or enables a button and changes its label.
 *
 * @param button    the button element.
 * @param enable    true to enable.
 */
globals.functions.enableButtonElement = function (button, enable, enabledLabel, disabledLabel) {
    if (enable) {
        button.attr("disabled", false);
        button.removeClass("ui-state-disabled");
        if (enabledLabel !== undefined) {
            button[0].innerHTML = enabledLabel;
        }
    } else {
        button.attr("disabled", true);
        button.addClass("ui-state-disabled");
        if (disabledLabel !== undefined) {
            button[0].innerHTML = disabledLabel;
        }
    }
};

/**
 * Disables or enables a button and changes its label, given its id.
 *
 * @param buttonId  the button's id.
 * @param enable    true to enable.
 */
globals.functions.enableButton = function (buttonId, enable, enabledLabel, disabledLabel) {
    var button = $("#" + buttonId);
    globals.functions.enableButtonElement(button, enable, enabledLabel, disabledLabel);
};

/**
 * Shows a div with the no data available message instead
 * of an empty table or shows the table if there is data.
 *
 * @param divId         the id of the div that contains the table
 * @param tableId       the id of the table
 * @param dataLength    the table's data length
 * @param noDataText    string to display if no data is available
 */
globals.functions.showNoDataDiv = function (divId, tableId, dataLength, noDataText) {
    var noDataDivString = "<div id='noDataDiv" + divId + "'>" + noDataText + "</div>";
    if (dataLength < 1) {
        $("#noDataDiv" + divId).remove();
        $("#" + tableId).addClass("hiddenElement");
        $("#" + divId).addClass("blackOnWhite");
        $("#" + divId).append(noDataDivString);

        var parentCssPos = $("#" + divId).css("position");
        if (parentCssPos != "absolute") {
            $("#" + divId).css("position", "relative");
        }
        var h = $("#noDataDiv" + divId).height();
        $("#noDataDiv" + divId).css({
            position: "absolute", width: "100%", left: "0px", top: "50%", margin: "-" + h + "px 0px 0px 0px"
        });
    } else {
        $("#noDataDiv" + divId).remove();
        $("#" + divId).removeClass("blackOnWhite");
        $("#" + tableId).removeClass("hiddenElement");
    }
};

/**
 * Resizes a content div so that it encompasses all of its elements. Used to prevent unwanted scrolls
 * 
 * @param contentDivId      The id of the content div
 * @param bottomElementId   The id of the bottom element of the div
 */
globals.functions.adjustContentHeight = function (contentDivId, bottomElementId) {
    var contentBottom = $("#" + contentDivId).offset().top + $("#" + contentDivId).height();
    var lastElementBottom = $("#" + bottomElementId).offset().top + $("#" + bottomElementId).height();
    if (contentBottom <= lastElementBottom) {
        var contentTop = $("#" + contentDivId).offset().top;
        var newContentHeight = lastElementBottom - contentTop;
        newContentHeight = newContentHeight + 5/100 * newContentHeight;
        $("#" + contentDivId).height(newContentHeight);
    }
};

/**
 * Parses the bytes from a string containing a number and a unit (b, K, M, G)
 * 
 * @param sizeString    Format: "536.8K"
 */
globals.functions.parseBytes = function (sizeString) {

    var bytesNumber = 0;
    var exponent = 0;
    if (sizeString.indexOf("b") > -1) {
        exponent = 0;
        bytesNumber += parseFloat(sizeString.substring(0, sizeString.indexOf("b")));
    } else if (sizeString.indexOf("K") > -1 || sizeString.indexOf("KB") > -1) {
        exponent = 1;
        bytesNumber += parseFloat(sizeString.substring(0, sizeString.indexOf("K")));
    } else if (sizeString.indexOf("M") > -1 || sizeString.indexOf("MB") > -1) {
        exponent = 2;
        bytesNumber += parseFloat(sizeString.substring(0, sizeString.indexOf("M")));
    } else if (sizeString.indexOf("G") > -1 || sizeString.indexOf("GB") > -1) {
        exponent = 3;
        bytesNumber += parseFloat(sizeString.substring(0, sizeString.indexOf("G")));
    }
    bytesNumber = bytesNumber * Math.pow(1024, exponent);
    return bytesNumber;
    
};

/**
 * View if updates are available and show the updates available icon if necessary
 */
globals.functions.showHideUpdatesAvailableIcon = function () {
    $.ajax({
        url: "/api/updates/available",
        cache: false,
        timeout: globals.variables.UPDATES_AVAILABLE_TIMEOUT,
        error: function (xmlHttpReq, textStatus, errorThrown) {
            $("#warningIcon").addClass("hiddenElement");
        },
        success: function (data) {
            if (data) {
                $("#warningIcon").removeClass("hiddenElement");
            } else {
                $("#warningIcon").addClass("hiddenElement");
            }
        }
    });
};

/**
 * Replaces the end line and space characters from a string with the respective html elements
 *
 * @param message   The string to parse
 */
globals.functions.parseMessage = function (message) {

    var parsedString = "";
    parsedString = message.replace(/(\\n)/g , "<br/>").replace(/(\\t)/g, "&nbsp&nbsp&nbsp&nbsp ");
    return parsedString;

};

/**
 * Opens the error message window.
 *
 * @param title     nature of the error
 * @param text      details of the error
 */
globals.functions.openErrorMessageWindow = function (title, text, showCheckbox) {
    $("#errorMessageModalHeader").text(title);
    if (text != "") {
        $("#errorMessageModalText").html(text);
    } else {
        $("#errorMessageModalText").html($("#js_unknownHttpError").text());
    }
    $("#errorMessageModalButtons button").unbind("click");
    if (showCheckbox) {
        $("#errorMessageModalCheckboxDiv").show();
    } else {
        $("#errorMessageModalCheckboxDiv").hide();
    }
    $("#errorMessageModalOk").click(function () {
        if (showCheckbox) {
            globals.variables.showErrorMessages = !$("#errorMessageModalCheckbox")[0].checked;
        }
        $.modal.close();
    });
    $('#errorMessageModal').modal({
        closeHTML: "<a href='' title='" + $("#js_close").text() + "'></a>",
        overlayId: 'modal-overlay',
        containerId: 'errorMessageModal-container'
    });

};

/**
 * Register a global error handler for AJAX calls
 *
 * @param xmlHttpReq    The request object
 * @param textStatus    Request status
 * @param errorThrown   The error thrown
 */
globals.functions.ajaxError = function (xmlHttpReq, textStatus, errorThrown) {

    if (textStatus == "timeout") {
        var timeoutText  = "";
        xmlHttpReq.abort();
        timeoutText = "<br>" + $("#js_httpErrorMessageLabel").text() + ": " + textStatus + "<br>";
        if (globals.variables.showErrorMessages) {
            globals.functions.openErrorMessageWindow($("#js_requestErrorLabel").text(), timeoutText, true);
        }
    } else if ((xmlHttpReq.error !== undefined && xmlHttpReq.error == "timeout")) {
        xmlHttpReq.abort();
        timeoutText = "<br>" + $("#js_httpErrorMessageLabel").text() + ": " + xmlHttpReq.error + "<br>";
        if (globals.variables.showErrorMessages) {
            globals.functions.openErrorMessageWindow($("#js_requestErrorLabel").text(), timeoutText, true);
        }
    } else {
        if (xmlHttpReq.status == 200 // XXX this is ugly
            && errorThrown
            && typeof(errorThrown) == "string"
            && errorThrown.indexOf('Session expired') >= 0) {
                  // session expired, redirect to login page
                 globals.functions.openErrorMessageWindow(
                         $("#js_requestErrorLabel").text(),
                         $("#js_sessionExpiredLabel").text(),
                         false);
            globals.functions.handleRedirection(globals.variables.REDIRECT_SESSION_EXPIRED);
        } else if (xmlHttpReq.status == 401) {
            // unauthenticated, automatically redirect to login page
            globals.functions.handleRedirection(globals.variables.REDIRECT_UNAUTHENTICATED_REQUEST);
        } else if (xmlHttpReq.status == 0) {
            // Webcontrol is down
            globals.dialog.showMessage(globals.dialog.NONE, $("#js_requestErrorLabel").text(),
                                       $("#js_ajaxNoAnswer").text(), "", function() {
                $.modal.close();
                globals.functions.handleRedirection(globals.variables.REDIRECT_WEBCONTROL_NOT_RESPONDING);
            });
        } else if (xmlHttpReq.status != 500 // hide internal server errors
                    && globals.variables.showErrorMessages) {
            var title = $("#js_requestErrorLabel").text();
            var details  = "";
            if (xmlHttpReq.responseText) {
                details += "<br>" + xmlHttpReq.responseText;
            } else if (xmlHttpReq.status == 408) {
                details += "<br>" + $("#js_requestTimeoutErrorLabel").text();
            } else {
                details += "<br>" + $("#js_httpErrorCodeLabel").text() + ": " + xmlHttpReq.status + "<br>";
                if (textStatus) {
                    details += "<br>" + $("#js_httpErrorMessageLabel").text() + ": " + textStatus + "<br>";
                }
                if (errorThrown) {
                    details += "<br>" + $("#js_httpErrorExceptionLabel").text() + ": " + errorThrown;
                }
            }
            if (details) {
                details = globals.functions.parseMessage(details);
                globals.functions.openErrorMessageWindow(title, details, true);
            }
        }
    }

};

/**
 * Vertically aligns a container
 */
globals.functions.verticalAlignContainer = function (containerId) {
    var pageHeight;
    if (navigator.appName.toLowerCase().indexOf("microsoft") != -1) {  // Internet Explorer
        if (document.compatMode == "CSS1Compat") { //standards mode
            pageHeight = document.documentElement.offsetHeight;
        } else {
            pageHeight = document.body.offsetHeight;
        }
    } else {
        pageHeight = window.innerHeight;
    }
    var containerHeight = $("#" + containerId).outerHeight();
    var newContainerTop = (pageHeight - containerHeight)/2;
    $("#" + containerId).css("margin-top", newContainerTop);
};

/**
 * Sets the height of the form container of the login page depending
 * on the components height (which differs from browser to browser)
 */
globals.functions.setHeightLogin = function () {
    var tableTop = $("#" + globals.variables.loginFormId).offset().top;
    var tableHeight = $("#" + globals.variables.loginFormId).outerHeight();
    var tableBottom = tableTop + tableHeight;
    var containerTop = $("#formContainer").offset().top;
    var newContainerHeight = tableBottom - containerTop + 20;
    $("#formContainer").height(newContainerHeight);
};

/**
 * Resizes the login page
 */
globals.functions.resizeLogin = function () {
    //globals.functions.setHeightLogin();
    globals.functions.verticalAlignContainer("loginContainer");
};

/**
 * Resizes the help window's elements
 */
globals.functions.resizeHelpWindow = function () {
    var windowWidth;
    if (navigator.appName.toLowerCase().indexOf("microsoft") != -1) {  // Internet Explorer
        windowWidth = document.body.offsetWidth;
    } else {
        windowWidth = window.innerWidth;
    }
    var newRightDivWidth = windowWidth - $("#helpLeftDiv").width() - 20;
    $("#helpRightDiv").width(newRightDivWidth);
};

/**
 * Redraws the cpu usage history chart
 */
globals.functions.redrawHistoryChart = function () {

    if (globals.variables.historyChartData != null) {
        if (globals.variables.usageHistoryPlot != null) {
            globals.variables.usageHistoryPlot.destroy();
        }
        $("#usageHistoryDiv").empty();
        $("#usageHistoryDiv").removeClass("blackOnWhite");
        var cpuUsageHistoryLegendText = "<label>" + $("#js_cpuUsageHistory").text() + "</label>";
        var notEnoughData = false;
        if (globals.variables.historyChartData.length < 2) {
            notEnoughData = true;
        } else {
            try {
                globals.variables.usageHistoryPlot = globals.functions.drawChart(
                    'usageHistoryDiv',
                    cpuUsageHistoryLegendText,
                    cpuUsageHistoryLegendText ,
                    [globals.variables.historyChartData],
                    $("#js_time").text(),
                    $("#js_usage").text(),
                    true
                );
            } catch (err) {
            }
        }
        if (notEnoughData) {
            $("#usageHistoryDiv").addClass("blackOnWhite");
            $("#usageHistoryDiv").append("<label id='notEnoughDataLabel'>" +
                                             $("#js_noCPUHistoryDataAvailable").text() + "</label>");
        }
    }

};

/**
 * Redraws the memory usage pie chart
 */
globals.functions.redrawMemoryUsagePie = function () {

    if (globals.variables.memoryUsagePlotData != null) {
        if (globals.variables.memoryUsagePlot != null) {
            globals.variables.memoryUsagePlot.destroy();
        }
        $("#memoryUsageDiv").empty();
        var memoryUsagePlotData = [
            globals.functions.formatPieLegend($("#js_used").text(), globals.variables.memoryUsagePlotData.used),
            globals.functions.formatPieLegend($("#js_free").text(), globals.variables.memoryUsagePlotData.free)
        ];
        try {
            globals.variables.memoryUsagePlot.series[0].data = memoryUsagePlotData;
            globals.variables.memoryUsagePlot.replot();
        } catch (e) {
            if ($("#memoryUsageDiv").length > 0) {
                globals.variables.memoryUsagePlot = globals.functions.drawPieChart(
                    "memoryUsageDiv",
                    $("#js_memoryUsage").text(),
                    globals.variables.memoryUsagePlotData,
                    $("#js_used").text(),
                    $("#js_free").text()
                );
            }
        }
    }

};

/**
 * Redraws the disk usage pie chart
 */
globals.functions.redrawDiskUsagePie = function () {

    if (globals.variables.diskUsagePlotData != null) {
        if (globals.variables.diskUsagePlot != null) {
            globals.variables.diskUsagePlot.destroy();
        }
        $("#diskUsageDiv").empty();
        var diskUsagePlotData = [
            globals.functions.formatPieLegend($("#js_used").text(), globals.variables.diskUsagePlotData.used),
            globals.functions.formatPieLegend($("#js_free").text(), globals.variables.diskUsagePlotData.free)
        ];
        try {
            globals.variables.diskUsagePlot.series[0].data = diskUsagePlotData;
            globals.variables.diskUsagePlot.replot();
        } catch (e) {
            if ($("#diskUsageDiv").length > 0) {
                globals.variables.diskUsagePlot = globals.functions.drawPieChart(
                    "diskUsageDiv",
                    $("#js_diskUsage").text(),
                    globals.variables.diskUsagePlotData,
                    $("#js_used").text(),
                    $("#js_free").text()
                );
            }
        }
    }

};

/**
 * Moves a div horizontally to the center of its container
 *
 * @param divId         Id of the div to move
 * @param containerId   Id of the container
 */
globals.functions.alignCenterDiv = function (divId, containerId) {

    var divWidth = $("#" + divId).width();
    var containerWidth = $("#" + containerId).width();
    var marginValue = (containerWidth - divWidth)/2;
    $("#" + divId).css("margin-left", marginValue);

};

/**
 * Checks if a string contains a sequence of characters smaller or equal to the maximum allowed
 *
 * @param string        The string to analise
 * @param maxAllowed    The maximum allowed length of the sequence
 */
globals.functions.checkIfAllowedSequence = function (string, maxAllowed) {

    for (var i = 0; i < string.length; i++) {
        var sequenceEnd = maxAllowed + i;
        if ((i + maxAllowed) > string.length) {
            sequenceEnd = string.length;
        }
        if (globals.functions.checkNormalSequence(string, i, sequenceEnd, maxAllowed) ||
                globals.functions.checkReversedSequence(string, i, sequenceEnd, maxAllowed) ||
                globals.functions.checkRepeatedSequence(string, i, sequenceEnd, maxAllowed)) {
            return false;
        }
    }
    return true;

};

/**
 * Checks if a substring of a string is a sequence of characters (example: "abcdef")
 *
 * @param string            The string to analyse
 * @param startIndex        Substring start position
 * @param stopIndex         Substring end position
 * @param maxAllowed        Maximum number of allowed sequential characters
 */
globals.functions.checkNormalSequence = function (string, startIndex, stopIndex, maxAllowed) {

    var sequenceCounter = 1;
    for (var j = startIndex; j < stopIndex; j++) {
        if ((string.charCodeAt(j) + 1) == string.charCodeAt(j+1)) {
            sequenceCounter++;
        }
    }
    return (sequenceCounter > maxAllowed);

};

/**
 * Checks if a substring of a string is a sequence of characters (example: "fedcba")
 *
 * @param string            The string to analyse
 * @param startIndex        Substring start position
 * @param stopIndex         Substring end position
 * @param maxAllowed        Maximum number of allowed sequential characters
 */
globals.functions.checkReversedSequence = function (string, startIndex, stopIndex, maxAllowed) {

    var sequenceCounter = 1;
    for (var j = startIndex; j < stopIndex; j++) {
        if ((string.charCodeAt(j) - 1) == string.charCodeAt(j+1)) {
            sequenceCounter++;
        }
    }
    return (sequenceCounter > maxAllowed);

};

/**
 * Checks if a substring of a string is a sequence of repeated characters (example: "aaaaa")
 *
 * @param string            The string to analyse
 * @param startIndex        Substring start position
 * @param stopIndex         Substring end position
 * @param maxAllowed        Maximum number of allowed sequential characters
 */
globals.functions.checkRepeatedSequence = function (string, startIndex, stopIndex, maxAllowed) {

    var sequenceCounter = 1;
    for (var j = startIndex; j < stopIndex; j++) {
        if ((string.charCodeAt(j)) == string.charCodeAt(j+1)) {
            sequenceCounter++;
        }
    }
    return (sequenceCounter > maxAllowed);

};

/**
 * Format a size in bytes to a more convenient format (MB, GB, etc.)
 *
 * @param bytes            The size in bytes
 * @param decimals         Precision of the parsed value
 */
globals.functions.bytesToSize = function bytesToSize(bytes, decimals) {
    var precision = (decimals != undefined) ? decimals : 1;
    var sizes = [$("#js_bytes").text(), $("#js_kb").text(), $("#js_mb").text(), $("#js_gb").text(), $("#js_tb").text()];
    if (bytes == 0) return '0';
    var i = parseInt(Math.floor(Math.log(bytes) / Math.log(1024)));
    return (bytes / Math.pow(1024, i)).toFixed(precision) + sizes[i];
};

/**
 * Transforms a netmask string into a network prefix
 *
 * @param netmask   Format: 255.255.255.0
 */
globals.functions.netmaskToPrefix = function (netmask) {

    var prefix = 0;
    var parts = netmask.split(".");
    for (var i = 0; i < parts.length; i++) {
        var binaryPart = parseInt(parts[i]).toString(2);
        for (var j = 0; j < binaryPart.length; j++) {
            prefix += parseInt(binaryPart.charAt(j));
        }
    }
    return prefix;

};

/**
 * Transforms a network prefix number into a subnet mask string
 *
 * @param prefix    Format: 24
 */
globals.functions.prefixToNetmask = function (prefix) {

    var netmask = "";
    var interestingOctet = "";
    var fullOctetNumber = parseInt(prefix / 8);
    var netmaskComposed = false;
    for (var i = 0; i < fullOctetNumber; i++) {
        if (i == 3) {
            netmask += "255";
            netmaskComposed = true;
        } else {
            netmask += "255.";
        }
    }
    if (!netmaskComposed) {
        for (var j = 0; j < 8; j++) {
            if (j < parseInt(prefix) % 8) {
                interestingOctet += "1";
            } else {
                interestingOctet += "0";
            }
        }
        netmask += parseInt(interestingOctet, 2).toString();
    }
    for (var z = netmask.split(".").length; z < 4; z++) {
        netmask += ".0";
    }
    return netmask;

};

if (typeof jQuery.fn.live == 'undefined' || !(jQuery.isFunction(jQuery.fn.live))) {
  jQuery.fn.extend({
      live: function (event, callback) {
         if (this.selector) {
              jQuery(document).on(event, this.selector, callback);
          }
      }
  });
}


globals.dialog = {};
globals.dialog.NONE = 0;
globals.dialog.INFO = 1;
globals.dialog.WARNING = 2;
globals.dialog.ERROR = 3;
globals.dialog.QUESTION = 4;
globals.dialog.CSS = ["", "infoDialog", "warningDialog", "errorDialog", "questionDialog"];

globals.dialog.show = function(type, title, content, actions, cssClass, canClose) {
    var dlgClass = globals.dialog.CSS[type];
    if (cssClass) {
        dlgClass += " " + cssClass;
    }
    $("#commonDialog").addClass(dlgClass);
    $("#commonDialog .modalHeader").text(title);
    $("#commonDialogContent").html(content);
    var actionsHTML = "";
    var i = 0;
    for (action in actions) {
        actionsHTML += "<button class=\"avayaWebcontrolButton\" id=\"commonDialogActionButton" + i + "\">" + action + "</button>";
       
        i++;
    }
    $("#commonDialogActions").html(actionsHTML);
    i = 0 ;
    for (action in actions) {
        $("#commonDialogActionButton" + i).off("click");
        $("#commonDialogActionButton" + i).click(actions[action]);
        i++;
    }
    

    if (canClose === undefined) {
        canClose = true;
    }
    $("#commonDialog").modal({
        close: canClose,
        closeHTML: (canClose) ? "<a href=''></a>" : "",
        escClose: canClose,
        overlayClose: canClose,
        overlayId: "modal-overlay",
        containerId: "commonDialogContainer",
        minWidth: 300,
        onClose: function() {
            $.modal.close();
            $("#commonDialog").removeClass(dlgClass);
            $("#commonDialogActions button").remove();
        }
    });

};

globals.dialog.showMessage = function(type, title, message, cssClass, okHandler, canClose) {
    var handler = null;
    if (okHandler) {
        handler = okHandler;
    } else {
        handler = function() {
            $.modal.close();
        };
    }
    if (canClose === undefined) {
        canClose = true;
    }
    var handlersObj = {};
    handlersObj[$("#js_Ok").text()] = handler;
    globals.dialog.show(type, title, message, handlersObj, (cssClass) ? cssClass : null, canClose);
};

globals.dialog.showConfirm = function(type, title, message, handler, cssClass, canClose) {
    var actionHandlers = {};
    actionHandlers[$("#js_Yes").text()] = function() {
        $.modal.close();
        handler(true);
    };
    actionHandlers[$("#js_No").text()] = function() {
        $.modal.close();
        handler(false);
    };
    globals.dialog.show(type, title, message, actionHandlers, (cssClass) ? cssClass : null, canClose);
}

/**
 * Sets the locale options for the datepicker.
 *
 * @param lang  the language code (ex: "en_US" , "fr_FR")
 */
globals.getLocaleOptions = function (lang) {
    var language = "";
    var format = "yy-mm-dd";
    var validationFormat = "yyyy-mm-dd";
    var errorText = $("#js_invalidDateAndFormat_en_US").text();
    switch (lang) {
        // TODO replace this switch with a dictionary
        case "en_US":
            language = "";
            format = "yy-mm-dd";
            validationFormat = "yyyy-mm-dd";
            errorText = $("#js_invalidDateAndFormat_en_US").text();
            break;
        case "fr_FR":
            language = "fr";
            format = "dd-mm-yy";
            validationFormat = "dd-mm-yyyy";
            errorText = $("#js_invalidDateAndFormat_Rest").text();
            break;
        case "de_DE":
            language = "de";
            format = "dd.mm.yy";
            validationFormat = "dd.mm.yyyy";
            errorText = $("#js_invalidDateAndFormat_de_DE").text();
            break;
        case "it_IT":
            language = "it";
            format = "dd-mm-yy";
            validationFormat = "dd-mm-yyyy";
            errorText = $("#js_invalidDateAndFormat_Rest").text();
            break;
        case "es_ES":
            language = "es";
            format = "dd-mm-yy";
            validationFormat = "dd-mm-yyyy";
            errorText = $("#js_invalidDateAndFormat_Rest").text();
            break;
        case "pt_BR":
            language = "pt-BR";
            format = "dd-mm-yy";
            validationFormat = "dd-mm-yyyy";
            errorText = $("#js_invalidDateAndFormat_Rest").text();
            break;
        case "nl_NL":
            language = "nl";
            format = "dd-mm-yy";
            validationFormat = "dd-mm-yyyy";
            errorText = $("#js_invalidDateAndFormat_Rest").text();
            break;
        case "ru_RU":
            language = "ru";
            format = "dd.mm.yy";
            validationFormat = "dd.mm.yyyy";
            errorText = $("#js_invalidDateAndFormat_Rest").text();
            break;
        case "zh_CN":
            language = "zh-CN";
            format = "yy.mm.dd";
            validationFormat = "yyyy.mm.dd";
            errorText = $("#js_invalidDateAndFormat_zh_CN").text();
            break;
    }
    return ({language: language, format: format, validationFormat: validationFormat, errorText: errorText});
};

/**
 * Transforms an integer format version to a string format version if possible
 *
 * @param integerForm   The version in integer format
 * @return {Object}     Object containing the parameters of the string format.
 *                      Ex: {versionNumber: 9.0.1.0, buildNumber: 567}
 */
globals.functions.parseVersion = function (integerForm) {

    if (integerForm.indexOf("el6") > -1) {
        integerForm = integerForm.substring(0, integerForm.indexOf("el6") - 1);
    }
    if (integerForm.indexOf("9.") == 0) {
        return {versionNumber: globals.functions.parseVersion9(integerForm).versionNumber,
                buildNumber: globals.functions.parseVersion9(integerForm).buildNumber};
    } else {
        return {versionNumber: globals.functions.parseVersionDefault(integerForm).versionNumber,
                buildNumber: globals.functions.parseVersionDefault(integerForm).buildNumber};
    }
};

/**
 * Default mode of version parsing.
 *
 * @param integerForm                               The version in integer format.
 * @returns {{versionNumber: *, buildNumber: *}}    Object containing the parameters of the string format.
 */
globals.functions.parseVersionDefault = function (integerForm) {
    var re = /^\d{1,}\.\d{1,}\.\d{1,5}\.\d{1,}$/;
    var buildNumber = null;
    var versionNumber = null;
    if (re.test(integerForm)) {
        var integerFormArray = integerForm.split(".");
        var majorVersionNumber = integerFormArray[0];
        var minorVersionNumber = integerFormArray[1];
        var featurePackVersion = Math.floor(parseInt(integerFormArray[2])/1000);
        var maintenaceVersionNumber = Math.floor((parseInt(integerFormArray[2])%1000)/100);
        var specialVersionNumber = Math.floor(parseInt(integerFormArray[2])%100);
        buildNumber = integerFormArray[3];
        var extraVersions = "";
        if (/*specialVersionNumber != 0*/true) {
            extraVersions = "." + featurePackVersion + "." + maintenaceVersionNumber + "." + specialVersionNumber;
        } else if (maintenaceVersionNumber != 0) {
            extraVersions = "." + featurePackVersion + "." + maintenaceVersionNumber;
        } else if (featurePackVersion) {
            extraVersions = "." + featurePackVersion;
        }
        versionNumber = majorVersionNumber + "." + minorVersionNumber + extraVersions;
    }
    return {versionNumber: versionNumber, buildNumber: buildNumber};
};

/**
 * Parse version mode for 9.0 and 9.1 versions.
 *
 * @param integerForm                               The version in integer format.
 * @returns {{versionNumber: *, buildNumber: *}}    Object containing the parameters of the string format.
 */
globals.functions.parseVersion9 = function (integerForm) {
    var re = /^\d{1,}\.\d{1,}\.\d{1,4}\.\d{1,}$/;
    var buildNumber = null;
    var versionNumber = null;
    if (re.test(integerForm)) {
        var integerFormArray = integerForm.split(".");
        var majorVersionNumber = integerFormArray[0];
        var minorVersionNumber = integerFormArray[1];
        var maintenaceVersionNumber = 0;
        var specialVersionNumber = 0;
        switch (integerFormArray[2].length) {
            case 1:
            case 2:
            default:
                maintenaceVersionNumber = 0;
                specialVersionNumber = parseInt(integerFormArray[2]);
                break;
            case 3:
                maintenaceVersionNumber = parseInt(integerFormArray[2].substring(0,1));
                specialVersionNumber = parseInt(integerFormArray[2].substring(1));
                break;
            case 4:
                maintenaceVersionNumber = parseInt(integerFormArray[2].substring(0,2));
                specialVersionNumber = parseInt(integerFormArray[2].substring(2));
        }
        buildNumber = integerFormArray[3];
        versionNumber = majorVersionNumber + "." + minorVersionNumber + "."
            + maintenaceVersionNumber + "." + specialVersionNumber;
    }
    return {versionNumber: versionNumber, buildNumber: buildNumber};
};

/**
 * Replaces placeholders with values when composing a translated string.
 *
 * @param elementId         The id of the html element to consider.
 * @param messageLabelId    The id of the label containing the original form of the message.
 * @param valuesArray       The values used to replace the placeholders.
 */
globals.functions.composeText = function (elementId, messageLabelId, valuesArray) {

    var currentString = $("#" + messageLabelId).text();
    for (var i = 0; i < valuesArray.length; i++) {
        currentString = currentString.replace("%s" + i, valuesArray[i]);
    }
    $("#" + elementId).html(currentString);

};

/**
 * Handle redirection (to login, etc.) and handle notifications to parent application, if any.
 *
 * @param event     The event that triggered the redirection
 *
 * There can be optional parameters passed, depending on the type of event.
 * These parameters will be accessible through the built-in 'arguments' parameter.
 */
globals.functions.handleRedirection = function(event) {
    if (typeof globals.variables.REFERRER === 'undefined') {
        switch (event) {
            case globals.variables.REDIRECT_WEBCONTROL_SETTINGS_CHANGED:
            case globals.variables.REDIRECT_NETWORK_CHANGED:
                var location = arguments[1];
                window.location = location;
                break;
            default:
                window.location.replace("/login");
                break;
        }
    } else {
        // WCP was launched in IFRAME, by a parent application (Web Manager) - have to send notification
        var strNotif = "" + event;
        if (event == globals.variables.REDIRECT_NETWORK_CHANGED) {
            strNotif += "|" + arguments[2];
        }
        window.parent.postMessage(strNotif, globals.variables.REFERRER);
    }
};

globals.functions.makeFixedTableHeader = function(tableId) {
    var tableIdSel = "#" + tableId;
    var parentElem = $(tableIdSel).parent();
    var headerHeight = parseInt($(tableIdSel + " thead").css("height"));
    if (parentElem.css("position") === "absolute") {
        parentElem.css("top", headerHeight);
    } else {
        parentElem.height(parseInt(parentElem.css("height")) - headerHeight);
    }
    var headerTableId = $(tableIdSel).attr("id") + "-header";
    if ($("#" + headerTableId).length > 0) {
        $("#" + headerTableId).remove();
    }
    var headerTableSkeletonHTML = "<table id='" + headerTableId + "'><thead><tr></tr></thead></table>";
    parentElem.before(headerTableSkeletonHTML);
    var destTR = $("#" + headerTableId + " thead tr");
    var headerCellSel = tableIdSel + " th";
    $(tableIdSel + " th").each(function(idx) {
        var cloned = $(this).clone(true);
        cloned.width($(this).width());
        cloned.appendTo(destTR);
        cloned.click(function(e) {
            $(headerCellSel)[idx].click(e);
        });
    });
    $("#" + headerTableId).width($(tableIdSel).width());
    $("#" + headerTableId).addClass($(tableIdSel).attr("class"));
    $(tableIdSel + " thead").css({visibility: 'hidden'});
    $(tableIdSel).css('margin-top', '-' + headerHeight + 'px');
    $(tableIdSel).bind("sortEnd",function() {
        $("#" + headerTableId).width($(tableIdSel).width());
        $("#" + headerTableId + " th").each(function(i) {
            $(this).attr("class", $(headerCellSel + ":nth-child(" + (i + 1) + ")").attr("class"));
        });
    });
};

globals.functions.resizeFixedTableHeader = function(tableId) {
    $("#" + tableId + "-header").width($("#" + tableId).width());
    $("#" + tableId + "-header th").each(function(i) {
        $(this).width($("#" + tableId + " th:nth-child(" + (i + 1) + ")").width());
    });
};

/**
 * Updates the password rules list.
 *
 * @param data  Object containing the password rules.
 *
 * Format:
 * {
 *      min_length: 8,
 *      min_uppercase: 0,
 *      min_lowercase: 0,
 *      min_numeric: 0,
 *      min_special: 0,
 *      allow_sequence: false,
 *      max_allowed_sequence: 4
 * }
 */
globals.functions.showHidePasswordRules = function (data) {

    $(".minLengthNumber").text(data.min_length);
    $(".minUppercaseNumber").text(data.min_uppercase);
    $(".minLowercaseNumber").text(data.min_lowercase);
    $(".minNumericNumber").text(data.min_numeric);
    $(".minSpecialNumber").text(data.min_special);
    $(".maxSequenceNumber").text(data.max_allowed_sequence);

    if (parseInt(data.min_uppercase) > 0) {
        $(".minUppercaseLi").show();
    } else {
        $(".minUppercaseLi").hide();
    }

    if (parseInt(data.min_lowercase) > 0) {
        $(".minLowercaseLi").show();
    } else {
        $(".minLowercaseLi").hide();
    }

    if (parseInt(data.min_numeric) > 0) {
        $(".minNumericLi").show();
    } else {
        $(".minNumericLi").hide();
    }

    if (parseInt(data.min_special) > 0) {
        $(".minSpecialLi").show();
    } else {
        $(".minSpecialLi").hide();
    }

    if (parseInt(data.max_allowed_sequence) > 0 && !data.allow_sequence) {
        $(".maxSequenceLi").show();
    } else {
        $(".maxSequenceLi").hide();
    }

};

/**
 * Returns the code point set of a given character (lowercase, uppercase, numeric, special).
 *
 * @param chr   The character to consider.
 */
globals.functions.codePointSet = function (chr) {
    if (/[A-Z]|[\u0080-\u024F]|[\u0400-\u045F]/.test(chr) && chr === chr.toUpperCase()) {
        return CodePointSets.UPPERCASE;
    } else if (/[a-z]|[\u0080-\u024F]|[\u0400-\u045F]/.test(chr) && chr === chr.toLowerCase()) {
        return CodePointSets.LOWERCASE;
    } else if (/[0-9]/.test(chr)) {
        return CodePointSets.NUMERIC;
    }
    return CodePointSets.SPECIAL;
};

/**
 * Count the characters of a given type from a string, and see if there are enough of them.
 *
 * @param type          The character type, as defined by the CodePointSets constant.
 * @param string        The string to analyze.
 * @param minNumber     The minimum number of characters of the given type that need to be present.
 */
globals.functions.validateCharacterType = function (type, string, minNumber) {
    var count = 0;
    for (var i = 0; i < string.length; i++) {
        if (globals.functions.codePointSet(string.charAt(i)) == type) {
            count++;
        }
        if (count >= minNumber) {
            return true;
        }
    }
    return false;
};

/**
 * Count the code point sets a password contains.
 *
 * @param password  The string to consider.
 */
globals.functions.countPointCodeSets = function (password) {
    var counter = 0;
    for (var i in CodePointSets) {
        for (var j = 0; j < password.length; j++) {
            var chr = password.charAt(j);
            if (globals.functions.codePointSet(chr) == CodePointSets[i]) {
                counter++;
                break;
            }
        }
    }
    return counter;
};

/**
 * Checks if a password contains an unpermitted character sequence
 *
 * @param password          The password to analise.
 * @param sequenceLength    The sequence length to consider.
 */
globals.functions.validSequences = function (password, sequenceLength) {
    for (var i = 0; i < password.length; i++) {
        var sequence = "";
        if (i + sequenceLength <= password.length) {
            sequence = password.substring(i, i + sequenceLength);
            if (sequence.length > 0) {
                var invalidSequence = true;
                for (var j = 0; j < sequence.length; j++) {
                    if (j + 1 < sequence.length) {
                        if (sequence.charAt(j) != sequence.charAt(j+1)) {
                            invalidSequence = false;
                            break;
                        }
                    }
                }
                if (invalidSequence) {
                    return false;
                }
            }
        } else {
            break;
        }
    }
    return true;
};
