#!/usr/bin/env python3

# Copyright 2010 Avaya Inc. All Rights Reserved.

"""
Main entry point for the Web Control application.
"""

import web
import time
import os
import glob

from core.common import configuration
from core.common import log

from web.wsgiserver import CherryPyWSGIServer
from web.session import Store
from web.cheroot.ssl.builtin import BuiltinSSLAdapter
from web.cheroot.server import HTTPServer


# Logging tag
TAG = 'webcontrol'

__author__      = "Avaya Inc."
__copyright__   = "Copyright 2010, Avaya Inc."

class MemoryStore(Store):
    """
    Store for saving sessions in memory.
    This store is not suitable when web.py is deployed in CGI mode.
    """

    def __init__(self):
        self.mem_store = {}
        self.del_store = []

    def __contains__(self, key):
        return key in self.mem_store and key not in self.del_store

    def __getitem__(self, key):
        if key in self.mem_store and key not in self.del_store:
            t, value = self.mem_store[key]
            self.mem_store[key] = (time.time(), value)
            return value
        else:
            return None

    def __setitem__(self, key, value):
        if key not in self.del_store:
            self.mem_store[key] = (time.time(), value)
            

    def __delitem__(self, key):
        if key not in self.del_store:
            self.del_store.append(key)
        if key in self.mem_store:
            del self.mem_store[key]

    def cleanup(self, timeout):
        now = time.time()
        to_del = []
        for k, (atime, value) in self.mem_store.items():
            if now - atime > timeout :
                to_del.append(k)
        for k in to_del:
            del self.mem_store[k]
            if k in self.del_store:
                self.del_store.remove(k)

log.info(TAG, "webcontrol is starting...")

CherryPyWSGIServer.version = 'CherryPy' #see IPOFFICE-18007, IPOFFICE-17968

# web.py configuration
web.config.debug = False
web.config.session_parameters['cookie_name'] = 'webpy_session_id'
web.config.session_parameters['cookie_domain'] = None
web.config.session_parameters['httponly_cookie'] = True
web.config.session_parameters['ignore_expiry'] = False
web.config.session_parameters['ignore_change_ip'] = True
web.config.session_parameters['secret_key'] = 'fLjUfxqXtfNoIldA0A0J'
web.config.session_parameters['expired_message'] = 'Session expired'

timeout = os.getenv('WEBCONTROL_SESSION_TIMEOUT')
if timeout is None:
    log.warn(TAG, 'WEBCONTROL_SESSION_TIMEOUT not set, using default value: %d s'
                           % configuration.DEFAULT_SESSION_TIMEOUT)
    timeout = configuration.DEFAULT_SESSION_TIMEOUT
else:
    timeout = int(timeout)
web.config.session_parameters['timeout'] = timeout

if os.getenv('WEBCONTROL_PROTOCOL', 'https').lower() == 'https':
    certificate = configuration.SHARED_CONFIGURATION['webapp']['certificate_file']
    privacy_key = configuration.SHARED_CONFIGURATION['webapp']['privacy_key_file']
    certificate_chain = configuration.SHARED_CONFIGURATION['webapp']['certificate_chain_file']
    CherryPyWSGIServer.ssl_certificate = certificate
    CherryPyWSGIServer.ssl_private_key = privacy_key
    if os.path.isfile(certificate_chain):
        CherryPyWSGIServer.ssl_certificate_chain = certificate_chain

    CherryPyWSGIServer.ssl_adapter = BuiltinSSLAdapter(certificate, privacy_key, None,ciphers="ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384")
    HTTPServer.ssl_adapter = BuiltinSSLAdapter(certificate=certificate, private_key=privacy_key,ciphers="ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384")

    web.config.session_parameters['secure_cookie'] = True

sessions = []
apps = []

log.info(TAG, "loading applications...")

for filename in glob.glob('*_webapp.py'):
    try:
        mod_name = os.path.basename(filename).replace('.py', '')
        log.info(TAG, "loading application from <%s>" % mod_name)
        mod =  __import__(mod_name, fromlist=[''])
        if not hasattr(mod, 'APP') or not hasattr(mod, 'URL'):
            log.error(TAG, 'missing either APP or URL attribute for <%s>, this application is NOT loaded'
                                    % mod_name)
        else:
            apps.append(mod)
    except:
        log.exception(TAG, 'unable to load application from <%s>' % filename)

# sort applications according to their LOAD_ORDER
# and prepare ('url', webapp...) tuple expected by web.py

apps.sort(key=lambda m: m.LOAD_ORDER if hasattr(m, "LOAD_ORDER") else 0)

log.info(TAG, 'loaded applications:\n%s' %
                       '\n'.join(['%-10s -> %s' % (m.URL or "/", m.__name__) for m in apps]))

log.info(TAG, 'webcontrol startup complete')

urls = [[m.URL, m.APP] for m in apps]
urls = tuple([item for l in urls for item in l])
root_app = web.application(urls, globals())

class SessionExpiredHandler(web.HTTPError):
    def __init__(self, message):
        if web.ctx.env.get('HTTP_X_REQUESTED_WITH') == 'XMLHttpRequest':
            web.HTTPError.__init__(self, '200 OK', {}, data=message)
        else:
            if ('/autologin' in web.ctx.fullpath) and (web.ctx.method.upper() == 'POST'):
                referrer = web.input(referrer=None).referrer
                auth = web.input(auth=None).auth
                lang = web.input(lang='en_US').lang
                csrf_token = web.input(csrf_token=None).csrf_token

                message = """ <html><body onload='document.autologin.submit()'>
                    <form action='' method='POST' name='autologin'>
                    <input type='hidden' name='auth' id='auth' value='%s'>
                    <input type='hidden' name='lang' id='lang' value='%s'>
                    <input type='hidden' name='referrer' id='referrer' value='%s'>
                    <input type='hidden' name='csrf_token' id='csrf_token' value='%s'>
                    </form><body></html> """ % (auth, lang, referrer,csrf_token)
                log.info(TAG, 'webcontrol AUTOLOGIN %s' % message)
                web.HTTPError.__init__(self, '200 OK', {}, data=message)
            else:
                raise web.seeother(web.ctx.fullpath)

web.session.SessionExpired = SessionExpiredHandler

session = web.session.Session(root_app, MemoryStore(),
                              initializer={'lang': configuration.DEFAULT_SESSION_LANGUAGE,
                                           'admin_rights': False,
                                           'security_rights': False})


def session_hook():
    """
    Session hook
    
    Used to share session information within sub-applications
    (see http://webpy.org/cookbook/sessions_with_subapp)
    """
    web.ctx.session = session

root_app.add_processor(web.loadhook(session_hook))


def custom_headers(handler):
    """
    Hook used to add custom headers.
    Works by inserting the usual no-cache headers, only if they were not previously set by the handler.
    """
    result = handler()
    web.header('Cache-Control', 'no-cache, no-store, must-revalidate', True)
    web.header('Pragma', 'no-cache', True)
    web.header('Expires', '0', True)
    frame_setting = 'https://*:'+str(int(os.getenv( 'WEBCONTROL_PORT',7071))) +' https://*:' + str(int(os.getenv( 'WEBCONTROL_PORT',7071))-1)
    web.header('Content-Security-Policy', 'frame-ancestors '+ frame_setting +' ;', True)
    web.header('X-Frame-Options', 'ALLOW_FROM ' + frame_setting, True)
    if os.getenv('WEBCONTROL_PROTOCOL', 'https').lower() == 'https':
        web.header('Strict-Transport-Security', 'max-age=31536000', True)
    web.header('X-XSS-Protection', '1; mode=block',True)
    web.header('X-Content-Type-Options',  'nosniff', True)

    return result

root_app.add_processor(custom_headers)

# here be dragons
if __name__ == "__main__":
    root_app.run()
